package com.bxm.localnews.payment.service.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Set;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.util.NidGeneratorUtil;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserAuthIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.param.AccountCashParam;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.config.WithdrawConfig;
import com.bxm.localnews.payment.config.WithdrawProperties;
import com.bxm.localnews.payment.constant.WithdrawEnum;
import com.bxm.localnews.payment.constant.WithdrawTypeEnum;
import com.bxm.localnews.payment.domain.WithdrawMapper;
import com.bxm.localnews.payment.proxy.WithdrawProxySerivce;
import com.bxm.localnews.payment.service.UserWithdrawService;
import com.bxm.localnews.payment.vo.WithdrawFlow;
import com.bxm.localnews.vo.UserAuth;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2019/3/11 14:39
 * @desc
 */
@Service
public class WithdrawServiceImpl extends BaseService implements UserWithdrawService {

    @Autowired
    private WithdrawConfig withdrawConfig;

    @Autowired
    private WithdrawProperties withdrawProperties;

    @Autowired
    private WithdrawMapper withdrawMapper;

    @Autowired
    private UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    private UserAuthIntegrationService userAuthIntegrationService;

    @Autowired
    private WithdrawProxySerivce withdrawProxySerivce;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private BizLogService bizLogService;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private DistributedLock distributedLock;

    @Autowired
    private PayProperties payProperties;

    @Override
    public Json<BigDecimal> userWithdraw(Long userId, BigDecimal amount, String equipment, String ip, Integer platform) {
        if (!withdrawProperties.getEnableSwitch()) {
            logger.warn("提现开关未开启---不允许提现");
            return ResultUtil.genFailedResult("请稍后再试");
        }
        logger.info("用户[{}]发起提现，金额为[{}]", userId, amount);
        Set<BigDecimal> withdrawSet = withdrawConfig.getWithdrawSet();
        if (!withdrawSet.contains(amount)) {
            logger.error("提现金额列表[{}]不包含金额为[{}]", JSON.toJSONString(withdrawSet), amount);
            return ResultUtil.genFailedResult("请稍后再试");
        }
        BigDecimal drawableCash = userAccountIntegrationService.getUserDrawableCash(userId);
        if (drawableCash.compareTo(amount) == -1) {
            logger.error("用户[{}]的账户余额为：[{}]，小于提现金额[{}],不允许提现", userId, drawableCash,
                    amount);
            return ResultUtil.genFailedResult("你的账户余额不足");
        }

        UserAuth userAuth = userAuthIntegrationService.selectUserAuthByUserId(userId);
        if (userAuth == null) {
            logger.error("用户[{}]未绑定openId, 不允许提现", userId);
            return ResultUtil.genFailedResult("请重新登陆后再尝试提现");
        }

        WithdrawFlow lastWithdrawFlow = withdrawMapper.getUserLastWithdraw(userId);
        if (lastWithdrawFlow != null && !(WithdrawEnum.SUCCESS_PAYMENT.getState().equals(lastWithdrawFlow.getState())
                || WithdrawEnum.FAIL_PAYMENT.getState().equals(lastWithdrawFlow.getState()))) {
            logger.error("用户[{}]已存在正在提现的订单，不允许提现", userId);
            return ResultUtil.genFailedResult("你的提现金额正在审批中，请勿重复提现");
        }

        String requestId = nextSequence().toString();
        if (distributedLock.lock(userId.toString(), requestId)) {
            Boolean isRisk = userIntegrationService.isRiskUser(userId);
            WithdrawFlow withdrawFlow = withDraw(userId, userAuth.getIdentifier(), amount, equipment, ip, isRisk, (byte) 1);
            //风险用户不允许直接提现，提现不大于于2元直接提现-----不走审核
            if (!isRisk) {
                BigDecimal limitAmount = new BigDecimal("2");
                if (amount.compareTo(limitAmount) < 1) {
                    withdrawFlow.setState(WithdrawEnum.DURING_PAYMENT.getState());
                    withdrawMapper.updateWithdrawFlow(withdrawFlow);
                    Message result = withdrawProxySerivce.withdraw(withdrawFlow);
                    if (null == result || !result.isSuccess()) {
                        if (null != result && StringUtils.isNotBlank(result.getLastMessage())) {
                            return ResultUtil.genFailedResult(result.getLastMessage());
                        }
                        return ResultUtil.genFailedResult("提现失败");
                    }
                }
            }

            //用户发起提现
            bizLogService.initiateWithdrawal(userId, amount, platform);
            distributedLock.unlock(userId.toString(), requestId);
            return ResultUtil.genSuccessResult(amount);
        } else {
            return ResultUtil.genFailedResult("今日系统提现金额已达上限，请明天再试");
        }
    }

    @Override
    public Message appletUserWithdraw(Long userId, String openId, BigDecimal amount, String devcId, String ip) {
        Message message = Message.build();
        if (!withdrawProperties.getEnableSwitch()) {
            logger.warn("提现开关未开启---不允许提现");
            return message.setSuccess(false).setMessage("不允许提现");
        }

        if (amount.compareTo(new BigDecimal(1)) < 0) {
            return message.setSuccess(false).setMessage("提现金额需要大于或等于1元");
        }

        logger.info("用户[{}]，openId: [{}]发起提现，金额为[{}]", userId, openId, amount);

        Double withdrawAmount = redisStringAdapter.increment(getWxMiniAppAmountKey(), amount.doubleValue());
        redisStringAdapter.expire(getWxMiniAppAmountKey(), DateUtils.getCurSeconds());

        if (payProperties.getMiniAppLimitAmount().compareTo(BigDecimal.valueOf(withdrawAmount)) < 0) {
            return message.setSuccess(false).setMessage("今日兑换已达限额，请明日再来");
        }

        //提现失败，需要回滚每日限额
        String requestId = nextSequence().toString();
        if (distributedLock.lock(userId.toString(), requestId)) {
            logger.debug("用户[{}]进入微信提现请求", userId);

            WithdrawFlow withdrawFlow = withDraw(userId, openId, amount, devcId, ip, false, (byte) 2);
            withdrawFlow.setState(WithdrawEnum.DURING_PAYMENT.getState());

            withdrawMapper.updateWithdrawFlow(withdrawFlow);

            Message result = withdrawProxySerivce.withdraw(withdrawFlow);

            logger.debug("用户提现返回结果：[{}]", result);

            if (null == result || !result.isSuccess()) {
                redisStringAdapter.decrement(getWxMiniAppAmountKey(), amount.doubleValue());
                if (null != result && StringUtils.isNotBlank(result.getLastMessage())) {
                    return result;
                }
                return message.setSuccess(false).setMessage("提现失败");
            }

            //用户发起提现
            bizLogService.initiateWithdrawal(userId, amount, 5);

            distributedLock.unlock(userId.toString(), requestId);
            return message.setSuccess(true).setMessage("提现成功");
        } else {
            redisStringAdapter.decrement(getWxMiniAppAmountKey(), amount.doubleValue());
            logger.debug("用户[{}]未能获取到redis锁----直接返回信息", userId);
            return message.setSuccess(false).setMessage("系统繁忙，请稍后重试");
        }
    }

    /**
     * 用户提现
     *
     * @param userId
     * @param amount
     */
    private WithdrawFlow withDraw(Long userId, String openId, BigDecimal amount, String equipment, String ip,
                                  Boolean isRisk, Byte withdrawType) {
        WithdrawFlow withdrawFlow = new WithdrawFlow();
        Long id = nextSequence();
        withdrawFlow.setId(id);
        withdrawFlow.setOrderNo(generateWithdrawNum());
        withdrawFlow.setPayChannel(getWithdrawChannel());
        withdrawFlow.setUserId(userId);
        withdrawFlow.setPayAccount(openId);
        withdrawFlow.setState(WithdrawEnum.AUDIT.getState());
        //风险用户直接进入延迟审核
        if (isRisk) {
            withdrawFlow.setState(WithdrawEnum.DELAY_AUDIT.getState());
        }
        withdrawFlow.setAmount(amount);
        withdrawFlow.setClientIp(ip);
        withdrawFlow.setEquipment(equipment);
        Date now = new Date();
        withdrawFlow.setCreateTime(now);
        withdrawFlow.setUpdateTime(now);
        withdrawFlow.setWithdrawType(withdrawType);

        //增加提现订单
        withdrawMapper.addWithdrawFlow(withdrawFlow);
        //更新账户信息
        updateUserAccount(userId, amount, id);

        return withdrawFlow;
    }

    /**
     * 提现更新账户信息
     *
     * @param userId
     * @param award
     * @param relationId
     */
    private void updateUserAccount(Long userId, BigDecimal award, Long relationId) {
        AccountCashParam accountCashParam = new AccountCashParam(userId, "DRAWABLEL_CASH", Boolean.FALSE,
                new BigDecimal("-" + award), relationId, "WECHAT_WITHDRAWAL", null);
        userAccountIntegrationService.addCash(accountCashParam);
    }

    /**
     * 生成订单号
     *
     * @return
     */
    private String generateWithdrawNum() {
        String prefix = WithdrawTypeEnum.getNameByType(WithdrawTypeEnum.WX_WITHDRAW.getType());
        return NidGeneratorUtil.getOrderNo(prefix);
    }

    private Long getWithdrawChannel() {
        return WithdrawTypeEnum.WX_WITHDRAW.getId();
    }

    private KeyGenerator getUserWithdrawKey(Long userId) {
        return RedisConfig.USER_WITHDRAW.copy().appendKey(userId);
    }

    private KeyGenerator getWxMiniAppAmountKey() {
        return RedisConfig.WEIXIN_MINI_APP_AMOUNT.copy().appendKey("balance");
    }
}
