package com.bxm.localnews.payment.order;

import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.util.NidGeneratorUtil;
import com.bxm.localnews.facade.UserFeignService;
import com.bxm.localnews.integration.MerchantIntegrationService;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.constant.OrderTypeEnum;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.domain.PaymentOrderMapper;
import com.bxm.localnews.payment.dto.PaymentOrderDTO;
import com.bxm.localnews.payment.param.*;
import com.bxm.localnews.payment.pay.PayProxyService;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.gexin.fastjson.JSON;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 支付订单构建工厂类
 * 进行通用的订单逻辑处理
 * @author liujia
 * @date 2020/05/08 14:28
 */
@Component
@Transactional(rollbackFor = Exception.class)
@Slf4j
@AllArgsConstructor
public class PaymentOrderFactory {

    private final OrderProcesserManager orderProcesserManager;

    private final PayProxyService payProxyService;

    private final PaymentOrderMapper paymentOrderMapper;

    private final SequenceCreater sequenceCreater;

    private final RedisStringAdapter redisStringAdapter;

    private final UserFeignService userFeignService;

    private final MerchantIntegrationService merchantIntegrationService;

    private final PayProperties payProperties;

    /**
     * 订单缓存过期时间
     */
    private static final long CACHE_EXPIRED_SECONDS = 3600 * 6;

    private static final BigDecimal DEFAULT_WITHDRQAW = BigDecimal.valueOf(0.01);

    public PaymentOrderDTO build(UserOrderParam param) {
        OrderProcesser processer = orderProcesserManager.get(param.getOrderType());
        //前置条件判断，是否可以创建订单
        Message message = processer.preCheck(param);

        if (message.isSuccess()) {
            //解决冲突的订单问题
            fixConflictOrder(param);

            //保存订单信息
            PaymentOrder order = createOrder(param, processer);

            //构建订单详情
            PaymentOrderDetail orderDetail = new PaymentOrderDetail();
            BeanUtils.copyProperties(order, orderDetail);
            orderDetail.setAuthCode(param.getWechatCode());
            orderDetail.setReturnUrl(param.getReturnUrl());
            //根据不同的支付平台，处理具体的订单逻辑，主要是回调地址
            payProxyService.create(orderDetail);

            //保存订单信息
            paymentOrderMapper.insertSelective(order);
            //创建订单缓存，用于轮询查询支付状态
            saveOrderCache(order);

//            建立邀请关系
            if(StringUtils.isNotEmpty(param.getInviterId()) && Objects.equals(param.getOrderType(), OrderTypeEnum.WELFARE_VIP.getCode())){
                userFeignService.bindInviteUserId(param.getUserId(),Long.parseLong(param.getInviterId().trim()));
            }
            PaymentOrderDTO paymentOrderDTO = new PaymentOrderDTO(orderDetail.getLink(), order.getPaymentNum());
            BeanUtils.copyProperties(orderDetail,paymentOrderDTO);
            return paymentOrderDTO;
        }

        log.warn("用户: {} 下单失败，请求参数: {} 错误信息: {}", param.getUserId(), JSON.toJSON(param), message.getLastMessage());

        return null;
    }

    /**
     * 创建商家购买商品订单
     * @param param
     * @return
     */
    public PaymentOrderDTO build(UserOrderFacadeParam param) {
        UserOrderParam orderParam = new UserOrderParam();
        BeanUtils.copyProperties(param, orderParam);
        return build(orderParam);
    }



    /**
     * 保存订单缓存信息，供客户端进行轮询
     * @param order 订单信息
     */
    private void saveOrderCache(PaymentOrder order) {
        KeyGenerator key = RedisConfig.WEIXIN_ORDER_STATUS.copy().appendKey(order.getPaymentNum());
        redisStringAdapter.set(key, order.getStatus(), CACHE_EXPIRED_SECONDS);
    }

    /**
     * 创建订单编号
     * @param payType 订单支付类型
     * @return 订单编号
     */
    private String generatePaymentNum(Byte payType) {
        String prefix = PayTypeEnum.getNameByType(payType);
        return NidGeneratorUtil.getOrderNo(prefix);
    }

    /**
     * 创建订单信息
     * @param param     请求参数
     * @param processer 订单处理器
     * @return 订单信息
     */
    private PaymentOrder createOrder(UserOrderParam param, OrderProcesser processer) {
        Date now = new Date();
        String paymentNum = generatePaymentNum(param.getPayType());

        PaymentOrder paymentOrder = new PaymentOrder();
        paymentOrder.setId(sequenceCreater.nextLongId());
        paymentOrder.setPaymentNum(paymentNum);
        paymentOrder.setStatus(PaymentStatusEnum.WAIT.getType());

        //订单金额由订单处理器提供，如果参数包含了金额，则优先使用，这里主要是兼容历史逻辑
        if (param.getAmount() == null) {
            paymentOrder.setAmount(processer.price(param));
        } else {
            if (Objects.equals(payProperties.getPriceMockSwitch(), Boolean.TRUE)) {
                log.info("测试换将创建订单，接mock, 原价: {} 修改现价: {}", param.getAmount(), DEFAULT_WITHDRQAW);
                // mock 订单金额数据为0.01
                paymentOrder.setAmount(DEFAULT_WITHDRQAW);
            } else {
                paymentOrder.setAmount(param.getAmount());
            }
        }

        String paymentTitle;
        if (Objects.equals(param.getOrderType(), OrderTypeEnum.USER_BUY_GOODS.getCode())
                || Objects.equals(param.getOrderType(), OrderTypeEnum.MERCHANT_VIP.getCode())) {
            paymentTitle = param.getPaymentTitle();
        } else {
            paymentTitle = payProperties.getDefaultPaymentTitle();
        }

        paymentOrder.setPayType(param.getPayType());
        paymentOrder.setUserId(param.getUserId());
        paymentOrder.setBizId(param.getBizId());
        paymentOrder.setStartTime(now);
        paymentOrder.setCreateTime(now);
        paymentOrder.setClientIp(param.getRequestIp());
        paymentOrder.setChannel(param.getChnl());
        paymentOrder.setOrderType(processer.match().getCode());
        paymentOrder.setOrderSn(param.getOrderSn());
        paymentOrder.setPlatform(param.getPlatform());
        paymentOrder.setPaymentTitle(paymentTitle);
        return paymentOrder;
    }

    /**
     * 解决冲突的订单
     * 判断相同用户，相同业务类型是否已经存在订单，如果存在则取消前一条
     * @param param 请求参数
     */
    private void fixConflictOrder(UserOrderParam param) {
        PaymentOrder paymentOrder = paymentOrderMapper.getUserPaymentOrderByStatus(param.getUserId(),
                PaymentStatusEnum.WAIT.getType(),
                param.getOrderType());

        if (paymentOrder == null) {
            return;
        }
        paymentOrder.setStatus(PaymentStatusEnum.UNDO.getType());
        paymentOrderMapper.updateByPrimaryKey(paymentOrder);
    }


    /**
     * 发起申请退款
     * @param  payRefundParam : 退单信息
     */
    public Message submitRefund(PayRefundParam payRefundParam) {
        //获取支付订单信息
        if(null == payRefundParam || null == payRefundParam.getOrderId()){
            return Message.build(false);
        }
        PaymentOrder paymentOrder = paymentOrderMapper.queryByOrderId(payRefundParam.getOrderId());

        if (paymentOrder == null) {
            log.info("提交退款申请失败，订单不存在，订单id：{}", paymentOrder.getId());
            return Message.build(false, "订单不存在，订单id为：" + paymentOrder.getId());
        }

        if (!Objects.equals(PaymentStatusEnum.SUCCEED.getType(), paymentOrder.getStatus())
                && !Objects.equals(PaymentStatusEnum.REFUND_FAIL.getType(), paymentOrder.getStatus())) {
            log.info("提交退款申请失败，订单id：{}", paymentOrder.getId());
            return Message.build(false, "当前订单状态不允许退款");
        }
        PaymentRefundParam paymentRefundParam = new PaymentRefundParam();
        paymentRefundParam.setPayOrder(paymentOrder);
        paymentRefundParam.setFullAmount(true);
        return payProxyService.submitRefund(paymentRefundParam);
    }

    /**
     * 发起申请退款
     * @param  param : 退单信息
     */
    public Message submitRefund(RefundParam param) {
        //获取支付订单信息
        if(isNull(param) || isBlank(param.getPaymentNum())){
            return Message.build(false);
        }
        PaymentOrder paymentOrder = paymentOrderMapper.queryByNo(param.getPaymentNum());

        if (isNull(paymentOrder)) {
            log.warn("支付流水不存在，流水单号为：{}", param.getPaymentNum());
            return Message.build(false, "支付流水不存在 流水单号为：" + param.getPaymentNum());
        }

        if (!Objects.equals(PaymentStatusEnum.SUCCEED.getType(), paymentOrder.getStatus())
                && !Objects.equals(PaymentStatusEnum.REFUND_FAIL.getType(), paymentOrder.getStatus())) {
            log.warn("提交退款申请失败，当前状态: {} 不允许退款，支付流水：{}", paymentOrder.getStatus(), param.getPaymentNum());
            return Message.build(false, "当前订单状态不允许退款");
        }

        PaymentRefundParam paymentRefundParam = new PaymentRefundParam();
        paymentRefundParam.setPayOrder(paymentOrder);
        paymentRefundParam.setFullAmount(Objects.equals(param.getFullAmount(), Boolean.TRUE));
        paymentRefundParam.setRefundFee(param.getRefundFee());
        return payProxyService.submitRefund(paymentRefundParam);
    }

    /**
     * 执行退款申请回调
     * @param request 请求参数
     * @param payType 支付类型
     * @return 回调结果
     */
    public String execRefundCallback(String request, PayTypeEnum payType) {
        return payProxyService.execRefundCallback(request,payType);
    }
}
