package com.bxm.localnews.payment.order.impl;

import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.param.UserActivationVipFacadeParam;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.constant.OrderTypeEnum;
import com.bxm.localnews.payment.domain.PaymentOrderMapper;
import com.bxm.localnews.payment.order.OrderProcesser;
import com.bxm.localnews.payment.param.PaymentOrderParam;
import com.bxm.localnews.payment.param.UserOrderParam;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

import static org.springframework.util.CollectionUtils.isEmpty;

/**
 * 福利VIP卡的处理器
 * @author liujia
 * @date 2020/05/08 21:01
 */
@Slf4j
@Component
@AllArgsConstructor
public class WelfareVipOrderProcesser implements OrderProcesser {

    private final PayProperties payProperties;

    private final UserIntegrationService userIntegrationService;

    private final PaymentOrderMapper paymentOrderMapper;


    @Override
    public OrderTypeEnum match() {
        return OrderTypeEnum.WELFARE_VIP;
    }

    @Override
    public Message preCheck(UserOrderParam param) {
        // 校验用户是否已经开卡
        if (Objects.isNull(param.getUserId())) {
            log.warn("用户id不存在，无法下单");
            return Message.build(false, "用户id不存在");
        }

        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(param.getUserId());
        if (Objects.isNull(userInfoDTO)) {
            log.warn("用户id: {} 对应的用户不存在，无法下单", param.getUserId());
            return Message.build(false, "用户不存在");
        }

        if (Objects.equals(userInfoDTO.getIsVip(), 1)) {
            return Message.build(false, "您已是VIP，请勿重复下单");
        }

        // 用户缓存可能有延迟，查询支付订单
        PaymentOrderParam paymentOrderParam = new PaymentOrderParam();
        paymentOrderParam.setUserId(param.getUserId());
        // 开通vip的订单类型
        paymentOrderParam.setOrderType(OrderTypeEnum.WELFARE_VIP.getCode());
        // 已支付成功的
        paymentOrderParam.setStatus((byte) 3);

        List<PaymentOrder> paymentOrders = paymentOrderMapper.selectUserOrder(paymentOrderParam);
        if (!isEmpty(paymentOrders)) {
            return Message.build(false, "已经购买VIP的成功订单，请勿重复下单");
        }

        return Message.build();
    }

    @Override
    public BigDecimal price(UserOrderParam param) {
        //todo ： 这里先暂时不做处理，目前只有一种VIP卡，后期多个的话这里需要处理
        return payProperties.getWelfareVipPrice();
    }

    @Override
    public Message afterPayment(PaymentOrder paymentOrder) {
        log.debug("用户购买激活VIP，对应订单：{}", paymentOrder);

        //升级成为VIP以及给师傅发送相关推送等操作
        UserActivationVipFacadeParam param = new UserActivationVipFacadeParam();
        param.setChannel(paymentOrder.getChannel());
        param.setOrderId(paymentOrder.getId());
        param.setPlatformEnum(paymentOrder.getPlatform());
        param.setUserId(paymentOrder.getUserId());
        //这里默认给个1，暂时只有一种VIP卡类型，后期在进行拓展动态
        param.setVipCardId("1");

        return Message.build(userIntegrationService.activationVip(param));
    }

    @Override
    public Message afterRefund(PaymentOrder paymentOrder) {

        return null;
    }
}
