package com.bxm.localnews.payment.pay.impl;

import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.domain.PaymentOrderMapper;
import com.bxm.localnews.payment.domain.UserPayorderRefundMapper;
import com.bxm.localnews.payment.dto.NotifyContext;
import com.bxm.localnews.payment.dto.PayCallBackResult;
import com.bxm.localnews.payment.param.PaymentRefundParam;
import com.bxm.localnews.payment.param.UserPayorderRefundBean;
import com.bxm.localnews.payment.pay.PayModeService;
import com.bxm.localnews.payment.pay.PaymentOrderService;
import com.bxm.localnews.payment.pay.callback.listener.CallbackNotifier;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.gexin.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 抽象的付款方式
 *
 * @author liujia 2018/6/25 19:47
 */
@Slf4j
public abstract class AbstractPayModeService implements PayModeService {

    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private PaymentOrderMapper paymentOrderMapper;

    @Autowired
    private UserPayorderRefundMapper userPayorderRefundMapper;

    @Autowired
    protected PaymentOrderService payService;

    @Resource
    private SequenceCreater sequenceCreater;

    @Autowired
    private CallbackNotifier callbackNotifier;

    /**
     * 将退款订单状态更新为已退款状态
     *
     * @param refundOrderNo 退款订单编号
     * @param result        退款结果
     * @param status        更新状态
     */
    protected void changeRefundOrderStatus(String refundOrderNo,
                                           String refundTradeNo,
                                           String result,
                                           PaymentStatusEnum status) {
        userPayorderRefundMapper.updateByOrderNo(
                UserPayorderRefundBean.builder()
                        .refundOrderNo(refundOrderNo)
                        .refundTradeNo(refundTradeNo)
                        .result(result)
                        .status(status.getType())
                        .build());

        paymentOrderMapper.updateRefundOrder(refundTradeNo, status.getType());

    }

    /**
     * 在调用申请退款成功后，创建退款单
     *
     * @param param         退款订单参数
     * @param refundOrderNo 退款订单编号
     * @param refundFee     退款订单金额
     */
    protected void createRefundOrder(PaymentRefundParam param,
                                     String refundOrderNo,
                                     BigDecimal refundFee) {

        UserPayorderRefundBean history = userPayorderRefundMapper.selectByOrderNo(refundOrderNo);
        //新增退款订单
        UserPayorderRefundBean refundBean = UserPayorderRefundBean.builder()
                .id(sequenceCreater.nextLongId())
                .createTime(new Date())
                .operatorId(param.getOperatorId())
                .orderId(param.getPayOrder().getId())
                .refundOrderNo(refundOrderNo)
                .refundFee(refundFee)
                .remark(param.getRemark())
                .status(PaymentStatusEnum.REFUNDING.getType())
                .build();

        if (null != history) {
            refundBean.setId(history.getId());
            refundBean.setResult("");
            refundBean.setCreateTime(null);

            userPayorderRefundMapper.updateByPrimaryKeySelective(refundBean);
        } else {
            userPayorderRefundMapper.insert(refundBean);
        }

        // TODO 更新订单状态
        // 这里的代码是有问题的 3.11.0版本之后有了多订单，支付时，使用父订单号关联进行支付，子订单在退款之后，就将父订单对应的支付单号状态变为了已退款，导致再进行支付时无法支付
        // 要从头开始写风险较大，处理方式为先不进行支付流水订单状态的变更，如果想要看订单是否进行了退款，通过查看退款单号进行排查
        // 下个版本支付这块进行优化，将分批退款进行处理下
        // paymentOrderMapper.updateRefundOrder(param.getPayOrder().getPaymentNum(), PaymentStatusEnum.REFUNDING.getType());

    }

    /**
     * 根据付款订单构建退款订单编号
     *
     * @param param 退款订单创建参数
     * @return 退款订单编号
     */
    protected String buildRefundOrderNo(PaymentRefundParam param) {
        return "R" + param.getPayOrder().getPaymentNum();
    }

    @Override
    public String callBack(String data) {
        if (isBlank(data)) {
            return null;
        }

        PayCallBackResult callBackResult = doCallBack(data);

        if (Objects.nonNull(callBackResult) && callBackResult.isCallBackSuccess()) {
            // 回调成功后的处理
            callbackNotify(true, callBackResult.getPaymentOrder(), data);
            return callBackResult.getData();
        }
        // 回调处理失败的处理
        callbackNotify(false, null, data);
        return null;
    }

    /**
     * 支付回调之后的处理
     * @param callBackSuccess 回调是否处理成功
     * @param paymentOrder 支付订单信息
     * @param callbackData 回调数据
     */
    private void callbackNotify(boolean callBackSuccess, PaymentOrder paymentOrder, String callbackData) {
        NotifyContext context =  new NotifyContext();
        context.setCallBackSuccess(callBackSuccess);
        context.setCallbackData(callbackData);
        context.setOrder(paymentOrder);

        try {
            callbackNotifier.callbackNotifyAll(context);
        } catch(Exception e) {
            log.error("支付回调处理后通知失败, context: {}", JSON.toJSONString(context), e);
        }
    }


    /**
     * 处理回调的方法
     * @param data 第三方回调数据
     * @return 是否回调成功
     */
    protected abstract PayCallBackResult doCallBack(String data);


    protected PayCallBackResult buildSuccessResult(String data, PaymentOrder paymentOrder) {
        return PayCallBackResult.builder().callBackSuccess(true).data(data).paymentOrder(paymentOrder).build();
    }

    protected PayCallBackResult buildFailedResult() {
        return PayCallBackResult.builder().callBackSuccess(false).build();
    }

}
