package com.bxm.localnews.payment.pay.alipay;

import com.alipay.api.AlipayClient;
import com.alipay.api.request.AlipayFundTransOrderQueryRequest;
import com.alipay.api.request.AlipayFundTransUniTransferRequest;
import com.alipay.api.response.AlipayFundTransOrderQueryResponse;
import com.alipay.api.response.AlipayFundTransUniTransferResponse;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.payment.config.WithdrawProperties;
import com.bxm.localnews.payment.constant.WithdrawEnum;
import com.bxm.localnews.payment.constant.WithdrawTypeEnum;
import com.bxm.localnews.payment.domain.WithdrawMapper;
import com.bxm.localnews.payment.param.AlipayFundTransUniTransferParam;
import com.bxm.localnews.payment.param.AlipayFundTransUniTransferPayeeInfoParam;
import com.bxm.localnews.payment.param.MerchantWithdrawParam;
import com.bxm.localnews.payment.service.WithdrawService;
import com.bxm.localnews.payment.vo.WithdrawFlow;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Map;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;

@Slf4j
@Service
@AllArgsConstructor
public class AliWithdrawServiceImpl implements WithdrawService {

    private final AlipayClient alipayCertClient;

    private final SequenceCreater sequenceCreater;

    private final DistributedLock distributedLock;

    private final WithdrawMapper withdrawMapper;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final WithdrawProperties withdrawProperties;

    @Override
    public WithdrawTypeEnum support() {
        return WithdrawTypeEnum.ALI_WITHDRAW;
    }

    @Override
    public Message withdraw(WithdrawFlow withdrawFlow) {
        log.info("提现用户[{}]的提现账号为：[{}]，提现金额为：[{}]", withdrawFlow.getUserId(), withdrawFlow.getPayAccount(),
                withdrawFlow.getAmount());
        //微信提现返回的信息
        String requestId = sequenceCreater.nextLongId().toString();
        String key = withdrawFlow.getUserId() + withdrawFlow.getOrderNo();
        if (!distributedLock.lock(key, requestId)) {
            return Message.build(false, "请勿频繁操作");
        }

        Message message;
        if ((message = doAlipayWithdraw(withdrawFlow)).isSuccess()) {
            afterPay(withdrawFlow, WithdrawEnum.SUCCESS_PAYMENT.getState(), "");
            distributedLock.unlock(key, requestId);
        } else {
            afterPay(withdrawFlow, WithdrawEnum.FAIL_PAYMENT.getState(), message.getLastMessage());
        }

        if (!message.isSuccess()) {
            log.warn("用户: {}发起支付宝提现失败，请求参数: {}失败信息: {}", withdrawFlow.getUserId(), toJSONString(withdrawFlow),
                    message.getMessages());
        }
        return message;
    }

    @Override
    public Message merchantWithdraw(MerchantWithdrawParam param) {
        log.info("商户: {} 发起提现打款 请求参数: {}", param.getMerchantId(), toJSONString(param));
        // 加锁
        String requestId = sequenceCreater.nextLongId().toString();
        String key = StringUtils.join(param.getMerchantId(), param.getWithdrawNo());
        if (!distributedLock.lock(key, requestId)) {
            return Message.build(false, "请勿频繁操作");
        }

        // 调用支付宝打款
        Message message = doAlipayWithdraw(param.getPayAccount(), param.getRealName(), param.getWithdrawNo(),
                param.getAmount());
        if (!message.isSuccess()) {
            log.warn("商户: {}发起支付宝提现失败，请求参数: {}失败信息: {}", param.getMerchantId(),
                    toJSONString(param), toJSONString(message));
        }

        return message;
    }

    /**
     * 更新账户信息
     * @param withdrawFlow
     */
    private Message updateUserAccount(WithdrawFlow withdrawFlow, Byte withdrawState) {
        if (withdrawFlow != null) {
            return userAccountIntegrationService.updateUserWithdraw(withdrawFlow.getUserId(),
                    withdrawFlow.getAmount(), withdrawState);
        }
        return Message.build(false);
    }

    private Message doAlipayWithdraw(WithdrawFlow withdrawFlow) {
        return doAlipayWithdraw(withdrawFlow.getPayAccount(), withdrawFlow.getRealName(), withdrawFlow.getOrderNo(),
                withdrawFlow.getAmount());
    }

    private Message doAlipayWithdraw(String payAccount, String realName, String orderNo, BigDecimal amount) {
        if (!Objects.equals(withdrawProperties.getRemitSwitch(), Boolean.TRUE)) {
            log.info("打款开关关闭，原提现价格: {} 修改提现价格为0.1", amount);
            amount = BigDecimal.valueOf(0.1);
        }

        AlipayFundTransUniTransferRequest request = new AlipayFundTransUniTransferRequest();

        // 收款人信息
        AlipayFundTransUniTransferPayeeInfoParam payeeInfoParam = new AlipayFundTransUniTransferPayeeInfoParam();
        payeeInfoParam.setIdentity(payAccount);
        // ALIPAY_LOGON_ID：支付宝登录号，支持邮箱和手机号格式
        payeeInfoParam.setIdentityType("ALIPAY_LOGON_ID");
        // 真实姓名
        payeeInfoParam.setName(realName);

        // 转账参数
        AlipayFundTransUniTransferParam transUniTransferParam = new AlipayFundTransUniTransferParam();
        transUniTransferParam.setOutBizNo(orderNo);
        transUniTransferParam.setTransAmount(amount);
        // 单笔无密转账到支付宝账户固定为: TRANS_ACCOUNT_NO_PWD；
        transUniTransferParam.setProductCode("TRANS_ACCOUNT_NO_PWD");
        // DIRECT_TRANSFER：单笔无密转账到支付宝/银行卡, B2C现金红包;
        transUniTransferParam.setBizScene("DIRECT_TRANSFER");
        transUniTransferParam.setOrderTitle("本地万事通提现");
        transUniTransferParam.setPayeeInfo(payeeInfoParam);
        transUniTransferParam.setRemark("本地万事通提现");
        // transUniTransferParam.setBusinessParams();

        request.setBizContent(toJSONString(transUniTransferParam));

        try {
            if (log.isDebugEnabled()) {
                log.debug("发起支付宝提现，请求参数: {}", toJSONString(request));
            }
            AlipayFundTransUniTransferResponse response = alipayCertClient.certificateExecute(request);

            if (log.isDebugEnabled()) {
                log.debug("发起支付宝提现，返回参数: {}", toJSONString(response));
            }

            if (Objects.equals(response.getStatus(), "SUCCESS")) {
                return Message.build(true);
            }

            log.warn("调用支付宝提现失败，payAccount: {}, realName: {}, orderNo: {}, amount: {}, transUniTransferParam: {}, response: {}" ,
                    payAccount, realName, orderNo, amount,
                    toJSONString(transUniTransferParam), toJSONString(response));
        } catch(Exception e) {
            log.error("调用支付宝提现申请失败，payAccount: {}, realName: {}, orderNo: {}, amount: {}, transUniTransferParam: {}" ,
                    payAccount, realName, orderNo, amount,
                    toJSONString(transUniTransferParam), e);
        }
        return Message.build(false, "支付宝提现出现错误");
    }

    /**
     * 将提现记录改为失败 or 成功
     */
    private void afterPay(WithdrawFlow withdrawFlow, Byte state, String reason) {
        // 设置提现状态为支付失败 or 成功
        WithdrawFlow update = new WithdrawFlow();
        update.setId(withdrawFlow.getId());
        update.setRemark(reason);
        update.setState(state);
        withdrawMapper.updateWithdrawFlow(update);

        // 提现操作完成后调用账号变更，更新为提现失败 or 成功
        updateUserAccount(withdrawFlow, state);
    }

    @Override
    public Message queryWithdraw(WithdrawFlow withdrawFlow) {
        this.query(withdrawFlow);
        return Message.build();
    }

    private Message query(WithdrawFlow withdrawFlow) {
        Date now = new Date();
        withdrawFlow.setUpdateTime(now);

        AlipayFundTransOrderQueryRequest request = new AlipayFundTransOrderQueryRequest();
        Map<String, Object> query = Maps.newHashMap();
        query.put("out_biz_no", withdrawFlow.getOrderNo());
        request.setBizContent(toJSONString(query));
        try {
            if (log.isDebugEnabled()) {
                log.debug("查询支付宝订单信息: 请求参数{}", toJSONString(request));
            }
            AlipayFundTransOrderQueryResponse response = alipayCertClient.execute(request);

            if (log.isDebugEnabled()) {
                log.debug("查询支付宝订单信息: 返回参数{}", toJSONString(response));
            }

            if (!Objects.equals(response.getStatus(), "SUCCESS")) {
                withdrawFlow.setState(WithdrawEnum.FAIL_PAYMENT.getState());
                log.warn("调用支付宝提现失败，withdrawFlow: {}, response: {}", toJSONString(withdrawFlow),
                        toJSONString(response));

                return Message.build(false, response.getFailReason());
            } else {
                // TODO 设置状态 更新
                return Message.build(true);
            }
        } catch(Exception e) {
            log.error("调用支付宝提现查询失败, withdrawFlow: {}", toJSONString(withdrawFlow), e);
        }

        return Message.build(false, "查询失败");
    }

    @Override
    public String callback(String data) {
        return null;
    }
}
