package com.bxm.localnews.payment.pay.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.domain.PaymentOrderMapper;
import com.bxm.localnews.payment.dto.OrderStatusDTO;
import com.bxm.localnews.payment.order.OrderProcesserManager;
import com.bxm.localnews.payment.pay.PayProxyService;
import com.bxm.localnews.payment.pay.PaymentOrderService;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Objects;

@Service
@Slf4j
public class PaymentOrderServiceImpl implements PaymentOrderService {

    private PaymentOrderMapper paymentOrderMapper;

    private PayProxyService payProxyService;

    private RedisStringAdapter redisStringAdapter;

    private final OrderProcesserManager orderProcesserManager;

    private final BizLogService bizLogService;

    @Autowired
    public PaymentOrderServiceImpl(PaymentOrderMapper paymentOrderMapper,
                                   PayProxyService payProxyService,
                                   RedisStringAdapter redisStringAdapter,
                                   OrderProcesserManager orderProcesserManager,
                                   BizLogService bizLogService) {
        this.paymentOrderMapper = paymentOrderMapper;
        this.payProxyService = payProxyService;
        this.redisStringAdapter = redisStringAdapter;
        this.orderProcesserManager = orderProcesserManager;
        this.bizLogService = bizLogService;
    }

    @Override
    public OrderStatusDTO queryOrder(String paymentNum) {
        return new OrderStatusDTO(paymentNum, getOrderStatus(paymentNum));
    }

    @Override
    public Message modifyStatus(PaymentOrder paymentOrder) {
        // 删除了历史激活本地vip的逻辑
        return modifyPaymentOrderStatus(paymentOrder);
    }

    @Override
    public Message modifyPaymentOrderStatus(PaymentOrder paymentOrder) {
        log.debug("订单状态变更，请求参数：{}", JSON.toJSONString(paymentOrder));

        if (paymentOrderMapper.updateByPrimaryKey(paymentOrder) > 0) {
            redisStringAdapter.set(getOrderStatusKey(paymentOrder.getPaymentNum()), paymentOrder.getStatus(), 3600 * 24);
        }

        Message message = Message.build();
        //如果是支付成功，进行对应的订单逻辑回调处理
        if (Objects.equals(PaymentStatusEnum.SUCCEED.getType(), paymentOrder.getStatus())) {
            message.append(orderProcesserManager.get(paymentOrder.getOrderType()).afterPayment(paymentOrder));

            if (!message.isSuccess()) {
                log.error("订单状态变更失败：{},订单参数：{}",
                        message.getLastMessage(),
                        JSON.toJSONString(paymentOrder));
            } else {
                // 上报购买订单生效埋点
                PointReportParam reportParam = PointReportParam
                        .build()
                        .e("3034")
                        .ev("118." + paymentOrder.getId())
                        .put("uid", String.valueOf(paymentOrder.getUserId()));

                if (paymentOrder.getPlatform() != null) {
                    reportParam.dtype(paymentOrder.getPlatform().toString());
                }

                bizLogService.report(reportParam);
            }
        }

        return message;
    }

    @Override
    public PaymentOrder getPaymentOrderByPaymentNo(String paymentNum) {
        return paymentOrderMapper.queryByNo(paymentNum);
    }

    @Override
    public OrderStatusDTO queryOrderStatus(String paymentNum, Byte payType) {
        PaymentOrder paymentOrder = getPaymentOrderByPaymentNo(paymentNum);
        //获取环境
        if (PaymentStatusEnum.WAIT.getType().equals(paymentOrder.getStatus())) {
            payProxyService.query(paymentOrder, PayTypeEnum.getPayTypeEnum(payType));
        }
        return new OrderStatusDTO(paymentNum, getOrderStatus(paymentNum));
    }

    /**
     * 获取订单状态
     *
     * @param paymentNum
     * @return
     */
    private KeyGenerator getOrderStatusKey(String paymentNum) {
        return RedisConfig.WEIXIN_ORDER_STATUS.copy().appendKey(paymentNum);
    }

    /**
     * 获取缓存中的订单状态
     *
     * @param paymentNum
     * @return
     */
    private Byte getOrderStatus(String paymentNum) {
        String status = redisStringAdapter.getString(getOrderStatusKey(paymentNum));

        //不存在的订单，尝试从库中获取
        if (StringUtils.isEmpty(status)) {
            PaymentOrder paymentOrder = getPaymentOrderByPaymentNo(paymentNum);

            if (paymentOrder == null) {
                log.info("请求查询的订单编号[{}]不存在", paymentNum);

                redisStringAdapter.set(getOrderStatusKey(paymentNum), PaymentStatusEnum.FAILED.getType(), 3600);
                return null;
            }

            redisStringAdapter.set(getOrderStatusKey(paymentNum), paymentOrder.getStatus(), 3600 * 24);
            return paymentOrder.getStatus();

        }

        return Byte.valueOf(status);
    }
}
