package com.bxm.localnews.payment.pay.wechat;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.service.BizLogService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.integration.PushMsgIntegrationService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.payment.config.WxPayProperties;
import com.bxm.localnews.payment.constant.WithdrawEnum;
import com.bxm.localnews.payment.constant.WithdrawTypeEnum;
import com.bxm.localnews.payment.constant.WxWithdrawEnum;
import com.bxm.localnews.payment.domain.WithdrawMapper;
import com.bxm.localnews.payment.param.MerchantWithdrawParam;
import com.bxm.localnews.payment.request.WxPayQueryWithdrawRequest;
import com.bxm.localnews.payment.request.WxPayWithdrawRequest;
import com.bxm.localnews.payment.result.QueryWithdrawResult;
import com.bxm.localnews.payment.result.WxWithdrawResult;
import com.bxm.localnews.payment.service.WithdrawService;
import com.bxm.localnews.payment.vo.WithdrawFlow;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.exception.WxPayException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;

/**
 * 微信提现实现类
 * @author zhaoyadong 2019/3/11 15:17
 * @desc
 */
@Service
public class WechatWithdrawServiceImpl implements WithdrawService {

    protected static final Logger logger = LoggerFactory.getLogger(WechatWithdrawServiceImpl.class);

    private final WxWithdrawProxyService wxWithdrawProxyService;

    private final WxPayProperties wxPayProperties;

    private final WithdrawMapper withdrawMapper;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final PushMsgIntegrationService pushMsgIntegrationService;

    private final BizLogService bizLogService;

    private final DistributedLock distributedLock;

    private final SequenceCreater sequenceCreater;

    private final RedisStringAdapter redisStringAdapter;

    @Autowired
    public WechatWithdrawServiceImpl(WxWithdrawProxyService wxWithdrawProxyService,
                                     WxPayProperties wxPayProperties,
                                     WithdrawMapper withdrawMapper,
                                     UserAccountIntegrationService userAccountIntegrationService,
                                     PushMsgIntegrationService pushMsgIntegrationService,
                                     BizLogService bizLogService,
                                     DistributedLock distributedLock,
                                     SequenceCreater sequenceCreater,
                                     RedisStringAdapter redisStringAdapter) {
        this.wxWithdrawProxyService = wxWithdrawProxyService;
        this.wxPayProperties = wxPayProperties;
        this.withdrawMapper = withdrawMapper;
        this.userAccountIntegrationService = userAccountIntegrationService;
        this.pushMsgIntegrationService = pushMsgIntegrationService;
        this.bizLogService = bizLogService;
        this.distributedLock = distributedLock;
        this.sequenceCreater = sequenceCreater;
        this.redisStringAdapter = redisStringAdapter;
    }

    @Override
    public WithdrawTypeEnum support() {
        return WithdrawTypeEnum.WX_WITHDRAW;
    }

    @Override
    public Message withdraw(WithdrawFlow withdrawFlow) {
        Message message = Message.build();
        logger.info("提现用户[{}]的提现账号为：[{}]，提现金额为：[{}]", withdrawFlow.getUserId(), withdrawFlow.getPayAccount(),
                withdrawFlow.getAmount());
        //微信提现返回的信息
        String requestId = sequenceCreater.nextLongId().toString();
        if (distributedLock.lock(withdrawFlow.getUserId() + withdrawFlow.getOrderNo(), requestId)) {
            // 第三方提现
            wxWithdraw(withdrawFlow);
            // 提现操作完成后调用账号变更，更新为提现成功或失败
            message = updateUserAccount(withdrawFlow);

            distributedLock.unlock(withdrawFlow.getUserId() + withdrawFlow.getOrderNo(), requestId);
        }
        if (!"SUCCESS".equals(withdrawFlow.getRemark())) {
            logger.warn("用户提现失败，提现信息[{}]", JSON.toJSONString(withdrawFlow));
            message.setSuccess(false).setMessage("提现失败，请稍后再试");
        } else {
            message.setSuccess(true).setMessage(StringUtils.EMPTY);
        }
        return message;
    }

    @Override
    public Message merchantWithdraw(MerchantWithdrawParam param) {
        // 微信暂时没支持
        return null;
    }

    @Override
    public Message queryWithdraw(WithdrawFlow withdrawFlow) {
        Message message = Message.build();
        this.queryWxWithdraw(withdrawFlow);
        return message;
    }

    /**
     * 更新账户信息及其邀请人账户信息
     * @param wxWithdrawFlow
     */
    private Message updateUserAccount(WithdrawFlow wxWithdrawFlow) {
        if (wxWithdrawFlow != null) {
            return userAccountIntegrationService.updateUserWithdraw(wxWithdrawFlow.getUserId(),
                    wxWithdrawFlow.getAmount(),
                    wxWithdrawFlow.getState());
        }
        return Message.build(false);
    }

    @Override
    public String callback(String data) {
        return null;
    }

    /**
     * 发起提现
     * @param withdrawFlow 提现请求参数
     * @return 提现请求参数
     */
    private WithdrawFlow wxWithdraw(WithdrawFlow withdrawFlow) {
        Date now = new Date();
        WxPayWithdrawRequest wxPayWithdrawRequest = generateWithdraw(withdrawFlow);
        try {
            WxWithdrawResult wxWithdrawResult = wxWithdrawProxyService.transfers(wxPayWithdrawRequest, withdrawFlow.getWithdrawType());

            logger.info("微信返回信息[{}]", JSON.toJSONString(wxWithdrawResult));

            if (!"SUCCESS".equals(wxWithdrawResult.getReturnCode()) ||
                    !"SUCCESS".equals(wxWithdrawResult.getResultCode())) {
                withdrawFlow.setState(WithdrawEnum.FAIL_PAYMENT.getState());
                if ("V2_ACCOUNT_SIMPLE_BAN".equals(wxWithdrawResult.getErrCode())) {
                    //未实名用户
                    logger.warn("该用户[{}]的微信未实名认证", withdrawFlow.getUserId());
                    pushWithdrawFailMessage(withdrawFlow.getUserId());
                } else {
                    logger.warn("用户[{}]提现失败", withdrawFlow.getUserId());
                }
                withdrawFlow.setRemark(wxWithdrawResult.getErrCode());
            } else {//成功
                logger.info("用户[{}]提现成功，金额为：[{}]", withdrawFlow.getUserId(), withdrawFlow.getAmount());
                withdrawFlow.setState(WithdrawEnum.SUCCESS_PAYMENT.getState());
                withdrawFlow.setRemark(wxWithdrawResult.getResultCode());
                withdrawFlow.setPaymentNo(wxWithdrawResult.getPaymentNo());
                withdrawFlow.setPaymentTime(wxWithdrawResult.getPaymentTime());

                pushWithdrawSuccessMessage(withdrawFlow.getUserId(), withdrawFlow.getAmount());
                bizLogService.successWithdrawal(withdrawFlow.getUserId(), withdrawFlow.getAmount());

                deductCompanyAccount(withdrawFlow.getAmount());
            }

            withdrawFlow.setUpdateTime(now);
            withdrawMapper.updateWithdrawFlow(withdrawFlow);
            return withdrawFlow;
        } catch (WxPayException e) {
            logger.error("用户[{}]提现金额为[{}]发生错误：[{}]", withdrawFlow.getUserId(), withdrawFlow.getAmount(),
                    e.getMessage());
            logger.error("微信提现发生错误", e);
            return null;
        }
    }

    /**
     * 生成微信提现信息
     * @return
     */
    private WxPayWithdrawRequest generateWithdraw(WithdrawFlow withdrawFlow) {
        WxPayWithdrawRequest wxPayWithdrawRequest = new WxPayWithdrawRequest();

        //如果是app提现，运营后台触发的将传递null
        if (null == withdrawFlow.getWithdrawType() || (byte) 1 == withdrawFlow.getWithdrawType()) {
            wxPayWithdrawRequest.setAppid(wxPayProperties.getWithdrawAppId());
        } else if ((byte) 2 == withdrawFlow.getWithdrawType()) {
            wxPayWithdrawRequest.setAppid(wxPayProperties.getAppletAppId());
        }
        wxPayWithdrawRequest.setOpenId(withdrawFlow.getPayAccount());
        wxPayWithdrawRequest.setAmount(BaseWxPayRequest.yuanToFen(withdrawFlow.getAmount().toString()));
        wxPayWithdrawRequest.setCheckName("NO_CHECK");
        wxPayWithdrawRequest.setDesc(wxPayProperties.getDesc());
        wxPayWithdrawRequest.setPartnerTradeNo(withdrawFlow.getOrderNo());
        wxPayWithdrawRequest.setSpbillCreateIp(withdrawFlow.getClientIp());
        return wxPayWithdrawRequest;
    }

    /**
     * 查询企业付款信息
     * @param withdrawFlow
     * @return
     */
    private WithdrawFlow queryWxWithdraw(WithdrawFlow withdrawFlow) {
        Date now = new Date();
        withdrawFlow.setUpdateTime(now);
        WxPayQueryWithdrawRequest request = generateQueryWithdraw(withdrawFlow);
        try {
            QueryWithdrawResult result = wxWithdrawProxyService.queryWithdraw(request);
            logger.info("微信返回信息[{}]", JSON.toJSONString(result));
            if (!"SUCCESS".equals(result.getReturnCode()) ||
                    !"SUCCESS".equals(result.getResultCode())) {
                withdrawFlow.setState(WithdrawEnum.FAIL_PAYMENT.getState());
                logger.info("用户[{}]查询订单[{}]付款信息失败:订单信息状态：[{}]", withdrawFlow.getUserId(), withdrawFlow.getPaymentNo(),
                        result.getErrCode());
//                withdrawFlow.setRemark(result.getErrCode());
            } else {//成功
                logger.info("查询企业付款信息：用户[{}]提现的金额为：[{}], 转账状态：[{}], 失败原因：[{}]",
                        withdrawFlow.getUserId(), withdrawFlow.getAmount(), result.getStatus(), result.getReason());
                withdrawFlow.setState(WxWithdrawEnum.getWxWithdrawEnumByStatus(result.getStatus()).getState());
                withdrawFlow.setRemark(result.getResultCode());
                withdrawFlow.setPaymentNo(result.getPartnerTradeNo());
                withdrawFlow.setPaymentTime(result.getPaymentTime());
                withdrawMapper.updateWithdrawFlow(withdrawFlow);
            }
            return withdrawFlow;
        } catch (WxPayException e) {
            return null;
        }
    }

    /**
     * 生成企业付款
     * @param withdrawFlow
     * @return
     */
    private WxPayQueryWithdrawRequest generateQueryWithdraw(WithdrawFlow withdrawFlow) {
        WxPayQueryWithdrawRequest wxPayQueryWithdrawRequest = new WxPayQueryWithdrawRequest();
        wxPayQueryWithdrawRequest.setPartnerTradeNo(withdrawFlow.getOrderNo());
        return wxPayQueryWithdrawRequest;
    }

    /**
     * 推送未实名认证的提现失败消息
     * @param userId 推送的目标用户
     */
    private void pushWithdrawFailMessage(Long userId) {
        String title = "您的微信号未实名认证，提现失败，请实名认证后再提现。";
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.FAIL_WITHDRAW);

        PushMessage message = PushMessage.build();
        message.setTitle("通知消息");
        message.setContent(title);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
        message.setPayloadInfo(info);

        this.pushMsgIntegrationService.pushMsg((message));
    }

    private void pushWithdrawSuccessMessage(Long userId, BigDecimal amount) {
        String title = "您的" + amount + "元提现已成功打款，可在微信钱包内查看";
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.SUCCESS_WITHDRAW);

        PushMessage message = PushMessage.build();
        message.setTitle("通知消息");
        message.setContent(title);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
        message.setPayloadInfo(info);

        this.pushMsgIntegrationService.pushMsg((message));
    }

    /**
     * 扣除公司账户余额
     */
    private void deductCompanyAccount(BigDecimal amount) {
        redisStringAdapter.decrement(getCompanyAmountKey(), amount.doubleValue());
    }

    /**
     * 公司账户余额key
     * @return
     */
    private KeyGenerator getCompanyAmountKey() {
        return RedisConfig.COMPANY_REMAIN_AMOUNT.copy().appendKey("balance");
    }
}
