package com.bxm.localnews.payment.service.impl;

import com.bxm.localnews.msg.integration.SmsIntegrationService;
import com.bxm.localnews.payment.domain.PaymentWithdrawAccountExtendMapper;
import com.bxm.localnews.payment.entity.PaymentWithdrawAccount;
import com.bxm.localnews.payment.param.AlipayAccountBindingParam;
import com.bxm.localnews.payment.service.PaymentWithdrawAccountService;
import com.bxm.localnews.payment.vo.WithdrawAccountVO;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.bxm.localnews.mq.common.constant.AppConst.SMS_TPL.ALIPAY_ACCOUNT_BIND;
import static com.bxm.localnews.payment.constant.RedisLockKey.BIND_WITHDRAW_ACCOUNT;
import static com.bxm.localnews.payment.constant.WithdrawAccountType.ALIPAY;

/**
 * @author gonzo
 * @date 2020-08-10 19:24
 **/
@Slf4j
@Service
@AllArgsConstructor
public class PaymentWithdrawAccountServiceImpl implements PaymentWithdrawAccountService {

    private final PaymentWithdrawAccountExtendMapper paymentWithdrawAccountExtendMapper;

    private final DistributedLock lock;

    private final SequenceCreater sequenceCreater;

    private final SmsIntegrationService smsIntegrationService;

    /**
     * 同时支持手机号和邮箱的正则表达式
     */
    private static final Pattern ALU_PAY_ACCOUNT_REG =  Pattern.compile("^(?:1[3-9]\\d{9}|[a-zA-Z\\d._-]*\\@[a-zA-Z\\d.-]{1,10}\\.[a-zA-Z\\d]{1,20})$");

    @Override
    public Message bindOrUpdateAlipayAccount(AlipayAccountBindingParam param) {

        // 加锁 避免重复递交
        String lockKey = BIND_WITHDRAW_ACCOUNT.copy().appendKey(Objects.toString(param.getUserId())).gen();
        String lockId = sequenceCreater.nextStringId();
        if (!lock.lock(lockKey, lockId)) {
            return Message.build(false).setMessage("请勿频繁提交");
        }

        // 验证码校验
        if (!Objects.equals(smsIntegrationService.verifyByType(ALIPAY_ACCOUNT_BIND, param.getPhone(), param.getSmsCode()),
                Boolean.TRUE)) {
            return Message.build(false).setMessage("验证码错误");
        }

        // 支付宝账号校验
        Matcher m = ALU_PAY_ACCOUNT_REG.matcher(param.getAccount());
        if (!m.matches()) {
            return Message.build(false).setMessage("请输入正确的支付宝账号");
        }

        Date now = new Date();
        List<PaymentWithdrawAccount> accounts = paymentWithdrawAccountExtendMapper.selectByUserIdAndType(param.getUserId(), ALIPAY);

        if (!CollectionUtils.isEmpty(accounts)) {
            if (accounts.size() > 1) {
                log.warn("用户: {} 的提现账号类型: {} 有: {} 个账号，只更新最后一个", param.getUserId(), ALIPAY, accounts.size());
            }

            PaymentWithdrawAccount account = accounts.get(accounts.size() - 1);

            // 更新
            PaymentWithdrawAccount update = new PaymentWithdrawAccount();
            update.setId(account.getId());
            update.setAccount(param.getAccount());
            update.setRealName(param.getRealName());
            update.setModifyTime(now);
            paymentWithdrawAccountExtendMapper.updateByPrimaryKeySelective(update);
        } else {
            // 新增
            PaymentWithdrawAccount create = new PaymentWithdrawAccount();
            create.setId(sequenceCreater.nextLongId());
            create.setUserId(param.getUserId());
            create.setType(ALIPAY);
            create.setAccount(param.getAccount());
            create.setRealName(param.getRealName());
            create.setCreateTime(now);
            create.setModifyTime(now);
            paymentWithdrawAccountExtendMapper.insertSelective(create);
        }
        // 解锁
        lock.unlock(lockKey, lockId);
        return Message.build();
    }

    @Override
    public Optional<WithdrawAccountVO> getPaymentAccount(Byte type, Long userId) {
        List<PaymentWithdrawAccount> accounts = paymentWithdrawAccountExtendMapper.selectByUserIdAndType(userId, type);
        if (CollectionUtils.isEmpty(accounts)) {
            return Optional.empty();
        }

        // 如果有多条（极端情况下）获取最新的一个
        return Optional.of(convert(accounts.get(accounts.size() - 1)));
    }

    private WithdrawAccountVO convert(PaymentWithdrawAccount account) {
        WithdrawAccountVO accountVO = new WithdrawAccountVO();
        accountVO.setId(account.getId());
        accountVO.setUserId(account.getUserId());
        accountVO.setType(account.getType());
        accountVO.setAccount(account.getAccount());
        accountVO.setRealName(account.getRealName());
        return accountVO;
    }
}
