package com.bxm.localnews.payment.withdraw.filter;

import com.bxm.localnews.integration.UserAuthIntegrationService;
import com.bxm.localnews.payment.constant.WithdrawTypeEnum;
import com.bxm.localnews.payment.service.PaymentWithdrawAccountService;
import com.bxm.localnews.payment.vo.WithdrawAccountVO;
import com.bxm.localnews.payment.withdraw.WithdrawContext;
import com.bxm.localnews.vo.UserAuth;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Objects;
import java.util.Optional;

import static com.bxm.localnews.payment.constant.WithdrawAccountType.ALIPAY;
import static com.bxm.newidea.component.tools.StringUtils.isLessThan;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 判断用户是否绑定了提现账号
 * @author liujia
 * @date 2020/05/18 21:51
 */
@Component
@Slf4j
@AllArgsConstructor
public class BindAccountWithdrawFilter extends AbstractWithdrawFilter {

    private final UserAuthIntegrationService userAuthIntegrationService;

    private final PaymentWithdrawAccountService paymentWithdrawAccountService;

    @Override
    public Message run(WithdrawContext context) {
        if (isBlank(context.getWithdrawAccount() )) {
            // 临时代码 3.5.0版本上线之后就可以剔除了
            if (isBlank(context.getOriginParam().getCurVer()) || isLessThan(context.getOriginParam().getCurVer(), "3.5.0")) {
                log.info("用户: {}属于低版本，使用微信提现", context.getUserId());
                // 3.5.0先强制设置为支付宝提现
                context.setWithdrawType(WithdrawTypeEnum.WX_WITHDRAW);
                return setWechatWithdrawAccountInfo(context);
            }

            if (Objects.equals(context.getWithdrawType(), WithdrawTypeEnum.ALI_WITHDRAW)) {
                return setAliPayWithdrawAccountInfo(context);
            } else if (Objects.equals(context.getWithdrawType(), WithdrawTypeEnum.WX_WITHDRAW)) {
                return setWechatWithdrawAccountInfo(context);
            }
            return Message.build(false, "不支持的提现方式");
        }
        return Message.build();
    }

    /**
     * 设置微信提现的账户信息
     * @param context context
     * @return smg
     */
    private Message setWechatWithdrawAccountInfo(WithdrawContext context) {
        UserAuth userAuth = userAuthIntegrationService.selectUserAuthByUserId(context.getUserId());

        if (userAuth == null) {
            log.warn("用户[{}]未绑定openId, 不允许提现", context.getUserId());
            return Message.build(false, "请重新登陆后再尝试提现");
        }

        context.setWithdrawAccount(userAuth.getIdentifier());

        return Message.build();
    }

    /**
     * 设置支付包提现的账户信息
     * @param context context
     * @return msg
     */
    private Message setAliPayWithdrawAccountInfo(WithdrawContext context) {
        Optional<WithdrawAccountVO> paymentAccount = paymentWithdrawAccountService.getPaymentAccount(ALIPAY, context.getUserId());

        if (!paymentAccount.isPresent()) {
            log.warn("用户[{}]选择支付宝提现，但是未绑定支付宝, 不允许提现", context.getUserId());
            return Message.build(false, "请先绑定支付宝");
        }

        if (isBlank(paymentAccount.get().getAccount())) {
            log.warn("用户[{}]选择支付宝提现，但支付宝账号为空, 不允许提现", context.getUserId());
            return Message.build(false, "请先绑定支付宝");
        }

        if (isBlank(paymentAccount.get().getRealName())) {
            log.warn("用户[{}]选择支付宝提现，但真实姓名为空, 不允许提现", context.getUserId());
            return Message.build(false, "请填写支付宝账户对应的真实姓名");
        }

        context.setWithdrawAccount(paymentAccount.get().getAccount());
        context.setRealName(paymentAccount.get().getRealName());

        return Message.build();
    }

}
