package com.bxm.localnews.quartz.facade.controller;


import java.util.Calendar;
import java.util.Date;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.quartz.param.*;
import com.bxm.localnews.quartz.runtime.*;
import com.bxm.newidea.component.quartz.service.ScheduleJobService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

@Api(tags = "1-1", description = "定时任务构建相关服务接口")
@RestController
@RequestMapping("quartz")
@Slf4j
public class QuartzManageController {

    private static final Logger LOGGER = LoggerFactory.getLogger(QuartzManageController.class);

    private final ScheduleJobService scheduleJobService;

    @Autowired
    public QuartzManageController(ScheduleJobService scheduleJobService) {
        this.scheduleJobService = scheduleJobService;
    }

    @ApiOperation(value = "1-1-1 构建新闻监控的定时任务", notes = "根据配置的监控条件构建持续执行的定时任务")
    @PostMapping("/add/monitor")
    public ResponseEntity<String> createMonitorTask(@RequestBody MonitorTaskBuildParam param) {
        NewsMonitorTask task = new NewsMonitorTask();
        task.setJobName(task.getJobName() + param.getIndexId());

        String cronExpression = "0 0 " + param.getFireTime() + " * * ?";

        LOGGER.debug("监控表达式：[{}]", cronExpression);

        task.setCron(cronExpression);
        task.addParam("parameter", param);

        Message message = scheduleJobService.addCronJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-2 移除新闻监控的定时任务")
    @PostMapping("/remove/monitor")
    public ResponseEntity<String> removeMonitorTask(@RequestParam Long indexId) {
        NewsMonitorTask task = new NewsMonitorTask();
        Message message = scheduleJobService.remove(task.getJobName() + indexId, task.getGroup());

        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-3 添加新闻发布的定时任务（仅运行一次）", notes = "运营后台进行定时发布时调用")
    @PostMapping("/add/publish")
    public ResponseEntity<String> createPublishNewsTask(@RequestBody PublishNewsBuildParam param) {
        PublishNewsTask task = new PublishNewsTask();
        task.setJobName(task.getJobName() + "_" + param.getNewsId());
        task.addParam("parameter", param);
        task.setStartTime(param.getStartTime());

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            scheduleJobService.remove(task.getJobName(), task.getGroup());
        }
        LOGGER.warn(" 添加新闻发布的定时任务（仅运行一次）:{}", JSON.toJSONString(task));
        Message message = scheduleJobService.addSimpleJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-4 添加消息推送的定时任务(仅运行一次)", notes = "运营后台配置了定时push时，进行定时推送")
    @PostMapping("/add/push")
    public ResponseEntity<String> createPushMsgTask(@RequestBody PushMsgBuildParam param) {
        PushMessageTask task = new PushMessageTask();
        task.setJobName(task.getJobName() + "_" + param.getMessageId());
        task.addParam("parameter", param);
        task.setStartTime(param.getStartTime());

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            scheduleJobService.remove(task.getJobName(), task.getGroup());
        }

        Message message = scheduleJobService.addSimpleJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-5 移除推送的定时任务")
    @PostMapping("/remove/push")
    public ResponseEntity<String> removePushTask(@RequestParam Long messageId) {
        PushMessageTask task = new PushMessageTask();
        log.debug("删除消息[{}]的定时任务", messageId);
        Message message = scheduleJobService.remove(task.getJobName() + "_" + messageId, task.getGroup());

        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-6 添加解除邀请关系的定时任务(仅运行一次)")
    @PostMapping("/add/invite")
    public ResponseEntity<String> createInviteTask(@RequestBody InviteExpireTaskParam param) {
        InviteRelationshipTask task = new InviteRelationshipTask();
        task.setJobName(task.getJobName() + "_" + param.getJobId());
        task.addParam("parameter", param);
        task.setStartTime(param.getStartTime());

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            return ResponseEntity.ok().build();
        }

        Message message = scheduleJobService.addSimpleJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-7 移除邀请的定时任务")
    @PostMapping("/remove/invite")
    public ResponseEntity<String> removeInviteTask(String jobId) {
        InviteRelationshipTask task = new InviteRelationshipTask();
        Message message = scheduleJobService.remove(task.getJobName() + "_" + jobId, task.getGroup());

        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-8 添加定时清理用户帖子置顶开始执行")
    @PostMapping("/remove/user/post")
    public ResponseEntity<String> removePostUserTopTask(@RequestBody UserTopPostBuildParam userTopPostBuildParam) {
        UserTopPostTask task = new UserTopPostTask();
        task.setJobName(task.getJobName() + "_" + userTopPostBuildParam.getPostId());
        task.addParam("parameter", userTopPostBuildParam);
        //一小时后开始执行清理
        task.setStartTime(DateUtils.addField(userTopPostBuildParam.getAddTime(), Calendar.HOUR, +1));

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            return ResponseEntity.ok().build();
        }
        LOGGER.warn(" 添加定时清理用户帖子置顶的定时任务（仅运行一次）:{}", JSON.toJSONString(task));
        Message message = scheduleJobService.addSimpleJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-9 添加定时增加帖子点击数的定时任务(仅运行一天)")
    @PostMapping("/add/postClick")
    public ResponseEntity<String> createPostClickTask(@RequestBody AddPostClickCountParam param) {
        AddPostClickCountTask task = new AddPostClickCountTask();
        Calendar now = Calendar.getInstance();

        task.setJobName(task.getJobName() + "_" + param.getPostId());
        param.setEndTime(DateUtils.addField(now.getTime(), Calendar.DAY_OF_MONTH, 1));
        task.addParam("parameter", param);

        String cronExpression = now.get(Calendar.SECOND) + " " + now.get(Calendar.MINUTE) + "/6 * * * ?";
        LOGGER.debug("任务表达式：[{}]", cronExpression);
        task.setCron(cronExpression);

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            return ResponseEntity.ok().build();
        }

        Message message = scheduleJobService.addCronJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-10 添加发送优化通知的定时任务(仅运行一次)")
    @PostMapping("/add/optimizationNotice")
    public ResponseEntity<String > createInviteTask(@RequestBody OptimizationNoticeParam param) {
        OptimizationNoticeTask task = new OptimizationNoticeTask();
        task.setJobName(task.getJobName() + "_" + param.getId());
        task.setStartTime(param.getStartTime());
        task.addParam("parameter", param);

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            scheduleJobService.remove(task.getJobName(), task.getGroup());
        }
        LOGGER.info(" 添加发送优化通知的定时任务:{}", JSON.toJSONString(task));

        Message message = scheduleJobService.addSimpleJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-11 添加蓝天救援队活动提醒的定时任务(仅运行三天)")
    @PostMapping("/add/helpRemind")
    public ResponseEntity<String> createHelpRemindTask(@RequestBody HelpRemindParam param) {
        HelpRemindTask task = new HelpRemindTask();
        task.addParam("parameter", param);

        Date endTime = DateUtils.parse(param.getEndTime());
        //结束时间小于当前时间，移除当前任务
        if (endTime.before(new Date())) {
            return removeHelpRemindTask();
        }

        Calendar startTime = Calendar.getInstance();
        startTime.setTime(endTime);
        startTime.add(Calendar.DAY_OF_MONTH, -3);

        //任务触发时间小于当前时间，则修改触发时间
        if (startTime.before(new Date())) {
            startTime.add(Calendar.DAY_OF_MONTH, DateUtils.getDiffDays(startTime.getTime(), new Date(), true) + 1);
        }

        String cronExpression = startTime.get(Calendar.SECOND) + " " + startTime.get(Calendar.MINUTE)
                + " " + startTime.get(Calendar.HOUR_OF_DAY) + " " + startTime.get(Calendar.DAY_OF_MONTH) + "/1 * ?";
        LOGGER.debug("任务表达式：[{}]", cronExpression);
        task.setCron(cronExpression);

        if (scheduleJobService.getJob(task.getJobName(), task.getGroup()) != null) {
            scheduleJobService.remove(task.getJobName(), task.getGroup());
        }

        Message message = scheduleJobService.addCronJob(task);
        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }

    @ApiOperation(value = "1-1-12 移除蓝天救援队活动提醒的定时任务")
    @PostMapping("/remove/helpRemind")
    public ResponseEntity<String> removeHelpRemindTask() {
        HelpRemindTask task = new HelpRemindTask();
        Message message = scheduleJobService.remove(task.getJobName(), task.getGroup());

        if (message.isSuccess()) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.badRequest().body(message.getLashMessaage());
        }
    }


}
