package com.bxm.localnews.quartz.runtime;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.quartz.config.MessageUrlProperties;
import com.bxm.localnews.quartz.constant.RedisConfig;
import com.bxm.localnews.quartz.constant.TaskGroup;
import com.bxm.localnews.quartz.domain.HelpInfoMapper;
import com.bxm.localnews.quartz.facade.PushMsgIntegService;
import com.bxm.localnews.quartz.integration.NewsIntegrationService;
import com.bxm.localnews.quartz.param.AddPostClickCountParam;
import com.bxm.localnews.quartz.param.HelpRemindParam;
import com.bxm.localnews.quartz.vo.HelpUserInfo;
import com.bxm.newidea.component.quartz.service.ScheduleJobService;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.Message;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;

@Component
public class HelpRemindTask extends AbstractRuntimeJob {

    @Resource
    private HelpInfoMapper helpInfoMapper;

    @Resource
    private ScheduleJobService scheduleJobService;

    @Resource
    private PushMsgIntegService pushMsgIntegService;

    @Resource
    private MessageUrlProperties messageUrlProperties;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    public HelpRemindTask() {
        super("helpRemindTask", TaskGroup.ONCE);
        super.setCount(0);
    }

    @Override
    protected Message run() {
        logger.debug("蓝天救援队定时提醒任务开始执行:", JSON.toJSONString(this));
        Map<String, Object> paramMap = this.getParam();

        if (null != paramMap) {
            HelpRemindParam param = (HelpRemindParam) this.getParam().get("parameter");
            logger.info("蓝天救援队定时提醒任务参数：{}", JSONObject.toJSONString(param));
            if (null == param || null == param.getEndTime()) {
                return Message.build(false).setMessage("参数为空");
            }

            //获取推送用户
            List<Long> userIdList = helpInfoMapper.getHelpUserIdList();

            Date endTime = DateUtils.parse(param.getEndTime());
            if (new Date().after(endTime)) {
                //活动结束提醒
                String title = "感谢你为蓝天救援队奉献了一份爱心，目前我们正在传递你的爱心，点击查看";
                PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.OPEN_URL);
                info.addExtend("url", messageUrlProperties.getPostDetail());

                PushMessage message = PushMessage.build();
                message.setTitle("通知消息");
                message.setContent(title);
                message.setType(TemplateTypeEnum.NOTIFCTION);
                message.setPushReceiveScope(PushReceiveScope.pushGroup(userIdList));
                message.setPayloadInfo(info);

                this.pushMsgIntegService.pushMsg(message);
                scheduleJobService.remove(this.getJobName(), this.getGroup());
                return Message.build(true);
            }

            if (redisStringAdapter.hasKey(RedisConfig.BLUE_SKY_HELP_COMPLETE.copy())) {
                return Message.build(true);
            }

            //活动前三天提醒
            Integer dayNum = DateUtils.getDiffDays(new Date(), endTime, true) + 1;
            String title = "距离活动结束还有" + dayNum + "天，目前爱心助力目标还未达成，请邀请好友再助力一次吧！";
            PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.OPEN_URL);

            PushMessage message = PushMessage.build();
            message.setTitle("通知消息");
            message.setContent(title);
            message.setType(TemplateTypeEnum.NOTIFCTION);

            if (CollectionUtils.isNotEmpty(userIdList)) {
                for (Long userId : userIdList) {
                    HelpUserInfo helpUserInfo = helpInfoMapper.getHelpUserInfo(userId);
                    info.addExtend("url", messageUrlProperties.getBlueSkySharePage() + "?userId=" + userId
                            + "&areaCode=" + helpUserInfo.getLocationCode() + "&areaName=" + helpUserInfo.getLocationName());
                    message.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
                    message.setPayloadInfo(info);

                    this.pushMsgIntegService.pushMsg(message);
                }
            }

            return Message.build(true);
        }

        return Message.build(false).setMessage("蓝天救援队定时提醒任务调度失败，缺少参数注入");
    }

}
