package com.bxm.newidea.recommend;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.dto.VideoDto;
import com.bxm.newidea.enums.NewsKindEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

@Component
public class RecommendEngine {

    private Logger logger = LoggerFactory.getLogger(RecommendEngine.class);

    private RecommendFilter recommendFilter;

    private RecommendManager recommendManager;

    @Autowired
    public RecommendEngine(RecommendManager recommendManager, RecommendFilter recommendFilter) {
        this.recommendManager = recommendManager;
        this.recommendFilter = recommendFilter;
    }

    public List<Long> recommendNews(Long userId, Integer kindId, Integer num, String areaCode,Integer curPage) {
        // 如果是本地标签则只执行本地新闻推荐
         if (kindId != null && kindId.intValue() == NewsKindEnum.LOCAL.getCode().intValue()) {
            return this.recommendLocalNews(userId, kindId, num, areaCode);
        } else {
            // 首页推荐频道将kindId置空，兼容以前逻辑
            if (kindId == null || (kindId.intValue() == NewsKindEnum.RECOMMEND.getCode().intValue())) {
                kindId = null;
                //首页推荐
                return this.recommendCommonNews(userId, kindId, num, areaCode,curPage);
            }
            //各频道的推荐
            return this.recommendAllNews(userId, kindId, num, areaCode, (byte) 0);
        }
    }

    private List<Long> recommendAllNews(Long userId, Integer kindId, Integer num, String areaCode,Byte type) {
        List<AbstractRecommender> recommenderWorkers;
        // 如果是推荐频道的推荐
        if (type==1) {
            recommenderWorkers = recommendManager.getCommonNewsRecommenders();
        }else{
            recommenderWorkers = recommendManager.getNewsRecommenders();
        }
        Iterator<AbstractRecommender> iterator = recommenderWorkers.iterator();
        List<Long> newsIds = new LinkedList<>();
        // fetchNum：权重*当前总新闻数 当某一个策略不足时候 顺序补足
        int fetchNum;
        int recommendNum = 0;


        while (iterator.hasNext()) {
            AbstractRecommender recommender = iterator.next();

            //获取推荐器权重对应的推荐数量
            fetchNum = new BigDecimal(num).multiply(new BigDecimal("" + recommender.getWeight())).intValue();
            //如果权重推荐数量大于剩余推荐数量，则选择剩余数量
            if (num - recommendNum < fetchNum) {
                fetchNum = num - recommendNum;
            }
            logger.debug("{} 预计推荐数量：{} 用户：{}", recommender.getClass(), fetchNum,userId);

            List<Long> results = recommender.recommendNews(userId, kindId, fetchNum, areaCode);

            logger.debug("{} 实际推荐数量：{} 用户：{}", recommender.getClass(), results.size(),userId);

            recommendNum += results.size();
            newsIds.addAll(results);

            if (newsIds.size() >= num) {
                break;
            }

        }
        // 全局黑名单过滤
        newsIds = recommendFilter.newsFilter(newsIds);
        // 去重
        removeDuplicateNews(newsIds);

        return newsIds.size() > num ? newsIds.subList(0, num) : newsIds;
    }

    /**
     * 填充首页推荐结果
     * @param userId
     * @param kindId
     * @param num
     * @param areaCode
     * @param curPage
     * @return
     */
    private List<Long> recommendCommonNews(Long userId, Integer kindId, Integer num, String areaCode,Integer curPage){

        //取得本地新闻
        List<Long> localNewsIds = getRecLocalnews(userId, kindId, 3, areaCode);

        logger.info("推荐引擎->本地:{},用户:{}", JSON.toJSONString(localNewsIds),userId);

        List<Long> commonNewsIds = new LinkedList<>();

        List<Long> hotNewsIdList = new ArrayList<>();
        List<Integer> kindList ;
//        如果是第一页
        if (curPage == 1) {
            //取得热门新闻
            hotNewsIdList = this.recommendHotNews(userId,kindId,1,areaCode);
            logger.info("推荐引擎->第一页推荐，热门:{},用户:{}",CollectionUtils.isEmpty(hotNewsIdList)?"无":hotNewsIdList.get(0),userId);
            kindList = generateHomeKind();
        }else{
            kindList = generateKind();
        }


        //取得各频道的新闻
        for (Integer kind : kindList) {
            commonNewsIds.addAll(recommendAllNews(userId,kind,1,areaCode,(byte) 1));
        }

        //把本地新闻穿插进去
        int count = localNewsIds.size();
        int index =0;
        for (int i=0;i<count;i++) {
            if (index > commonNewsIds.size() - 1) {
                ((LinkedList<Long>) commonNewsIds).addLast(localNewsIds.get(i));
            } else {
                commonNewsIds.add(index,localNewsIds.get(i));
                index = index + 2;
            }
        }


        if (!CollectionUtils.isEmpty(hotNewsIdList)) {
            ((LinkedList<Long>) commonNewsIds).addFirst(hotNewsIdList.get(0));
        }

        //有少的就补充(这里增加五条是防止有重复的)
        int fetchNum = num - commonNewsIds.size()+5;
        commonNewsIds.addAll(recommendAllNews(userId,null,fetchNum,areaCode,(byte) 1));

        return commonNewsIds.stream().distinct().limit(num).collect(Collectors.toList());
    }

    /**
     * 当本地新闻不足时，填充全国新闻
     * @param userId
     * @param kindId
     * @param num
     * @param areaCode
     * @return
     */
    private List<Long> getRecLocalnews(Long userId, Integer kindId, Integer num, String areaCode) {
        List<Long> localNewsIds = recommendLocalNews(userId, kindId, num, areaCode);
        if (localNewsIds.size() < num) {
            int leftNum = num - localNewsIds.size();
            List<Integer> kindIds = generateNationalKind();
            for (Integer kind : kindIds) {
                localNewsIds.addAll(recommendAllNews(userId, kind, leftNum, areaCode,(byte) 1));
                if (localNewsIds.size() >= num) {
                    break;
                } else {
                    leftNum = num - localNewsIds.size();
                }
            }
        }
        return localNewsIds;
    }

    /**
     * 得到第二页的新闻频道(可以放在配置文件中)
     * @return
     */
    private List<Integer> generateKind(){
        List<Integer> kindList = new ArrayList<>();
        kindList.add(32);
        kindList.add(20);
        kindList.add(18);
        kindList.add(14);
        kindList.add(69);
        kindList.add(27);
        return kindList;
    }

    /**
     * 得到第一页的新闻频道(可以放在配置文件中)
     * @return
     */
    private List<Integer> generateHomeKind(){
        List<Integer> kindList = new ArrayList<>();
        kindList.add(69);
        kindList.add(18);
        kindList.add(69);
        kindList.add(27);
        return kindList;
    }

    /**
     * 得到本地新闻不存在时的全国新闻频道(可以放在配置文件中)
     * @return
     */
    private List<Integer> generateNationalKind(){
        List<Integer> kindList = new ArrayList<>();
        kindList.add(18);
        kindList.add(14);
        kindList.add(69);
        kindList.add(27);
        kindList.add(22);
        return kindList;
    }

    /**
     * @param userId
     * @param kindId
     * @param num
     * @param areaCode
     * @return
     * @Description 方法描述：推荐本地新闻
     * @author leon 2019年1月23日 下午3:06:53
     * @CopyRight 杭州微财网络科技有限公司
     */
    private List<Long> recommendLocalNews(Long userId, Integer kindId, Integer num, String areaCode) {
        AbstractRecommender recommender = recommendManager.getLocalNewsRecommender();
        logger.debug("推荐本地新闻参数: userId={}，kindId={}，num={}，areaCode={}", userId, kindId, num, areaCode);
        List<Long> results = recommender.recommendNews(userId, kindId, num, areaCode);
        logger.debug("推荐本地新闻:{} 推荐结果：{} 用户：{}", recommender.getClass(), results.size(),userId);
        // 全局黑名单过滤
        results = recommendFilter.newsFilter(results);
        return results;
    }

    /**
     * 热门新闻推荐
     * @param userId
     * @param kindId
     * @param num
     * @param areaCode
     * @return
     */
    private List<Long> recommendHotNews(Long userId, Integer kindId, Integer num, String areaCode) {
        AbstractRecommender recommender = recommendManager.getHotNewsRecommender();
        logger.debug("推荐热门新闻参数: userId={}，kindId={}，num={}，areaCode={}", userId, kindId, num, areaCode);
        List<Long> results = recommender.recommendNews(userId, kindId, num, areaCode);
        logger.debug("推荐热门新闻:{} 推荐结果：{} 用户：{}", recommender.getClass(), results.size(),userId);
        // 全局黑名单过滤
        results = recommendFilter.newsFilter(results);
        return results;
    }

    public List<VideoDto> recommendVideo(Long userId, Integer num) {
        List<AbstractRecommender> recommenderWorkers = recommendManager.getVideoRecommenders();
        Iterator<AbstractRecommender> iterator = recommenderWorkers.iterator();
        logger.debug("视频新闻推荐引擎调用链：{}", recommenderWorkers);
        List<VideoDto> videoDtoList = new LinkedList<>();
        // fetchNum：权重*当前总新闻数 当某一个策略不足时候 顺序不足
        int fetchNum;
        int differenceNum = 0;
        while (iterator.hasNext()) {
            AbstractRecommender recommender = iterator.next();
            fetchNum = new BigDecimal(num).multiply(new BigDecimal(recommender.getWeight())).intValue() + differenceNum;
            List<VideoDto> results = recommender.recommendVideo(userId, fetchNum);
            logger.debug("{} 推荐结果：{}", recommender.getClass(), results.size());
            differenceNum = fetchNum - results.size();
            videoDtoList.addAll(results);
            if (videoDtoList.size() == num) {
                break;
            }
        }
        // 全局黑名单过滤
        videoDtoList = recommendFilter.videoFilter(videoDtoList);
        // 去重
        videoDtoList = removeDuplicateVideo(videoDtoList);
        return videoDtoList.size() > num ? videoDtoList.subList(0, num) : videoDtoList;
    }

    private void removeDuplicateNews(List<Long> list) {
        for (int i = 0; i < list.size() - 1; i++) {
            for (int j = list.size() - 1; j > i; j--) {
                if (list.get(j).equals(list.get(i))) {
                    list.remove(j);
                }
            }
        }
    }

    private List<VideoDto> removeDuplicateVideo(List<VideoDto> list) {
        HashSet<VideoDto> h = new HashSet<>(list);
        list.clear();
        list.addAll(h);
        return list;
    }
}
