package com.bxm.newidea.recommend.handler.news;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.config.NewsCacheThresholdConfig;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.NewsRecommendedMapper;
import com.bxm.newidea.domain.NewsRecordMapper;
import com.bxm.newidea.recommend.AbstractNewsRecommender;
import com.bxm.newidea.vo.News;
import com.bxm.newidea.vo.NewsRecord;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnels;
import org.apache.mahout.cf.taste.common.TasteException;
import org.apache.mahout.cf.taste.impl.common.FastByIDMap;
import org.apache.mahout.cf.taste.impl.common.FastIDSet;
import org.apache.mahout.cf.taste.impl.model.GenericBooleanPrefDataModel;
import org.apache.mahout.cf.taste.impl.neighborhood.NearestNUserNeighborhood;
import org.apache.mahout.cf.taste.impl.recommender.GenericUserBasedRecommender;
import org.apache.mahout.cf.taste.impl.similarity.LogLikelihoodSimilarity;
import org.apache.mahout.cf.taste.model.DataModel;
import org.apache.mahout.cf.taste.neighborhood.UserNeighborhood;
import org.apache.mahout.cf.taste.recommender.RecommendedItem;
import org.apache.mahout.cf.taste.recommender.Recommender;
import org.apache.mahout.cf.taste.similarity.UserSimilarity;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

/**
 * 单机版的协同过滤推荐
 * 暂停使用
 */
@Component
public class CollaborativeNewsRecommender extends AbstractNewsRecommender {
    private static SoftReference<FastByIDMap<FastIDSet>> cache;

    private NewsRecordMapper newsRecordMapper;

    @Autowired(required = false)
    public CollaborativeNewsRecommender(NewsRecordMapper newsRecordMapper, RedisListAdapter redisListAdapter, NewsRecommendedMapper newsRecommendedMapper) {
        super(0.5, 3);
        this.newsRecordMapper = newsRecordMapper;
        this.redisListAdapter = redisListAdapter;
        this.newsRecommendedMapper = newsRecommendedMapper;
    }

    @Override
    protected List<Long> syncRecommendNews(Long userId, Integer kindId, Integer num, String areaCode) {
        long begin = System.currentTimeMillis();
        if (kindId != null) {
            //具体的类目推荐时 不走协同 1：目前样本支撑不了 2:阅读记录里没有记录kindid需要很后面的时候 在考虑
            return new ArrayList<>();
        }
        KeyGenerator userCacheKey = RedisKeyConstant.NEWS_COLLABORATIVE_RECOMMEND.copy().appendKey(userId);
        long size = redisListAdapter.size(userCacheKey);
        if (size == 0) {
            //缓存池无内容则直接进行返回
            return new ArrayList<>();
        }
        //已经推荐过的新闻key
        KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_RECOMMENDED.copy();
        recommendedNewsKey.appendKey(userId);
        long recommendedSize = redisListAdapter.size(recommendedNewsKey);
        List<Long> recommendedNewIds = redisListAdapter.leftIndex(recommendedNewsKey, recommendedSize - 1, Long.class);
        //布隆过滤器去重 防止随着已经推荐过的记录膨胀 去重效率降低
        BloomFilter<Long> bloomFilter = null;
        if (recommendedNewIds != null && !recommendedNewIds.isEmpty()) {
            bloomFilter = BloomFilter.create(Funnels.longFunnel(), recommendedNewIds.size(), 0.0001);
            for (Long id : recommendedNewIds) {
                bloomFilter.put(id);
            }
        }
        List<Long> resultList = new ArrayList<>();
        recommendedFilter(userCacheKey, num, bloomFilter, resultList,userId);
        logger.debug("同步推荐耗时：{}", System.currentTimeMillis() - begin);
        return resultList;
    }


//    @Async
    @Override
    protected void asyncRecommendNews(Long userId, Integer kindId, String areaCode) {
        logger.debug("协同推荐补充池");
        long begin = System.currentTimeMillis();
        try {
            if (kindId != null) {
                //具体的类目推荐时 不走协同 1：目前样本支撑不了 2:阅读记录里没有记录kindid需要很后面的时候 在考虑
                return;
            }
            KeyGenerator userCacheKey = RedisKeyConstant.NEWS_COLLABORATIVE_RECOMMEND.copy().appendKey(userId);
            int size = redisListAdapter.size(userCacheKey).intValue();
            if (size > NewsCacheThresholdConfig.ALARM_CACHE_SIZE) {
                //缓存池内容尚无到达触发异步推荐的阀值 则暂不进行异步推荐
                return;
            }
            FastByIDMap<FastIDSet> dataMap = null;
            if (cache == null || cache.get() == null) {
                dataMap = getCache();
            }else{
                dataMap = cache.get();
            }
            if (dataMap == null) {
                return;
            }
            DataModel dataModel = new GenericBooleanPrefDataModel(dataMap);
            UserSimilarity similarity = new LogLikelihoodSimilarity(dataModel);
            //使用两个邻域用户 作为协同参考
            UserNeighborhood neighborhood = new NearestNUserNeighborhood(2, similarity, dataModel);
            Recommender recommender = new GenericUserBasedRecommender(dataModel, neighborhood, similarity);
            List<RecommendedItem> list = recommender.recommend(userId, NewsCacheThresholdConfig.MAX_CACHE_SIZE);
            List<News> ids = new ArrayList<>();
            list.forEach(recommendedItem -> {
                News news = new News();
                news.setId(recommendedItem.getItemID());
                ids.add(news);
            });
            News[] newsDtos = ids.toArray(new News[0]);
            redisListAdapter.remove(userCacheKey);
            redisListAdapter.rightPush(userCacheKey, newsDtos);
            //保存1小时，考虑到很少用户会看1个小时以上的新闻，所以没必要存很长时间
            redisListAdapter.expire(userCacheKey, 3600);

            logger.debug("给{}进行协同过滤新闻推荐 结果结果：{}。花费时间{}", userId, ids, System.currentTimeMillis() - begin);
        } catch (TasteException e) {
            logger.error("协同过滤推荐异常：{}", e);
        }
    }

    private FastByIDMap<FastIDSet> getCache() {
        Calendar calendar = Calendar.getInstance();
        //协同参考7天内数据 7天前不做参考
        calendar.add(Calendar.DAY_OF_YEAR, -7);
        List<Long> userIdList = newsRecordMapper.selectAllUser(calendar.getTime());
        if (userIdList.isEmpty()) {
            logger.error("协同过滤推荐用户样本为空，中止此次推荐");
            return null;
        }
        List<NewsRecord> newsRecordList = newsRecordMapper.selectBooleanPrefList(calendar.getTime());
        if (newsRecordList.isEmpty()) {
            logger.error("协同过滤推荐数据样本为空，中止此次推荐");
            return null;
        }
        FastByIDMap<FastIDSet> dataMap = new FastByIDMap<>();
        FastIDSet itemIDSet;
        for (Long uid : userIdList) {
            itemIDSet = new FastIDSet();
            for (NewsRecord record : newsRecordList) {
                if (record.getUserId().longValue() == uid.longValue()) {
                    itemIDSet.add(record.getNewsId());
                }
            }
            if (!itemIDSet.isEmpty()) {
                dataMap.put(uid, itemIDSet);
            }
        }
        cache = new SoftReference<>(dataMap);
        return dataMap;
    }
}
