package com.bxm.newidea.recommend.handler.news;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.config.NewsCacheThresholdConfig;
import com.bxm.newidea.constant.RedisKeyConstant;
import com.bxm.newidea.domain.NewsMapper;
import com.bxm.newidea.domain.NewsRecommendedMapper;
import com.bxm.newidea.integration.NewsSyncIntegrationService;
import com.bxm.newidea.param.NewsRecommendParam;
import com.bxm.newidea.recommend.AbstractNewsRecommender;
import com.bxm.newidea.service.WeightService;
import com.bxm.newidea.vo.News;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnels;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

@Component
public class OldMysqlRecNewsRecommender extends AbstractNewsRecommender {

    private NewsMapper newsMapper;

    private WeightService weightService;

    @Autowired(required = false)
    public OldMysqlRecNewsRecommender(NewsRecommendedMapper newsRecommendedMapper,
                                   RedisListAdapter redisListAdapter,
                                   NewsMapper newsMapper,WeightService weightService,
                                      NewsSyncIntegrationService newsSyncIntegrationService) {
        super(1, 1);
        this.newsRecommendedMapper = newsRecommendedMapper;
        this.redisListAdapter = redisListAdapter;
        this.newsMapper = newsMapper;
        this.weightService = weightService;
        this.newsSyncIntegrationService = newsSyncIntegrationService;
    }


    @Override
    protected List<Long> syncRecommendNews(Long userId, Integer kindId, Integer num, String areaCode) {
        long begin = System.currentTimeMillis();
        KeyGenerator userCacheKey = RedisKeyConstant.NEWS_COMMON_RECOMMEND.copy();
        if (kindId != null) {
            userCacheKey.setKey(String.valueOf(kindId));
        }

        userCacheKey.appendKey(userId);
        long size = redisListAdapter.size(userCacheKey);
        if (size == 0) {
            //缓存池无内容则直接进行返回
            return new ArrayList<>();
        }
        //已经推荐过的新闻key
        KeyGenerator recommendedNewsKey = RedisKeyConstant.NEWS_RECOMMENDED.copy();
        recommendedNewsKey.appendKey(userId);
        long recommendedSize = redisListAdapter.size(recommendedNewsKey);
        List<Long> recommendedNewIds = redisListAdapter.leftIndex(recommendedNewsKey, recommendedSize - 1, Long.class);
        //布隆过滤器去重 防止随着已经推荐过的记录膨胀 去重效率降低
        BloomFilter<Long> bloomFilter = null;
        if (recommendedNewIds != null && !recommendedNewIds.isEmpty()) {
            bloomFilter = BloomFilter.create(Funnels.longFunnel(), recommendedNewIds.size(), 0.0001);
            for (Long id : recommendedNewIds) {
                bloomFilter.put(id);
            }
        }
        List<Long> resultList = new ArrayList<>();
        recommendedFilter(userCacheKey, num, bloomFilter, resultList,userId);
        logger.debug("同步推荐耗时：{}", System.currentTimeMillis() - begin);
        return resultList;
    }


    @Override
    @Async
    protected void asyncRecommendNews(Long userId, Integer kindId, String areaCode) {
        logger.info("异步推荐，用户：{}，频道：{}，地区：{}",userId,kindId,areaCode);
        long current = System.currentTimeMillis();
        NewsRecommendParam newsRecommendParam = new NewsRecommendParam();
        KeyGenerator userCacheKey = RedisKeyConstant.NEWS_COMMON_RECOMMEND.copy();
        if (kindId != null) {
            userCacheKey.setKey(String.valueOf(kindId));
            newsRecommendParam.setKindId((long) kindId);
        }
        userCacheKey.appendKey(userId);
        int size = redisListAdapter.size(userCacheKey).intValue();

        if (kindId != null) {
            if (size > NewsCacheThresholdConfig.ALARM_RECOMMEND_CACHE_SIZE) {
                //缓存池内容尚无到达触发异步推荐的阀值 则暂不进行异步推荐
                return;
            }
        }else{
            if (size > NewsCacheThresholdConfig.ALARM_RECOMMEND_OTHER_CACHE_SIZE) {
                //其它类目缓存池内容尚无到达触发异步推荐的阀值 则暂不进行异步推荐
                return;
            }
        }

        //重置用户标签
        long currentRecalcu = System.currentTimeMillis();
        this.weightService.recalcu(userId);
        logger.debug("给{}进行用户标签重置 花费时间{}", userId, System.currentTimeMillis() - currentRecalcu);

        newsRecommendParam.setIsHot(1);
        newsRecommendParam.setUserId(userId);

        //补充满推荐缓存池
        List<News> result ;
        if (kindId != null) {
            newsRecommendParam.setPagesize(NewsCacheThresholdConfig.MAX_RECOMMEND_CACHE_SIZE);
            result = newsMapper.recommendNews(newsRecommendParam);
            if (result.size()<NewsCacheThresholdConfig.ALARM_RECOMMEND_CACHE_SIZE) {
                result = newsMapper.recommendNewsWithoutTime(newsRecommendParam);
                logger.info("由于新闻时效性导致推荐过少,所以重新进行一次推荐,参数：{}，数量：{}", JSON.toJSONString(newsRecommendParam),result.size());
            }
        }else{
            newsRecommendParam.setPagesize(NewsCacheThresholdConfig.MAX_RECOMMEND_OTHER_CACHE_SIZE);
            result = newsMapper.recommendOtherNews(newsRecommendParam);
            if (result.size()<NewsCacheThresholdConfig.ALARM_RECOMMEND_OTHER_CACHE_SIZE) {
                result = newsMapper.recommendOtherNewsWithoutTime(newsRecommendParam);
                logger.info("由于新闻时效性导致推荐过少,所以重新进行一次推荐,参数：{}，数量：{}",JSON.toJSONString(newsRecommendParam),result.size());
            }
        }

        News[] newsDtos = result.toArray(new News[0]);
        redisListAdapter.remove(userCacheKey);
        redisListAdapter.rightPush(userCacheKey, newsDtos);
        //保存1小时，考虑到很少用户会看1个小时以上的新闻，所以没必要存很长时间
        redisListAdapter.expire(userCacheKey, 3600);
        logger.info("给{}进行频道为{}的新闻推荐 结果结果：{}。花费时间{}", userId, kindId == null ? "推荐" : kindId, newsDtos.length, System.currentTimeMillis() - current);

    }



}
