package com.bxm.localnews.thirdparty.controller;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.thirdparty.config.WechatMPConfig;
import com.bxm.localnews.thirdparty.constant.WechatMpTypeEnum;
import com.bxm.localnews.thirdparty.param.RedirectToWechatMpAuthParam;
import com.bxm.localnews.thirdparty.service.impl.WechatMpService;
import com.bxm.newidea.component.annotations.ApiVersion;
import com.bxm.newidea.component.vo.ResponseJson;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.bean.WxJsapiSignature;
import me.chanjar.weixin.mp.api.WxMpConfigStorage;
import me.chanjar.weixin.mp.api.WxMpMessageRouter;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.bean.message.WxMpXmlMessage;
import me.chanjar.weixin.mp.bean.message.WxMpXmlOutMessage;
import me.chanjar.weixin.mp.util.WxMpConfigStorageHolder;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Map;

import static com.gexin.fastjson.JSON.toJSONString;

/**
 * 微信公众号相关功能接口
 *
 * @author gonzo
 * @date 2020-07-30 17:29
 **/
@Slf4j
@Api(tags = "3-23 微信公众号相关功能接口", description = "如：微信授权地址获取")
@RestController
@RequestMapping("{version}/thirdparty/wechatMp")
@Controller
@AllArgsConstructor(onConstructor = @__(@Autowired))
public class WechatMpController {

    private final WxMpService wxMpService;

    private final WechatMpService wechatMpService;

    private final WxMpMessageRouter wxMpMessageRouter;

    private final WechatMPConfig wechatMPConfig;

    @ApiOperation(value = "3-23-1 [v1]组装并重定向到微信授权页面", notes = "根据请求参数，选用中间页域名，组装并重定向到微信授权页面")
    @GetMapping("auth")
    @ApiVersion(1)
    public ResponseJson<String> redirectToWechatMpAuth(RedirectToWechatMpAuthParam param, HttpServletResponse response) {
        if (param.isDoRedirect()) {
            try {
                response.sendRedirect(wechatMpService.wechatMpAuthUrlGen(param));
            } catch (Exception e) {
                log.error("重定向失败, param: {}", toJSONString(param), e);
            }
            return ResponseJson.ok().build();
        } else {
            return ResponseJson.ok(wechatMpService.wechatMpAuthUrlGen(param));
        }
    }

    @ApiOperation(value = "3-23-1 [v1]获取js签名验证，并将url转成短链用作二次分享的打开链接")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "url", value = "签名认证的url", dataType = "String", required = true),
    })
    @GetMapping("public/jsSignature")
    @ApiVersion(1)
    public Json<WxJsapiSignature> getWxJsApiSignature(@RequestParam("url") String url) {
        return ResultUtil.genSuccessResult(wechatMpService.createJsApiSignature(url));
    }


    @ApiOperation(value = "3-23-2 [v1]公众号推送")
    @ApiVersion(1)
    @RequestMapping("/public/push")
    public void eventHandler(HttpServletRequest request, HttpServletResponse response) throws IOException {
        // 获取配置用来接受微信回调的appId
        Map<String, String> appIdByType = wechatMPConfig.getAppIdByType();
        String appId = appIdByType.get(WechatMpTypeEnum.NOTIFY.name());
        log.info("获取微信推送信息: {}, 接收微信通知的AppId: {}", JSON.toJSONString(request.getParameterMap()), appId);

        String signature = request.getParameter("signature");
        String nonce = request.getParameter("nonce");
        String timestamp = request.getParameter("timestamp");

        response.setContentType("text/html;charset=utf-8");
        response.setStatus(HttpServletResponse.SC_OK);

        // 用之前先设置
        WxMpConfigStorageHolder.set(appId);
        if (!wxMpService.checkSignature(timestamp, nonce, signature)) {
            // 消息签名不正确，说明不是公众平台发过来的消息
            log.error("消息签名不正确");
            response.getWriter().println("非法请求");
            return;
        }

        String echostr = request.getParameter("echostr");
        if (StringUtils.isNotBlank(echostr)) {
            // 说明是一个仅仅用来验证的请求，回显echostr
            log.error("验证的请求,回调的信息为:{}", echostr);
            response.getWriter().println(echostr);
            return;
        }

        String encryptType = StringUtils.isBlank(request.getParameter("encrypt_type")) ?
                "raw" :
                request.getParameter("encrypt_type");


        WxMpXmlMessage inMessage;
        // 用之前先设置
        WxMpConfigStorageHolder.set(appId);
        WxMpConfigStorage wxMpConfigStorage = wxMpService.getWxMpConfigStorage();

        if ("raw".equals(encryptType)) {
            // 明文传输的消息
            inMessage = WxMpXmlMessage.fromXml(request.getInputStream());
        } else if ("aes".equals(encryptType)) {
            // 是aes加密的消息
            String msgSignature = request.getParameter("msg_signature");
            inMessage = WxMpXmlMessage.fromEncryptedXml(request.getInputStream(), wxMpConfigStorage, timestamp, nonce, msgSignature);
        } else {
            log.error("不可识别的加密类型");
            response.getWriter().println("不可识别的加密类型");
            return;
        }

        //业务处理
        WxMpXmlOutMessage outMessage = wxMpMessageRouter.route(inMessage);

        if (outMessage != null) {
            log.info("业务处理完成后的数据:{}", JSON.toJSONString(outMessage));
            if ("raw".equals(encryptType)) {
                log.debug("明文传输,不进行加密");
                response.getWriter().write(outMessage.toXml());
                log.debug("返回的response:{}", outMessage.toXml());
            } else if ("aes".equals(encryptType)) {
                log.debug("密文传输,进行加密");
                response.getWriter().write(outMessage.toEncryptedXml(wxMpConfigStorage));
                log.debug("返回的response:{}", outMessage.toEncryptedXml(wxMpConfigStorage));
            }
        }
    }

}
