package com.bxm.localnews.thirdparty.controller;

import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.thirdparty.constant.WechatMpTypeEnum;
import com.bxm.localnews.thirdparty.dto.WxUserInfo;
import com.bxm.localnews.thirdparty.param.WechatOnAuth2UrlParam;
import com.bxm.localnews.thirdparty.service.WxMpAuthService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.base.Splitter;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.api.WxConsts;
import me.chanjar.weixin.common.bean.WxJsapiSignature;
import me.chanjar.weixin.mp.api.WxMpService;
import me.chanjar.weixin.mp.bean.result.WxMpUser;
import me.chanjar.weixin.mp.bean.result.WxMpUserList;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Map;
import java.util.Objects;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author zhaoyadong 2018/10/17 13:52
 * @desc
 */
@Api(tags = "3-10 微信授权相关接口", description = "微信授权相关接口操作")
@RestController
@RequestMapping("api/public/auth")
@Slf4j
public class WxMpAuthController {

    private WxMpService wxMpService;

    private WxMpAuthService wxMpAuthService;

    private BizConfigProperties bizConfigProperties;

    @Autowired
    public WxMpAuthController(WxMpAuthService wxMpAuthService, WxMpService wxMpService,
                              BizConfigProperties bizConfigProperties) {
        this.wxMpAuthService = wxMpAuthService;
        this.wxMpService = wxMpService;
        this.bizConfigProperties = bizConfigProperties;
    }

    /**
     * 访问这个时便会发起微信的网页授权
     *
     * @return
     */
    @ApiOperation(value = "3-10-2 生成重定向地址", notes = "返回用户信息")
    @GetMapping("/authorize")
    public Json<String> authorize(@RequestParam("returnUrl") String returnUrl) {
        //设置回调地址
        String redirectUrl = wxMpService.oauth2buildAuthorizationUrl(returnUrl, WxConsts.OAuth2Scope.SNSAPI_USERINFO, "STAT");
        String str = StringUtils.substringAfterLast(redirectUrl, "?");
        Map<String, String> map = Splitter.on("&").withKeyValueSeparator("=").split(str);
        String code = map.get("code");
        return ResultUtil.genSuccessResult(code);
    }

    @ApiOperation(value = "3-10-1 根据code获取用户信息", notes = "返回结果：result.userId == null则需要进行注册或者绑定")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "授权登陆code", dataType = "String", required = true),
            @ApiImplicitParam(name = "type", value = "公众号类型: 为空时:默认公众号, JS_API: JS_API支付，其他非空为活动公众号", dataType = "String")
    })
    @GetMapping("/userInfo")
    public Json<WxUserInfo> userInfo(@RequestParam("code") String code,
                                     @RequestParam(required = false, value = "type") String type) {
        log.debug("根据微信授权code获取用户信息,code:{},type:{}", code, type);

        Message wxUserInfoMessage = this.wxMpAuthService.getUserByCode(code, type);
        if (wxUserInfoMessage.isSuccess()) {
            return ResultUtil.genSuccessResult(wxUserInfoMessage.getParam("wxUserInfo"));
        } else {
            return ResultUtil.genFailedResult(wxUserInfoMessage.getLastMessage());
        }
    }

    @ApiOperation(value = "根据openId获取用户列表", hidden = true)
    @GetMapping("/userList")
    public Json<WxMpUserList> userList(String nextOpenId) {
        return ResultUtil.genSuccessResult(this.wxMpAuthService.getUserList(nextOpenId));
    }

    @ApiOperation(value = "根据openId获取用户信息", hidden = true)
    @GetMapping("/{opendId}")
    public Json<WxMpUser> userInfoByOpenId(@PathVariable String opendId) {
        return ResultUtil.genSuccessResult(this.wxMpAuthService.getUser(opendId));
    }

    @ApiOperation(value = "3-10-3 获取js签名验证")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "url", value = "签名认证的url", dataType = "String", required = true),
            @ApiImplicitParam(name = "type", value = "公众号类型: 为空时:默认公众号, JS_API: JS_API支付，其他非空为活动公众号", dataType = "String", required = true)
    })
    @GetMapping("/jssignature")
    public Json<WxJsapiSignature> getWxJsapiSignature(@RequestParam("url") String url,
                                                      @RequestParam(required = false, value = "type") String type) {
        return ResultUtil.genSuccessResult(this.wxMpAuthService.getWxJsapiSignature(url, type));
    }

    @GetMapping("wechatOnAuth2Url")
    @ApiOperation(value = "3-10-4 生成微信授权地址", notes = "返回微信授权地址，打开后会唤起微信的网页授权")
    public Json<String> wechatOnAuth2Url(WechatOnAuth2UrlParam param) {

        return ResultUtil.genSuccessResult(wxMpAuthService.getWechatOnAuth2Url(param.getRedirectURI(),
                param.getScope(),
                param.getState(),
                param.getType()));
    }
}
