package com.bxm.localnews.thirdparty.service.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.thirdparty.config.FutureWeatherProperties;
import com.bxm.localnews.thirdparty.config.WeatherProperties;
import com.bxm.localnews.thirdparty.config.WindFutureWeatherProperties;
import com.bxm.localnews.thirdparty.constant.WeatherEnum;
import com.bxm.localnews.thirdparty.dto.WeatherDTO;
import com.bxm.localnews.thirdparty.dto.WeatherExtendDTO;
import com.bxm.localnews.thirdparty.service.WeatherService;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2018/10/18 13:54
 * @desc
 */
@Service
public class WeatherServiceImpl implements WeatherService {

    private static final Logger logger = LoggerFactory.getLogger(WeatherServiceImpl.class);

    private WeatherProperties weatherProperties;

    private FutureWeatherProperties futureWeatherProperties;

    private HttpClientService httpClientService;

    private RedisStringAdapter redisStringAdapter;

    private AliyunOSSService aliyunOSSService;

    private WindFutureWeatherProperties windFutureWeatherProperties;

    @Autowired
    public WeatherServiceImpl(WeatherProperties weatherProperties, FutureWeatherProperties futureWeatherProperties,
                              HttpClientService httpClientService, RedisStringAdapter redisStringAdapter,
                              AliyunOSSService aliyunOSSService, WindFutureWeatherProperties windFutureWeatherProperties) {
        this.weatherProperties = weatherProperties;
        this.futureWeatherProperties = futureWeatherProperties;
        this.httpClientService = httpClientService;
        this.redisStringAdapter = redisStringAdapter;
        this.aliyunOSSService = aliyunOSSService;
        this.windFutureWeatherProperties = windFutureWeatherProperties;
    }

    @Override
    public WeatherDTO getWeatherByAdcode(String adcode) {
        WeatherDTO weatherDTO = this.redisStringAdapter.get(getWeatherKey(adcode), WeatherDTO.class);
        if (weatherDTO == null) {
            HashMap<String, String> map = addCommonParams(adcode, "base");
            String json = httpClientService.doGet(weatherProperties.getWeatherUrl(), map);

            weatherDTO = getWeatherByJson(json);
        }

        return weatherDTO;
    }

    @Override
    public WeatherDTO getHomeWeatherByAdcode(String adcode) {
        WeatherDTO weatherDTO = getWeatherByAdcode(adcode);
        setHomeWeather(weatherDTO);

        return weatherDTO;
    }

    @Override
    public List<WeatherExtendDTO> getFutureWeather(String adcode) {
        TypeReference<List<WeatherExtendDTO>> typeReference = new TypeReference<List<WeatherExtendDTO>>() {

        };
        List<WeatherExtendDTO> weatherExtendDTOS = redisStringAdapter.get(getFutureWeatherKey(adcode), typeReference);
        if (CollectionUtils.isEmpty(weatherExtendDTOS)) {

//            HashMap<String, String> params = addParams(city, province);
            //调用和风天气的api
            HashMap<String, String> params = addWindParams(adcode);
            String json = httpClientService.doGet(windFutureWeatherProperties.getWeatherUrl(), params);
            weatherExtendDTOS = parseFutureWeather(json);

            //和风天气查询失败---查询高德天气
            if (CollectionUtils.isEmpty(weatherExtendDTOS)) {
                params = addCommonParams(adcode, "all");
                json = httpClientService.doGet(weatherProperties.getWeatherUrl(), params);
                weatherExtendDTOS = parseFutureWeatherByMap(json);
            }

            if (CollectionUtils.isNotEmpty(weatherExtendDTOS)) {
                redisStringAdapter.set(getFutureWeatherKey(adcode), weatherExtendDTOS, 3600);
            }
        }

        //设置图标
        if (CollectionUtils.isNotEmpty(weatherExtendDTOS)) {
            weatherExtendDTOS.forEach(weatherExtendDTO -> {
                setHomeWeather(weatherExtendDTO);
            });
        }
        return weatherExtendDTOS;
    }

    /**
     * 设置对象信息
     *
     * @param weatherDTO
     */
    private void setHomeWeather(WeatherDTO weatherDTO) {
        WeatherEnum weatherEnum = WeatherEnum.getWeatherEnum(weatherDTO.getWeather());
        weatherDTO.setWeather(weatherEnum == null ? "" : weatherEnum.getName());
        weatherDTO.setWeatherImg(weatherEnum == null ? "" : weatherEnum.getHomeImg());
    }

    /**
     * 组装请求参数
     *
     * @param adcode
     * @return
     */
    private HashMap<String, String> addCommonParams(String adcode, String extensions) {
        HashMap<String, String> map = Maps.newHashMap();
        map.put("key", weatherProperties.getKey());
        map.put("city", adcode);
        map.put("extensions", extensions);
        return map;
    }

    /**
     * 组装请求参数（和风天气）
     *
     * @param adcode
     * @return
     */
    private HashMap<String, String> addWindParams(String adcode) {
        HashMap<String, String> map = Maps.newHashMap();
        map.put("key", windFutureWeatherProperties.getKey());
        map.put("location", adcode);
        return map;
    }

    /**
     * 解析json转换为对象
     *
     * @param json
     * @return
     */
    private WeatherDTO getWeatherByJson(String json) {
        WeatherDTO weatherDTO = new WeatherDTO();

        if (StringUtils.isBlank(json)) {
            return weatherDTO;
        }

        JSONObject jsonObject = JSONObject.parseObject(json);
        if ("0".equals(jsonObject.getString("status"))) {
            return weatherDTO;
        }

        JSONArray jsonArray = jsonObject.getJSONArray("lives");
        com.alibaba.fastjson.TypeReference<List<WeatherDTO>> typeReference = new com.alibaba.fastjson.TypeReference<List<WeatherDTO>>() {
        };
        List<WeatherDTO> weatherDTOS = JSON.parseObject(jsonArray.toJSONString(), typeReference);

        if (CollectionUtils.isNotEmpty(weatherDTOS)) {
            weatherDTO = weatherDTOS.get(0);
            WeatherEnum weatherEnum = WeatherEnum.getWeatherEnum(weatherDTO.getWeather());
            weatherDTO.setWeather(weatherEnum == null ? "" : weatherEnum.getName());
            weatherDTO.setWeatherImg(weatherEnum == null ? "" : weatherEnum.getImg());
            weatherDTO.setTemperature(weatherDTO.getTemperature() + "°");
            this.redisStringAdapter.set(getWeatherKey(weatherDTO.getAdcode()), weatherDTO, 1800);
        }

        return weatherDTO;
    }

    /**
     * 转换jsonArray
     *
     * @param jsonArray
     */
    private List<WeatherExtendDTO> convertWeatherExtend(JSONArray jsonArray) {
        List<WeatherExtendDTO> weatherExtendDTOS = Lists.newArrayList();
        for (int i = 0; i < jsonArray.size(); i++) {
            weatherExtendDTOS.add(convertWeather(jsonArray.getJSONObject(i)));
        }
        return weatherExtendDTOS;
    }

    /**
     * 转换成天气对象
     *
     * @param jsonObject
     * @return
     */
    private WeatherExtendDTO convertWeather(JSONObject jsonObject) {
        WeatherExtendDTO weatherExtendDTO = new WeatherExtendDTO();
        //最高温度--最低温度
        String maxtmp = jsonObject.getString("tmp_max");
        String mintmp = jsonObject.getString("tmp_min");
        weatherExtendDTO.setMaxTemp(maxtmp);
        weatherExtendDTO.setMinTemp(mintmp);
        String date = jsonObject.getString("date");
        weatherExtendDTO.setDate(DateUtils.formatAtWill(DateUtils.parse(date), "MM/dd"));
        Date now = new Date();
        int diffDay = DateUtils.getDiffDays(DateUtils.parse(date),
                DateUtils.parse(DateUtils.formatAtWill(now, DateUtils.DATE_FORMAT)), true);
        weatherExtendDTO.setWeek(dateToWeek(date));
        if (diffDay == 0) {
            weatherExtendDTO.setWeek("今天");
        } else if (diffDay == 1) {
            weatherExtendDTO.setWeek("明天");
        }
        weatherExtendDTO.setHumidity(jsonObject.getString("hum"));
        weatherExtendDTO.setWindpower(jsonObject.getString("wind_sc"));
        weatherExtendDTO.setWinddirection(jsonObject.getString("wind_dir"));

        weatherExtendDTO.setWeather(jsonObject.getString("cond_txt_d"));
        WeatherEnum weatherEnum = WeatherEnum.getWeatherEnum(weatherExtendDTO.getWeather());
        if (weatherEnum != null) {
            weatherExtendDTO.setWeatherImg(weatherEnum.getImg());
        } else {
            logger.info("当天的天气信息为：{}", weatherExtendDTO.getWeather());
        }
        weatherExtendDTO.setWeatherType(convertWeatherType(weatherExtendDTO.getWeather()));
        return weatherExtendDTO;
    }

    /**
     * 解析json数据---获取和风未来7天的天气信息
     *
     * @param json
     * @return
     */
    private List<WeatherExtendDTO> parseFutureWeather(String json) {

        List<WeatherExtendDTO> weatherExtendDTOS = Lists.newArrayList();

        if (StringUtils.isBlank(json)) {
            return weatherExtendDTOS;
        }
        JSONObject jsonObject = JSONObject.parseObject(json);
        JSONArray resultJsonArray = jsonObject.getJSONArray("HeWeather6");

        //解析结果
        JSONObject result = resultJsonArray.getJSONObject(0);

        if (!"ok".equals(result.getString("status"))) {
            return weatherExtendDTOS;
        }

        JSONArray jsonArray = result.getJSONArray("daily_forecast");
        weatherExtendDTOS = convertWeatherExtend(jsonArray);

        return weatherExtendDTOS;
    }

    /**
     * 解析高德天气
     *
     * @param json
     * @return
     */
    private List<WeatherExtendDTO> parseFutureWeatherByMap(String json) {
        List<WeatherExtendDTO> weatherExtendDTOS = Lists.newArrayList();
        if (StringUtils.isEmpty(json)) {
            return weatherExtendDTOS;
        }

        JSONObject jsonObject = JSONObject.parseObject(json);
        if ("0".equals(jsonObject.getString("status"))) {
            return weatherExtendDTOS;
        }

        JSONArray resultJsonArray = jsonObject.getJSONArray("forecasts");

        JSONArray jsonArray = resultJsonArray.getJSONObject(0).getJSONArray("casts");
        for (int i = 0; i < jsonArray.size(); i++) {
            weatherExtendDTOS.add(convertMapWeather(jsonArray.getJSONObject(i)));
        }
        return weatherExtendDTOS;
    }

    /**
     * 转换高德天气信息
     *
     * @param jsonObject
     * @return
     */
    private WeatherExtendDTO convertMapWeather(JSONObject jsonObject) {
        WeatherExtendDTO weatherExtendDTO = new WeatherExtendDTO();

        String date = jsonObject.getString("date");
        weatherExtendDTO.setDate(DateUtils.formatAtWill(DateUtils.parse(date), "MM/dd"));
        Date now = new Date();
        int diffDay = DateUtils.getDiffDays(DateUtils.parse(date),
                DateUtils.parse(DateUtils.formatAtWill(now, DateUtils.DATE_FORMAT)), true);
        weatherExtendDTO.setWeek(dateToWeek(date));
        if (diffDay == 0) {
            weatherExtendDTO.setWeek("今天");
        } else if (diffDay == 1) {
            weatherExtendDTO.setWeek("明天");
        }

        weatherExtendDTO.setWeather(jsonObject.getString("dayweather"));
        weatherExtendDTO.setMaxTemp(jsonObject.getString("daytemp"));
        weatherExtendDTO.setMinTemp(jsonObject.getString("nighttemp"));
        weatherExtendDTO.setWinddirection(jsonObject.getString("daywind"));
        weatherExtendDTO.setWindpower(jsonObject.getString("daypower"));
        WeatherEnum weatherEnum = WeatherEnum.getWeatherEnum(weatherExtendDTO.getWeather());
        if (weatherEnum != null) {
            weatherExtendDTO.setWeatherImg(weatherEnum.getImg());
        } else {
            logger.info("当天的天气信息为：{}", weatherExtendDTO.getWeather());
        }
        weatherExtendDTO.setWeatherType(convertWeatherType(weatherExtendDTO.getWeather()));
        return weatherExtendDTO;
    }

    /**
     * 获取天气的key
     *
     * @param areaCode 地区编码
     * @return
     */
    private KeyGenerator getWeatherKey(String areaCode) {
        return RedisConfig.THIRDPARTY_WEATHER.copy().appendKey(areaCode);
    }

    /**
     * 获取未来7天天气的key
     *
     * @param areaCode 地区编码
     * @return
     */
    private KeyGenerator getFutureWeatherKey(String areaCode) {
        return RedisConfig.THIRDPARTY_FUTURE_WEATHER.copy().appendKey(areaCode);
    }

    /**
     * 根据日期返回星期几
     *
     * @param datetime
     * @return
     */
    private String dateToWeek(String datetime) {
        String[] weekDays = {"周日", "周一", "周二", "周三", "周四", "周五", "周六"};
        // 获得一个日历
        Calendar cal = Calendar.getInstance();
        cal.setTime(DateUtils.parse(datetime));
        // 指示一个星期中的某天。
        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0) {
            w = 0;
        }
        return weekDays[w];
    }

    /**
     * 天气转换类型
     *
     * @param weather
     * @return
     */
    private Byte convertWeatherType(String weather) {
        Byte[] weatherType = {1, 2, 3, 4, 5};
        if (weather.contains("云")) {
            return weatherType[0];
        }
        if (weather.contains("晴")) {
            return weatherType[1];
        }
        if (weather.contains("阴")) {
            return weatherType[2];
        }
        if (weather.contains("雪")) {
            return weatherType[3];
        }
        if (weather.contains("雨")) {
            return weatherType[4];
        }
        return 1;
    }
}
