package com.bxm.localnews.user.event.subscribe;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.user.event.UserActionEvent;
import com.bxm.localnews.user.event.stream.MessageProcessor;
import com.google.common.eventbus.Subscribe;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.integration.support.MessageBuilder;
import org.springframework.stereotype.Component;

/**
 * 用户行为定义分发，将用户行为分发到消息队列供其他服务消费
 *
 * @author liujia
 * @date 2020-07-16 15:11
 **/
@Component
@Slf4j
public class UserActionStreamSubscriber {

    private final MessageProcessor messageProcessor;

    @Autowired
    public UserActionStreamSubscriber(MessageProcessor messageProcessor) {
        this.messageProcessor = messageProcessor;
    }

    /**
     * 将用户行为分发到不同的消息队列供其他服务进行消费
     * 通过注册到EventBus进行订阅处理
     *
     * @param event 用户行为事件
     */
    @Subscribe
    public void consume(UserActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("消费用户行为数据：{}", JSON.toJSONString(event));
        }

        switch (event.getAction()) {
            case LOCATION:
                messageProcessor.locationActionOutput().send(MessageBuilder.withPayload(event).build());
                break;
            case ATTRIBUTE:
                messageProcessor.attributeActionOutput().send(MessageBuilder.withPayload(event).build());
                break;
            case REDUNDANCY:
                messageProcessor.redundancyActionOutput().send(MessageBuilder.withPayload(event).build());
                break;
            case ACTIVE:
                messageProcessor.activeActionOutput().send(MessageBuilder.withPayload(event).build());
                break;
            default:
                log.info("用户行为事件未定义：{}", JSON.toJSONString(event));
                break;
        }
    }
}
