package com.bxm.localnews.user.service;

import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.dto.LocationUserInfoDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.param.TempUserParam;
import com.bxm.localnews.user.param.UserParam;
import com.bxm.localnews.user.vo.LoginInfo;
import com.bxm.localnews.user.vo.LoginMeta;
import com.bxm.localnews.user.vo.User;
import com.bxm.newidea.component.vo.Message;

import org.springframework.web.multipart.MultipartFile;

public interface UserService {

    /**
     * 用途：app初始静默注册
     * 场景：在用户打开app时，系统会为用户生成一个默认用户，以这个用户来记录用户阅读的新闻
     *
     * @param loginInfo
     * @param basicParam    基础参数
     * @param firstOpenType 是否是第一次打开应用，1：首次打开
     * @return
     */
    Json<LoginMeta> register(LoginInfo loginInfo, BasicParam basicParam, String firstOpenType);

    /**
     * 用途：用户h5注册
     * 场景：当用户分享后，别的用户可以在h5上注册，通过验证码方式验证
     *
     * @param loginInfo
     * @param platform  平台信息
     * @param chnl      渠道信息,当前APP的平台
     * @return
     */
    Json<User> doH5Register(LoginInfo loginInfo, Integer platform, String chnl);

    /**
     * 用途：从redis中取得用户信息，如果redis中不存在，则在数据库中查找并更新至redis中
     *
     * @param userId
     * @return
     */
    UserInfoDTO getUserFromRedisDB(long userId);

    /**
     * 用途：user->userInfoDTO
     *
     * @param user
     * @return
     */
    UserInfoDTO getUserToDTO(User user);

    /**
     * 用途：获取用户支付token
     *
     * @return
     */
    String getToken();

    /**
     * 用途：得到token失效时间
     *
     * @return
     */
    Long getExpireTime();

    /**
     * 用途：重置用户token
     *
     * @param userId
     * @param refreshToken
     * @return
     */
    Json<UserInfoDTO> refreshToken(Long userId, String refreshToken);

    /**
     * 用途：新增用户以及用户相关的配置（包括用户账户，新手计划，邀请人信息）
     *
     * @param loginInfo
     * @param platform
     * @param chnl
     * @param userType
     * @return
     */
    User addUser(LoginInfo loginInfo, Integer platform, String chnl, Byte userType);

    /**
     * 用途：绑定账号（手机、微信、qq，微博）
     *
     * @param loginInfo
     * @param userId
     * @return
     */
    Json<UserInfoDTO> binding(LoginInfo loginInfo, Long userId);

    /**
     * 用途：更新用户信息
     *
     * @param user
     * @param userId
     * @return
     */
    Message updateUserInfo(UserParam user, Long userId, MultipartFile file);

    /**
     * 用途：判断该手机号是否注册
     *
     * @param phone
     * @return
     */
    /**
     * 用途：判断该手机号是否注册
     *
     * @param phone
     * @param flag  标记是否验证微信绑定 0不用验证/1需要验证
     * @return
     */
    boolean checkUserExistByPhone(String phone, Integer flag);

    /**
     * 获取用户定位区域、基本信息
     *
     * @param userId
     * @return
     */
    LocationUserInfoDTO getLocationUserInfo(Long userId);

    /**
     * 获取用户默认头像
     *
     * @return
     */
    String getDefaultHeadImgUrl();

    /**
     * 从数据库读取用户信息
     *
     * @param userId
     * @return
     */
    User selectByPrimaryKey(Long userId);

    /**
     * 判断unionId是否被使用
     *
     * @param unionId
     * @return
     */
    Boolean checkUnionIdIsUser(String unionId);

    /**
     * 创建临时用户
     *
     * @param userParam
     * @return
     */
    Message createUser(TempUserParam userParam);

    /**
     * 根据unionId获取用户信息
     *
     * @param unionId
     * @return
     */
    User selectUserByUnionId(String unionId);

    /**
     * 根据设备id获取用户信息
     *
     * @param equipment
     * @return
     */
    User selectUserByEquipment(String equipment);

    /**
     * 判断对应设备是否存在
     *
     * @param equipment
     * @return
     */
    Long checkExistEquipment(String equipment);

    /**
     * 更新用户已领取红包
     *
     * @param userId
     * @return
     */
    Boolean updateUserRecieveRedPacket(Long userId);

    /**
     * 用户是否是临时用户
     *
     * @param userId
     * @return
     */
    Boolean isTempUser(Long userId);

    /**
     * 用户是否是风险用户
     *
     * @param userId
     * @return
     */
    Boolean isRiskUser(Long userId);

    /**
     * 帖子、评论数+1
     *
     * @param userId
     * @param type
     * @return
     */
    Boolean addPostReplyNum(Long userId, Integer type);
}