package com.bxm.localnews.user.service.impl;

import java.math.BigDecimal;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.config.CommonProperties;
import com.bxm.localnews.user.domain.UserLocationHistoryMapper;
import com.bxm.localnews.user.dto.LocationDTO;
import com.bxm.localnews.user.dto.UserCashGoldDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.UserRedPacketDTO;
import com.bxm.localnews.user.enums.AwardTypeEnum;
import com.bxm.localnews.user.enums.CashEnum;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.InviteRecordStateEnum;
import com.bxm.localnews.user.integration.LocationIntegrationService;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.service.*;
import com.bxm.localnews.user.vo.InviteRecord;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserEquipment;
import com.bxm.localnews.user.vo.UserLocationHistory;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.service.BaseService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2019/3/8 14:38
 * @desc
 */
@Service
@RefreshScope
public class RedPacketServiceImpl extends BaseService implements RedPacketService {

    private UserService userService;

    private UserAccountService userAccountService;

    private LocationIntegrationService locationIntegrationService;

    private InviteRecordService inviteRecordService;

    private CommonProperties commonProperties;

    private DistributedLock distributedLock;

    private UserLocationHistoryMapper userLocationHistoryMapper;

    private UserEquipmentService userEquipmentService;

    @Autowired
    public RedPacketServiceImpl(UserService userService, UserAccountService userAccountService,
                                LocationIntegrationService locationIntegrationService,
                                InviteRecordService inviteRecordService, CommonProperties commonProperties,
                                DistributedLock distributedLock, UserLocationHistoryMapper userLocationHistoryMapper,
                                UserEquipmentService userEquipmentService) {
        this.userService = userService;
        this.userAccountService = userAccountService;
        this.locationIntegrationService = locationIntegrationService;
        this.inviteRecordService = inviteRecordService;
        this.commonProperties = commonProperties;
        this.distributedLock = distributedLock;
        this.userLocationHistoryMapper = userLocationHistoryMapper;
        this.userEquipmentService = userEquipmentService;
    }

    @Override
    public Json<BigDecimal> userRecieveRedPacket(Long userId, BasicParam basicParam, String areaCode) {
        LocationDTO locationDTO = getLocationByAreaCode(areaCode);
        if (!userIsReceiveRedPacket(userId, locationDTO, basicParam.getDevcId())) {
            logger.info("当前用户[{}]已领取红包", userId);
            return ResultUtil.genFailedResult("请稍后再试");
        }
        logger.info("用户[{}]在地区[{}]领取红包金额为：[{}]", userId, areaCode, locationDTO.getNewRedPacket());
        String requestId = nextSequence().toString();
        if (distributedLock.lock(userId.toString(), requestId)) {
            //用户的邀请人增加永久赏金------用户增加可提现现金
            User user = userService.selectByPrimaryKey(userId);
            if (user.getInviteUserId() != null) {
                logger.info("用户[{}]的邀请人不为空，邀请人id：[{}]", userId, user.getInviteUserId());
                InviteRecord inviteRecord = inviteRecordService.getInviteRecordByInvitedUser(user.getInviteUserId(), userId);
                if (inviteRecord != null) {
                    logger.info("邀请人[{}]增加永久赏金[{}]", user.getInviteUserId(), locationDTO.getPromoteBounty());
                    inviteRecord.setAward(inviteRecord.getAward().add(locationDTO.getPromoteBounty()));
                    inviteRecord.setInviteState(InviteRecordStateEnum.LOGIN_APP.getName());
                    inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
                    updateInviteRecord(inviteRecord);

                    addUserAccountByPromote(inviteRecord.getUserId(), locationDTO.getPromoteBounty(), inviteRecord.getId());
                }
            }

            logger.info("当前用户[{}]增加红包金额[{}]", userId, locationDTO.getNewRedPacket());
            addUserAccountByRedPacket(userId, locationDTO.getNewRedPacket(), null);

            //更新用户已领取红包
            userService.updateUserRecieveRedPacket(userId);

            //添加对应设备号已领取红包
            addReceiveEquipment(basicParam.getDevcId());

            distributedLock.unlock(userId.toString(), requestId);
        } else {
            return ResultUtil.genFailedResult("你已领取过新人红包");
        }


        return ResultUtil.genSuccessResult(locationDTO.getNewRedPacket());
    }

    /**
     * 添加对应设备号已领取红包
     *
     * @param equipment
     */
    private void addReceiveEquipment(String equipment) {
        UserEquipment userEquipment = new UserEquipment(nextId(), equipment, "DRAW");
        userEquipmentService.addUserEquipment(userEquipment);
    }

    /**
     * 添加红包金额---用户增加可提现赏金
     *
     * @param userId     用户
     * @param award      红包金额
     * @param relationId 关系id
     */
    private void addUserAccountByRedPacket(Long userId, BigDecimal award, Long relationId) {
        AccountCashParam accountCashParam = new AccountCashParam(userId, CashEnum.DRAWABLEL_CASH.name(), Boolean.TRUE,
                award, relationId, CashFlowTypeEnum.RED_PACKET.name(), null);
        userAccountService.addCash(accountCashParam);
    }

    /**
     * 邀请人增加永久赏金
     *
     * @param userId     用户
     * @param award      赏金金额
     * @param relationId 关系id
     */
    private void addUserAccountByPromote(Long userId, BigDecimal award, Long relationId) {
        AccountCashParam accountCashParam = new AccountCashParam(userId, CashEnum.STABILIZE_CASH.name(), Boolean.TRUE,
                award, relationId, CashFlowTypeEnum.PROMOTE_BOUNTY.name(), null);
        userAccountService.addCash(accountCashParam);
    }

    /**
     * 更新邀请关系
     *
     * @param inviteRecord
     */
    private void updateInviteRecord(InviteRecord inviteRecord) {
        inviteRecordService.updateInviteRecord(inviteRecord);
    }

    @Override
    public UserRedPacketDTO enableRedPacket(BasicParam basicParam, Long userId, String areaCode) {

        UserRedPacketDTO userRedPacketDTO = new UserRedPacketDTO();
        LocationDTO locationDTO = getLocationByAreaCode(areaCode);
        if (!userIsReceiveRedPacket(userId, locationDTO, basicParam.getDevcId())) {
            return userRedPacketDTO;
        }

        userRedPacketDTO.setEnableRedPacket(byteToBoolean(locationDTO.getEnablePaidPromote()));
        userRedPacketDTO.setAmount(locationDTO.getNewRedPacket());

        userRedPacketDTO.setInviteUserHeadImg(commonProperties.getAppIconUrl());
        userRedPacketDTO.setInviteUserNickname(commonProperties.getAppName());
        userRedPacketDTO.setLinkUrl(commonProperties.getLinkUrl());
        User user = userService.selectByPrimaryKey(userId);
        if (user != null && user.getInviteUserId() != null) {
            InviteRecord inviteRecord = inviteRecordService.getInviteRecordByInvitedUser(user.getInviteUserId(), userId);
            if (inviteRecord != null) {
                UserInfoDTO userInfoDTO = userService.getUserFromRedisDB(user.getInviteUserId());
                userRedPacketDTO.setInviteUserNickname(userInfoDTO.getNickname());
                userRedPacketDTO.setInviteUserHeadImg(userInfoDTO.getHeadImg());
            }
        }

        return userRedPacketDTO;
    }

    /**
     * 用户是否可以领取红包
     *
     * @param userId      用户id
     * @param equipment   设备号
     * @param locationDTO 地区信息
     * @return
     */
    private Boolean userIsReceiveRedPacket(Long userId, LocationDTO locationDTO, String equipment) {
        logger.info("-----------  进入用户是否可以领取红包的判断  -----------------");

        logger.info("当前用户[{}]的设备号[{}]，实时定位信息：[{}]", userId, equipment, JSON.toJSONString(locationDTO));

        //当前区域是否开通新人红包
        if (locationDTO == null || !byteToBoolean(locationDTO.getEnablePaidPromote())) {
            logger.info("当前地区[{}]未开通推广赚钱", locationDTO.getName());
            return Boolean.FALSE;
        }

        UserLocationHistory userLocationHistory = userLocationHistoryMapper.getLastLocationByUser(userId);
        if (userLocationHistory != null) {
            LocationDTO userLocation = getLocationByAreaCode(userLocationHistory.getLocationCode());

            //当前区域是否开通新人红包
            if (userLocation == null || !byteToBoolean(userLocation.getEnablePaidPromote())) {
                logger.info("当前用户选择的地区[{}]未开通推广赚钱", userLocation.getName());
                return Boolean.FALSE;
            }
        }

        //判断设备号是否可以领取红包
        if (userEquipmentService.checkExistEquipment(equipment, "DRAW")) {
            logger.info("当前设备[{}]已领取红包", equipment);
            return Boolean.FALSE;
        }

        //判断用户是否可以领取红包
        User user = userService.selectByPrimaryKey(userId);
        if (user == null) {
            logger.info("当前用户[{}]不存在", userId);
            return Boolean.FALSE;
        }

        if (user != null && !byteToBoolean(user.getReceiveRedPacket())) {
            logger.info("当前用户[{}]已领取红包", userId);
            return Boolean.FALSE;
        }

        //我的钱包余额不为0
        UserCashGoldDTO userCashGoldDTO = userAccountService.getUserCashGold(userId);
        if (userCashGoldDTO.getCash().compareTo(BigDecimal.ZERO) != 0) {
            logger.info("用户[{}]的余额[{}]不为0", userId, userCashGoldDTO.getCash());
            return Boolean.FALSE;
        }

        logger.info("-----------  当前用户[{}]可以领取红包  -----------------", userId);

        return Boolean.TRUE;
    }

    /**
     * 根据地区编码获取区域信息
     *
     * @param areaCode
     * @return
     */
    private LocationDTO getLocationByAreaCode(String areaCode) {
        return locationIntegrationService.getLocationByGeocode(areaCode);
    }

    /**
     * byte类型转为Boolean类型
     *
     * @return
     */
    private Boolean byteToBoolean(byte byteNumber) {
        return byteNumber == 1 ? Boolean.TRUE : Boolean.FALSE;
    }
}
