package com.bxm.localnews.user.service.impl;

import java.math.BigDecimal;
import java.util.Date;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.PushSoundEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserAccountMapper;
import com.bxm.localnews.user.dto.UserAccountDTO;
import com.bxm.localnews.user.dto.UserAccountDayCashDTO;
import com.bxm.localnews.user.dto.UserCashGoldDTO;
import com.bxm.localnews.user.dto.UserWithdrawDTO;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.integration.MessageUserIntegrationService;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.integration.UserEventIntegrationService;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.param.AccountGoldParam;
import com.bxm.localnews.user.service.*;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.vo.Message;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2019/3/6 9:25
 * @desc
 */
@Service
public class UserAccountServiceImpl extends BaseService implements UserAccountService {

    private UserAccountMapper userAccountMapper;

    private CashFlowService cashFlowService;

    private WithdrawService withdrawService;

    @Autowired
    private UserService userService;

    @Autowired
    private InviteRecordService inviteRecordService;

    @Autowired
    private UserEventIntegrationService userEventIntegrationService;

    @Autowired
    private GoldFlowService goldFlowService;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Autowired
    private MessageUserIntegrationService messageUserIntegrationService;

    @Autowired
    public UserAccountServiceImpl(UserAccountMapper userAccountMapper, CashFlowService cashFlowService,
                                  WithdrawService withdrawService) {
        this.userAccountMapper = userAccountMapper;
        this.cashFlowService = cashFlowService;
        this.withdrawService = withdrawService;
    }

    @Override
    public BigDecimal getUserTotalCash(Long userId) {
        return getUserAccountDetail(userId).getTotalCash();
    }

    @Override
    public UserAccount getUserAccountDetail(Long userId) {
        logger.debug("从数据库获取用户账户信息：用户id:[{}]", userId);
        return getAndSetUserAccount(userId);
    }

    @Override
    public Message createUserAccount(Long userId) {
        logger.debug("创建用户账户信息：用户id:[{}]", userId);
        getAndSetUserAccount(userId);
        return Message.build();
    }

    @Override
    public UserAccountDTO getUserAccount(Long userId) {
        logger.debug("转换用户账户信息：从userAccount 转为userAccountDTO;用户id:[{}]", userId);
        UserAccountDTO userAccountDTO = convertUserAccount(getAndSetUserAccount(userId));

        if (!redisStringAdapter.hasKey(getUserWithdrawPop(userId))) {
            if (BigDecimal.ZERO.compareTo(userAccountDTO.getDrawablelCash().add(userAccountDTO.getNotArriveCash())) < 1) {
                redisStringAdapter.set(getUserWithdrawPop(userId), 1);
            }
            userAccountDTO.setEnablePopWithdraw(true);
        }
        return userAccountDTO;
    }

    @Override
    public UserAccountDayCashDTO getUserAccountAndDetail(Long userId) {
        UserWithdrawDTO userWithdrawDTO = withdrawService.getUserRecentWithdrawInfo(userId);
        //判断用户最后一笔订单是否处于审核状态
        if (userWithdrawDTO == null || WithdrawEnum.SUCCESS_PAYMENT.getState().equals(userWithdrawDTO.getStatus())
                || WithdrawEnum.FAIL_PAYMENT.getState().equals(userWithdrawDTO.getStatus())) {
            userWithdrawDTO = null;
        }

        return new UserAccountDayCashDTO(getUserAccount(userId),
                cashFlowService.getRecentSevenDayCashFlowByUser(userId),
                userWithdrawDTO);
    }

    @Override
    public UserCashGoldDTO getUserCashGold(Long userId) {
        logger.debug("获取用户账户在个人中心的信息：用户id:[{}]", userId);
        UserAccount userAccount = getAndSetUserAccount(userId);
        BigDecimal cash = userAccount.getDrawablelCash().add(userAccount.getTempCash())
                .add(userAccount.getStabilizeCash());
        return new UserCashGoldDTO(cash, userAccount.getUsableGold());
    }

    @Override
    public Boolean addCash(AccountCashParam param) {
        if (null == param || null == param.getUserId() || null == param.getCashType() ||
                null == param.getAddTotal() || null == param.getCash() || new BigDecimal(0).compareTo(param.getCash()) == 0) {
            return false;
        }

        logger.debug("用户账户金额更新：{}", param);
        int isAdded = 0;

        if (CashEnum.TEMP_CASH.name().equals(param.getCashType())) {
            isAdded = userAccountMapper.addUserTempCash(param.getUserId(), param.getCash());
        } else if (CashEnum.STABILIZE_CASH.name().equals(param.getCashType())) {
            isAdded = userAccountMapper.addUserStabilizeCash(param.getUserId(), param.getCash());
        } else if (CashEnum.DRAWABLEL_CASH.name().equals(param.getCashType())) {
            isAdded = userAccountMapper.addUserDrawableCash(param.getUserId(), param.getCash());
        }

        if (isAdded > 0) {
            if (param.getAddTotal()) {
                userAccountMapper.addUserTotalCash(param.getUserId(), param.getCash());
                addUserEvent(param.getUserId(), param.getCash());
                // 不发送推送  用户红包不发
                if (!CashFlowTypeEnum.RED_PACKET.name().equals(param.getCashFlowType())) {
                    pushChangeAccountMsg(param.getUserId(), param.getCash());
                }
            }

            CashFlow cashFlow = new CashFlow();
            cashFlow.setId(nextSequence());
            cashFlow.setUserId(param.getUserId());
            cashFlow.setAmount(param.getCash());
            cashFlow.setRelationId(param.getRelationId());
            cashFlow.setType(param.getCashFlowType());
            cashFlow.setConvertType(param.getConvertType());
            cashFlow.setCreateTime(new Date());
            cashFlowService.addCashFlow(cashFlow);

            return true;
        } else {
            return false;
        }

    }

    @Override
    public Boolean transferCash(AccountCashParam param) {
        logger.debug("转换的账户信息：[{}]", JSON.toJSONString(param));
        if (null == param || null == param.getUserId() || null == param.getCashType() || null == param.getCash()
                || null == param.getConvertType() || param.getCash().compareTo(new BigDecimal(0)) <= 0) {
            return false;
        }

        logger.debug("用户账户金额转化：{}", param);
        int isTransferred = 0;

        if (CashFlowConvertTypeEnum.TEMP_TO_STABILIZE.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.tempToStabilize(param.getUserId(), param.getCash());
        } else if (CashFlowConvertTypeEnum.STABILIZE_TO_DRAWABLE.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.stabilizeToDrawable(param.getUserId(), param.getCash());
        } else if (CashFlowConvertTypeEnum.DRAWABLE_TO_WITHDRAWAL.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.drawableToWithdrawal(param.getUserId(), param.getCash());
        } else if (CashFlowConvertTypeEnum.TEMP_TO_DRAWABLE.name().equals(param.getConvertType())) {
            isTransferred = userAccountMapper.tempToDrawable(param.getUserId(), param.getCash());
        }

        if (isTransferred > 0) {
            CashFlow cashFlow = new CashFlow();
            cashFlow.setId(nextSequence());
            cashFlow.setUserId(param.getUserId());
            cashFlow.setAmount(param.getCash());
            cashFlow.setRelationId(param.getRelationId());
            cashFlow.setType(param.getCashFlowType());
            cashFlow.setConvertType(param.getConvertType());
            cashFlow.setCreateTime(new Date());
            cashFlowService.addCashFlow(cashFlow);

            return true;
        } else {
            return false;
        }

    }

    @Override
    public Message updateUserWithdrawInfo(Long userId, BigDecimal amount, Byte state) {
        //提现成功
        if (WithdrawEnum.SUCCESS_PAYMENT.getState().equals(state)) {
            //更新用户提现赏金转为已提现赏金
            updateUserCash(userId, amount);
            //更新邀请人的赏金为可提现赏金
            updateInviteUserCash(userId);
        }

        //提现失败
        if (WithdrawEnum.FAIL_PAYMENT.getState().equals(state)) {
            AccountCashParam accountCashParam = new AccountCashParam(userId, CashEnum.DRAWABLEL_CASH.name(), Boolean.FALSE,
                    amount, null, CashFlowTypeEnum.FAIL_WITHDRAWAL.name(),
                    null);
            addCash(accountCashParam);
        }
        return Message.build();
    }

    @Override
    public BigDecimal getUserDrawablelCash(Long userId) {
        return getUserAccount(userId).getDrawablelCash();
    }

    @Override
    public Boolean addGold(AccountGoldParam param) {
        logger.debug("用户账户更新金币的参数：[{}]", param);
        if (null == param || null == param.getUserId() || null == param.getGoldType()
                || null == param.getAddTotal() || null == param.getGold() || new BigDecimal(0).equals(param.getGold())) {
            return false;
        }

        logger.debug("用户账户金币更新：{}", param);
        Integer gold = param.getGold();
        if (GoldEnum.TEMP_GOLD.name().equals(param.getGoldType())) {
            userAccountMapper.addUserTempGold(param.getUserId(), gold);
        } else if (GoldEnum.USABLE_GOLD.name().equals(param.getGoldType())) {
            userAccountMapper.addUserUsableGold(param.getUserId(), gold);
        } else if (GoldEnum.CONSUME_GOLD.name().equals(param.getGoldType())) {
            userAccountMapper.addUserConsumeGold(param.getUserId(), gold);
            gold = Math.negateExact(gold);
        }

        if (param.getAddTotal()) {
            userAccountMapper.addUserTotalGold(param.getUserId(), gold);
        }

        GoldFlow goldFlow = new GoldFlow(nextSequence(), param.getUserId(), gold, param.getGoldFlowType(),
                param.getRelationId());
        goldFlowService.addGoldFlow(goldFlow);

        addMsg(goldFlow);

        return true;
    }

    @Override
    public Integer getUserUsableGold(Long userId) {
        logger.debug("获取用户[{}]的账户金币可用余额", userId);
        return getUserAccountDetail(userId).getUsableGold();
    }

    @Override
    public void userWithdrawPop(Long userId) {
        redisStringAdapter.set(getUserWithdrawPop(userId), 1);
    }

    /**
     * 构建一个用户账户信息
     *
     * @param userId
     * @return
     */
    private UserAccount generateUserAccount(Long userId) {
        UserAccount userAccount = new UserAccount(userId);
        userAccount.setId(nextSequence());
        return userAccount;
    }

    /**
     * 转换账户信息
     *
     * @param userAccount
     * @return
     */
    private UserAccountDTO convertUserAccount(UserAccount userAccount) {
        UserAccountDTO userAccountDTO = new UserAccountDTO();
        userAccountDTO.setId(userAccount.getId());
        userAccountDTO.setUserId(userAccount.getUserId());
        userAccountDTO.setDrawablelCash(userAccount.getDrawablelCash());
        //计算未到账金额---临时金额加上固化现金
        userAccountDTO.setNotArriveCash(userAccount.getTempCash().add(userAccount.getStabilizeCash()));
        userAccountDTO.setTotalCash(userAccount.getTotalCash());

        return userAccountDTO;
    }

    /**
     * 获取用户账户信息（不存在则创建）
     *
     * @param userId
     * @return
     */
    private UserAccount getAndSetUserAccount(Long userId) {
        UserAccount userAccount = this.userAccountMapper.getUserAccount(userId);
        if (userAccount == null) {
            logger.error("用户[{}]账户信息不存在", userId);
            userAccount = generateUserAccount(userId);
            this.userAccountMapper.addUserAccount(userAccount);
        } else {
            BigDecimal usableGold = BigDecimal.valueOf(userAccount.getUsableGold());
            if (usableGold != null && usableGold.compareTo(new BigDecimal(0)) < 0) {
                userAccount.setUsableGold(0);
            }
        }

        return userAccount;
    }

    /**
     * 添加用户账户变化事件
     */
    private void addUserEvent(Long userId, BigDecimal amount) {
        UserCashGoldDTO userCashGoldDTO = getUserCashGold(userId);
        PushPayloadInfo pushPayloadInfo = PushPayloadInfo.build(PushMessageEnum.USER_ACCOUNT_ENVET)
                .addExtend("userId", userId).addExtend("current", userCashGoldDTO.getCash())
                .addExtend("val", amount);
        userEventIntegrationService.addUserEvent(pushPayloadInfo);
    }

    /**
     * 推送金额变化
     *
     * @param userId
     * @param amount
     */
    private void pushChangeAccountMsg(Long userId, BigDecimal amount) {
        String title = "\uD83D\uDCB0" + "新增赏金" + amount.setScale(2, BigDecimal.ROUND_HALF_UP) + "元，点击查看 >";
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.ADD_CASH);

        PushMessage message = PushMessage.build();
        message.setTitle("通知消息");
        message.setContent(title);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
        message.setPayloadInfo(info);
        message.setSound(PushSoundEnum.GOLD_SOUND);

        this.pushMsgIntegrationService.pushMsg((message));
    }

    /**
     * 添加金币消息
     *
     * @param goldFlow
     */
    private void addMsg(GoldFlow goldFlow) {
        String title = "";
        Integer type = 1;
        if (goldFlow.getAmount() > 0) {
            title = "账户余额增加" + goldFlow.getAmount() + "金币，目前共有" + getUserUsableGold(goldFlow.getUserId())
                    + "金币啦！";
        } else {
            title = "金币账户扣除" + Math.abs(goldFlow.getAmount()) + "金币，目前剩余"
                    + getUserUsableGold(goldFlow.getUserId()) + "金币";
            type = 2;
        }

        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.UPDATE_GOLD);
        info.setMsgId(nextId());
        info.addExtend("type", type);
        PushMessage message = PushMessage.build();
        message.setTitle("通知消息");
        message.setContent(title);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(goldFlow.getUserId()));
        message.setPayloadInfo(info);
        message.setSound(PushSoundEnum.GOLD_SOUND);

        messageUserIntegrationService.addMessage(message, goldFlow.getUserId());
    }

    /**
     * 更新邀请人的赏金信息
     *
     * @param userId
     */
    private void updateInviteUserCash(Long userId) {
        logger.info("------------- 更新邀请人的永久赏金为可提现金额 ----------------");
        User user = userService.selectByPrimaryKey(userId);
        if (user.getInviteUserId() == null) {
            logger.info("用户[{}]的邀请人为空", userId);
            return;
        }

        logger.info("用户[{}]的邀请人不为空，邀请人id：[{}]", userId, user.getInviteUserId());
        InviteRecord inviteRecord = inviteRecordService.getInviteRecordByInvitedUser(user.getInviteUserId(), userId);
        if (inviteRecord == null) {
            logger.warn("用户[{}]与邀请人[{}]的关系无效", userId, user.getInviteUserId());
        }

        if (!InviteRecordStateEnum.LOGIN_APP.getName().equals(inviteRecord.getInviteState())) {
            logger.info("当前邀请关系不是处于还未成功提现状态----不进行更新操作");
            return;
        }

        logger.info("邀请人[{}]增加可提现赏金[{}]", user.getInviteUserId(), inviteRecord.getAward());
        inviteRecord.setInviteState(InviteRecordStateEnum.SUCCESS_WITHDRAW.getName());
        inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
        updateInviteRecord(inviteRecord);

        //转换邀请人金额
        AccountCashParam accountCashParam = new AccountCashParam(inviteRecord.getUserId(), CashEnum.DRAWABLEL_CASH.name(), Boolean.FALSE,
                inviteRecord.getAward(), inviteRecord.getId(), CashFlowTypeEnum.PROMOTE_BOUNTY.name(),
                CashFlowConvertTypeEnum.STABILIZE_TO_DRAWABLE.name());
        transferCash(accountCashParam);
    }

    /**
     * 更新邀请关系
     *
     * @param inviteRecord
     */
    private void updateInviteRecord(InviteRecord inviteRecord) {
        inviteRecordService.updateInviteRecord(inviteRecord);
    }

    /**
     * 更新用户可提现赏金转为已提现赏金
     *
     * @param userId
     * @param amount
     */
    private void updateUserCash(Long userId, BigDecimal amount) {
        userAccountMapper.addUserWithdrawalCash(userId, amount);
    }

    private KeyGenerator getUserWithdrawPop(Long userId) {
        return RedisConfig.USER_WITHDRAW_POP.copy().appendKey(userId);
    }
}
