package com.bxm.localnews.user.service.impl;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.*;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.config.BizConfigProperties;
import com.bxm.localnews.user.config.UserProperties;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserAuthMapper;
import com.bxm.localnews.user.domain.UserLocationHistoryMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.LocationDTO;
import com.bxm.localnews.user.dto.LocationUserInfoDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.InviteTypeEnum;
import com.bxm.localnews.user.enums.LevelEunm;
import com.bxm.localnews.user.enums.TaskEnum;
import com.bxm.localnews.user.integration.*;
import com.bxm.localnews.user.param.PushParam;
import com.bxm.localnews.user.param.TempUserParam;
import com.bxm.localnews.user.param.UserParam;
import com.bxm.localnews.user.service.*;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.*;
import com.bxm.newidea.component.vo.Message;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import static com.bxm.localnews.user.enums.TaskEnum.TASK_IMPROVE_INFO;

@Service("userService")
@RefreshScope
public class UserServiceImpl extends BaseService implements UserService, UserInternalService {

    private final static Logger log = LoggerFactory.getLogger(UserServiceImpl.class);

    private final static long MAX_HEAD_IMAGE_SIZE = 1024 * 1024 * 5;

    private static final int AREA_CODE_LENGTH = 12;

    @Resource
    private RecommendIntegrationService recommendIntegrationService;

    @Resource
    private SmsIntegrationService smsIntegrationService;

    @Resource
    private UserAuthService userAuthService;

    @Resource
    private NoviceTaskRecordService noviceTaskRecordService;

    @Resource
    private PayFlowService payFlowService;

    @Resource
    private UserAccountService userAccountService;

    @Resource
    private UserMapper userMapper;

    @Resource
    private UserAuthMapper userAuthMapper;

    @Resource
    private UserProperties userProperties;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private UserLocationHistoryMapper userLocationHistoryMapper;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private AliyunOSSService aliyunOSSService;

    @Resource
    private BizConfigProperties bizConfigProperties;

    @Resource
    private ChannelService channelService;

    @Resource
    private MissionIntegrationService missionIntegrationService;

    @Resource
    private BizLogIntegrationService bizLogIntegrationService;

    @Resource
    private UserLoginHistoryService userLoginHistoryService;

    @Resource
    private UserNewsIntegrationService userNewsIntegrationService;

    @Resource
    private UserSyncService userSyncService;

    @Resource
    private InviteRecordService inviteRecordService;

    @Autowired
    private DistributedLock distributedLock;

    @Autowired
    private AppVersionIntegrationService appVersionIntegrationService;

    @Override
    public UserInfoDTO getUserFromRedisDB(long userId) {
        UserInfoDTO dto = this.getUserRedis(userId);
        if (null == dto) {
            dto = this.loadUserToRedis(userId);
        }
        return dto;
    }

    /**
     * 从redis中获取用户信息
     *
     * @param userId
     * @return
     */
    private UserInfoDTO getUserRedis(long userId) {
        logger.debug("从redis获取用户[{}]数据", userId);
        String userStr = this.redisHashMapAdapter.get(RedisConfig.USER_INFO, userId + "", String.class);
        return JSON.parseObject(userStr, UserInfoDTO.class);
    }

    /**
     * 从数据库中查找用户信息，并更新至redis中
     *
     * @param userId
     * @return
     */
    private UserInfoDTO loadUserToRedis(Long userId) {
        User user = this.userMapper.selectByPrimaryKey(userId);
        if (user != null) {
            UserInfoDTO dto = this.getUserToDTO(user);
            this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));
            return dto;
        }
        return null;
    }

    @Override
    public boolean checkUserExistByPhone(String phone, Integer flag) {
        User user = userMapper.findByPhone(phone);
        if (null == user) {
            return false;
        } else {
            UserAuth userAuth = userAuthMapper.selectByUserId((byte) 3, user.getId());
            if (flag == 1) {
                if (userAuth == null) {
                    return false;
                } else {
                    return true;
                }
            } else {
                return true;
            }
        }
    }

    @Override
    public UserInfoDTO getUserToDTO(User user) {
        UserInfoDTO dto = new UserInfoDTO();
        BeanUtils.copyProperties(user, dto);
        if (StringUtils.isBlank(dto.getNickname())) {
            dto.setNickname(StringUtils.hideMobile(user.getPhone()));
        }
        if (StringUtils.isBlank(dto.getHeadImg())) {
            dto.setHeadImg(userProperties.getDefaultHeadImgUrl());
        }
        return dto;
    }

    @Override
    public String getToken() {
        long currentTime = System.currentTimeMillis();
        Random random = new Random();
        String appkey = userProperties.getAppKey();
        return MD5Util.hgmd5(currentTime + appkey + random.nextInt(1000));
    }

    @Override
    public Long getExpireTime() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, 30);
        return calendar.getTime().getTime();
    }

    @Override
    public Json<UserInfoDTO> refreshToken(Long userId, String refreshToken) {
        User user = this.userMapper.getUserByRefreshToken(userId, refreshToken);
        if (null == user) {
            return ResultUtil.genFailedResult(RespCode.UNAUTHORIZED, "登录失效");
        }
        user.setExpiretime(this.getExpireTime());
        user.setToken(this.getToken());
        this.userMapper.updateByPrimaryKeySelective(user);
        UserInfoDTO dto = this.getUserToDTO(user);
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));
        return ResultUtil.genSuccessResult(dto);
    }

    @Override
    public Json<LoginMeta> register(LoginInfo loginInfo, BasicParam basicParam, String firstOpenType) {
        if (StringUtils.isEmpty(loginInfo.getEquipment())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "缺少参数");
        }
        // 根据设备号查询
        User user = this.userMapper.findByEquipment(loginInfo.getEquipment());

        if (null == user) {
            //如果为空，则创建一条游客用户
            user = this.addUser(loginInfo, basicParam.getPlatform(), basicParam.getChnl(), AppConst.USER_STATE.VISITOR);
            //初始化用户默认分类
            this.logger.debug(LogMarker.BIZ, "用户静默注册生成默认频道列表，loginInfo:[{}]", loginInfo);
            userNewsIntegrationService.initUserNewsImage(user.getId(), null);

            //保存用户最后登录信息
            userLoginHistoryService.save(user, loginInfo);
        }
        UserInfoDTO dto = this.getUserToDTO(user);
        //缓存redis
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, user.getId().toString(), JSON.toJSONString(dto));
        return ResultUtil.genSuccessResult(new LoginMeta(dto));
    }

    @Override
    public Json<User> doH5Register(LoginInfo loginInfo, Integer platform, String chnl) {
        if (!checkH5RegisterParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "缺少参数");
        }

        if (!checkSmsCode(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "请输入正确的验证码");
        }

        String messageByCheckAuth = checkBindExist(loginInfo);
        if (StringUtils.isNotBlank(messageByCheckAuth)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth);
        }

        User user = this.userMapper.findByPhone(loginInfo.getLoginName());

        if (null != user) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "手机号已注册");
        }

        user = addUser(loginInfo, platform, chnl, AppConst.USER_STATE.NORMAL);

        if (user.getId() > 0) {
//            if (null != loginInfo.getInviteCode()) {
//                UserInfoDTO dto = this.getUserFromRedisDB(loginInfo.getInviteCode());
//                if (null != dto) {
//                    this.inviteRelationIntegrationService.addInviteRelation(user.getId(), loginInfo.getInviteCode());
//                }
//            }

            //初始化用户标签
            this.logger.debug(LogMarker.BIZ, "用户H5注册，loginInfo:[{}]", loginInfo);
            userNewsIntegrationService.initUserNewsImage(user.getId(), null);
            return ResultUtil.genSuccessResult(user);
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "注册失败");
    }

    /**
     * 检查验证码是否合理
     *
     * @param loginInfo
     * @return
     */
    private boolean checkSmsCode(LoginInfo loginInfo) {
        if (!this.smsIntegrationService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_REGISTER, loginInfo.getLoginName(), loginInfo.getCode())) {
            this.logger.debug("checkVcode failed,loginInfo[{}]", loginInfo);
            return false;
        }
        return true;
    }

    /**
     * 检查H5注册参数
     *
     * @param loginInfo
     * @return
     */
    private boolean checkH5RegisterParam(LoginInfo loginInfo) {
        if (StringUtils.isBlank(loginInfo.getLoginName())
                || AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                || StringUtils.isBlank(loginInfo.getCode())) {
            return false;
        }
        return true;
    }

    @Override
    public User addUser(LoginInfo loginInfo, Integer platform, String chnl, Byte userType) {
        //新增用户信息
        User user = this.saveUser(loginInfo, platform, chnl, userType);
        boolean notVisitor = AppConst.USER_STATE.VISITOR != userType;
        //判断用户是否是游客
        if (notVisitor) {
            //添加授权登录认证信息
            this.saveAuths(user.getId(), loginInfo);
            //生成计划新手计划
            this.noviceTaskRecordService.batchAdd(user.getId());
            //生成账户(外部模块)
            this.userAccountService.createUserAccount(user.getId());
            //如果是邀请注册,则添加注册来源
            this.saveRegSourceInfo(loginInfo, user, chnl);
            bizLogIntegrationService.newUser(user.getId(), chnl, loginInfo.getLoginName(), BigDecimal.ZERO, "0");
        }

        return user;
    }

    /**
     * 生成用户并新增入库
     *
     * @param loginInfo
     * @param platform
     * @param chnl
     * @param userType
     * @return
     */
    private User saveUser(LoginInfo loginInfo, Integer platform, String chnl, Byte userType) {
        Date date = new Date();
        User user = new User();
        user.setToken(this.getToken());
        user.setRefreshtoken(this.getToken());
        user.setRegisterClient(null == platform ? "0" : String.valueOf(platform));
        //不维护年龄字段，因为这个字段每年都在变
        user.setAge(0);
        user.setSex((byte) 1);
        user.setPhone(loginInfo.getPhone());
        user.setEquipment(loginInfo.getEquipment());
        user.setRegisteredAddress(loginInfo.getRegisteredaddress());
        user.setPhoneModel(loginInfo.getPhonemodel());
        user.setState(userType);
        user.setLevel(LevelEunm.INITIAL_USER.getType());
        user.setIsNew(AppConst.IS_NEW_USER.NEW_USER);
        user.setExpiretime(this.getExpireTime());
        user.setLastLoginTime(date);
        user.setIsTempNickName((byte) 1);
        user.setIsTempHeadImg((byte) 1);
        if (null != loginInfo.getRegIp()) {
            user.setRegIp(loginInfo.getRegIp());
        }
        if (null != loginInfo.getLastLoginIp()) {
            user.setLastLoginIp(loginInfo.getLastLoginIp());
        }

        user.setLocationCode(bizConfigProperties.getLocationCode());
        user.setLocationName(bizConfigProperties.getLocationName());
        if (null != loginInfo.getId()) {
            User originalUser = userMapper.selectByPrimaryKey(loginInfo.getId());
            if (null != originalUser && null != originalUser.getLocationCode() && null != originalUser.getLocationName()) {
                user.setLocationCode(originalUser.getLocationCode());
                user.setLocationName(originalUser.getLocationName());
            }
        }

        user.setChannelId(this.getChannelId(chnl));
        this.userMapper.insertSelective(user);

        if (AppConst.USER_STATE.VISITOR != userType) {
            User loginUser = new User();
            loginUser.setId(user.getId());
            this.setLoginName(loginUser, loginInfo);
            this.userMapper.updateByPrimaryKeySelective(loginUser);
        }

        return user;
    }

    /**
     * 添加授权登录认证信息
     *
     * @param userId
     * @param loginInfo
     */
    private void saveAuths(Long userId, LoginInfo loginInfo) {
        List<UserAuth> auths = new ArrayList<>();

        // 1.单纯手机号注册 2.手机号+微信注册 3.微信号注册
        UserAuth auth = new UserAuth();
        auth.setUserId(userId);
        auth.setType(loginInfo.getType());
        auth.setIdentifier(loginInfo.getLoginName());
        if (StringUtils.isNotBlank(loginInfo.getPassword())) {
            auth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
        }
        auths.add(auth);
        logger.warn("添加权限验证:{}", loginInfo.getLoginName());

        //如果三方注册
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()) {
            //绑定三方的时候绑定手机
            if (null != loginInfo.getPhone()) {
                UserAuth phoneAuth = new UserAuth();
                phoneAuth.setUserId(userId);
                phoneAuth.setType(AppConst.LOGIN_TYPE.PHONE);
                phoneAuth.setIdentifier(loginInfo.getPhone());
                if (StringUtils.isNotBlank(loginInfo.getPassword())) {
                    phoneAuth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
                }
                auths.add(phoneAuth);
            }

            //绑定三方的时候绑定openid
            if (loginInfo.getOpenId() != null) {
                UserAuth openIdAuth = new UserAuth();
                openIdAuth.setUserId(userId);
                openIdAuth.setType(AppConst.LOGIN_TYPE.APP_OPENID);
                openIdAuth.setIdentifier(loginInfo.getOpenId());
                if (StringUtils.isNotBlank(loginInfo.getPassword())) {
                    openIdAuth.setCredential(MD5Util.hgmd5(loginInfo.getPassword()));
                }
                auths.add(openIdAuth);
            }


        }

        this.userAuthMapper.batchAdd(auths);
    }

    /**
     * 保存用户注册来源信息
     *
     * @param loginInfo 注册信息
     * @param user      用户信息
     * @param chnl      注册渠道
     */
    private void saveRegSourceInfo(LoginInfo loginInfo, User user, String chnl) {
        Long inviteCode = loginInfo.getInviteCode();
        if (null == inviteCode) {
            return;
        }

        this.logger.debug(LogMarker.BIZ, "用户邀请记录,loginInfo:[{}],user:[{}],chnl:[{}]", loginInfo, user, chnl);

        //新增用户注册来源记录入库
        this.saveRegSourceCount(loginInfo, user, chnl);

        bizLogIntegrationService.inviteSuccessed(inviteCode, user.getId());
        missionIntegrationService.completeTask(inviteCode, TaskEnum.TASK_INVITED_FRIEND, user.getId().toString());
    }

    /**
     * 统一调用missionService提供的方法
     */
    private void statisticsGold(Long inviteCode) {
        //统计当天完成任务获得的金币总数
        KeyGenerator userRewardSumKey = RedisConfig.USER_MISSION_REWARD_SUM_PER_DAY.copy()
                .setKey(inviteCode + ":" + DateUtils.formatDate(new Date()));
        redisStringAdapter.incrementWithDefault(userRewardSumKey, AppConst.INVITE_NEW_USER_REWARD_GOLD_NUM.longValue(),
                AppConst.INVITE_NEW_USER_REWARD_GOLD_NUM.intValue());
        redisStringAdapter.expire(userRewardSumKey, DateUtils.getCurSeconds());
    }

    /**
     * 统一调用missionService提供的方法
     */
    private void generatorInvitationReward(Long userId, Long inviteCode) {
        PayFlow payFlow = PayFlow.initPayFlow(inviteCode, TaskEnum.TASK_INVITED_FRIEND.getType(),
                AppConst.REWARD_TYPE_GOLD, userId.toString());
        this.payFlowService.modifyAccountFlowAndStatByGold(payFlow, AppConst.INVITE_NEW_USER_REWARD_GOLD_NUM,
                false);
    }

    private void saveRegSourceCount(LoginInfo loginInfo, User user, String chnl) {
        RegSourceCounter counter = new RegSourceCounter();
        counter.setId(this.nextSequence().toString());
        counter.setRegTime(new Date());
        counter.setUserId(user.getId());
        //判断用户渠道
        counter.setChannel(this.getChannelId(chnl));
        counter.setInviteUserId(loginInfo.getInviteCode());
//        this.regSourceMapper.insert(counter);
    }

    /**
     * 获取渠道
     *
     * @param chnl
     * @return
     */
    private Long getChannelId(String chnl) {
        //判断用户渠道
        return channelService.getChannelByCode(chnl).getId();
    }

    /**
     * 设置登陆名称
     *
     * @param user
     * @param loginInfo
     */
    private void setLoginName(User user, LoginInfo loginInfo) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user.setPhone(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
            user.setQq(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
            user.setWeibo(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user.setWeixin(loginInfo.getLoginName());
            user.setNickname(user.getNickname() == null ? loginInfo.getNickname() : user.getNickname());
            user.setHeadImg(user.getHeadImg() == null ? getImageFromUrl(loginInfo.getHeadImg(), user.getId()) : user.getHeadImg());
        } else {
            throw new RuntimeException("参数错误");
        }
    }

    /**
     * 根据微信的url保存在oss中并获得图片地址
     *
     * @param headImg
     * @param userId
     * @return
     */
    private String getImageFromUrl(String headImg, Long userId) {
        try {
            URL url = new URL(headImg);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();

            // 设置连接超时时间
            conn.setConnectTimeout(3000);
            InputStream inputStream = null;

            // 正常响应时获取输入流, 在这里也就是图片对应的字节流
            if (conn.getResponseCode() == 200) {
                inputStream = conn.getInputStream();
                logger.debug("根据url获得图片的流状态：{}", inputStream.available());
            }
            int hashCode = (int) (userId % 10);
            String path = "avatar/" + hashCode + "/" + UUIDUtils.nextID() + ".jpeg";
            return aliyunOSSService.upload(inputStream, path) + "?x-oss-process=style/head";
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public Json<UserInfoDTO> binding(LoginInfo loginInfo, Long userId) {
        if (!this.bindingCheckParam(loginInfo)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "缺少必要参数");
        }

        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && !this.smsIntegrationService.verifySmsCodeByType(AppConst.SMS_TPL.VCODE_TYPE_BIND_PHONE, loginInfo.getLoginName(), loginInfo.getCode())) {
            return ResultUtil.genFailedResult("请输入正确的验证码");
        }

        String messageByCheckAuth = checkBindExist(loginInfo);
        if (StringUtils.isNotBlank(messageByCheckAuth)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckAuth);
        }

        User user = this.userMapper.selectByPrimaryKey(userId);

        String messageByCheckUser = checkBindRepeat(loginInfo.getType(), user);
        if (StringUtils.isNotBlank(messageByCheckUser)) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, messageByCheckUser);
        }

        this.setLoginName(user, loginInfo);
        //更新用户信息(如果有旧的头像和旧的昵称存在，则不用微信的头像和昵称覆盖)
        this.userMapper.updateByPrimaryKeySelective(user);
        //添加登录认证
        this.userAuthService.addUserAuth(loginInfo.getType(), user.getId(), loginInfo.getLoginName(), null);

        UserInfoDTO dto = this.getUserToDTO(user);
        checkCompleteInfo(dto);
        logger.debug("把用户[{}]数据[{}]存入redis", userId, JSON.toJSONString(dto));
        this.redisHashMapAdapter.put(RedisConfig.USER_INFO, user.getId() + "", JSON.toJSONString(dto));

        return ResultUtil.genSuccessResult(dto);
    }

    /**
     * 检测微信号（第三方）是否已被其他账号绑定
     *
     * @param loginInfo
     * @return
     */
    private String checkBindExist(LoginInfo loginInfo) {
        UserAuth auth = this.userAuthMapper.selectByUserAuth(loginInfo.getType(), loginInfo.getLoginName());
        String message = "";
        if (null != auth) {
            if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
                message = "手机号已被绑定";
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
                message = "微信号已被绑定";
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
                message = "微博已被绑定";
            } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
                message = "QQ号已被绑定";
            }
        }
        return message;
    }

    /**
     * 只检查了当前要绑定第三方的用户有没有重复绑定，但是忽视了别的用户也可能已经绑定了第三方
     *
     * @param loginInfoType
     * @param user
     * @return
     */
    private String checkBindRepeat(Byte loginInfoType, User user) {
        String message = "";
        if (AppConst.LOGIN_TYPE.PHONE == loginInfoType && StringUtils.isNotBlank(user.getPhone())) {
            message = "请勿重复绑定手机号";
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfoType && StringUtils.isNotBlank(user.getWeixin())) {
            message = "请勿重复绑定微信账号";
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfoType && StringUtils.isNotBlank(user.getWeibo())) {
            message = "请勿重复绑定微博账号";
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfoType && StringUtils.isNotBlank(user.getQq())) {
            message = "请勿重复绑定QQ账号";
        }
        return message;
    }

    @Override
    public Message updateUserInfo(UserParam updateUser, Long userId, MultipartFile file) {
        User user = this.userMapper.selectByPrimaryKey(userId);
        if (1 != user.getState()) {
            return Message.build(false, "用户尚未登录");
        }
        if (null != updateUser.getNickname()) {
            if (null == user.getIsTempNickName() || user.getIsTempNickName() == 1) {
                user.setNickname(updateUser.getNickname());
                user.setIsTempNickName((byte) 0);
            } else {
                return Message.build(false, "该用户修改昵称已达上限");
            }
        }
        if (null != updateUser.getSex()) {
            user.setSex(updateUser.getSex());

            bizLogIntegrationService.changeGender(user.getId(), updateUser.getSex());
        }
        if (null != updateUser.getAddress()) {
            user.setAddress(updateUser.getAddress());
        }
        if (null != updateUser.getBirthday()) {
            user.setBirthday(updateUser.getBirthday());
        }
        String headImg = null;
        if (null != file) {
            if (file.getSize() > MAX_HEAD_IMAGE_SIZE) {
                return Message.build(false, "头像图片过大,请重新上传");
            }
            headImg = upLoadHeadImage(userId, file);
            if (null != headImg) {
                user.setHeadImg(upLoadHeadImage(userId, file));
                user.setIsTempHeadImg((byte) 0);
            }
        }
        int i = this.userMapper.updateByPrimaryKeySelective(user);
        if (i > 0) {
            UserInfoDTO dto = this.getUserToDTO(user);

            //将变更的用户信息同步到冗余的表设计字段中
            userSyncService.sync(user);
            checkCompleteInfo(dto);
            logger.debug("完善个人资料:[{}]", JSON.toJSONString(dto));
            this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));
            return Message.build(true).addParam("user", dto);
        }
        return Message.build(false, "更新失败");
    }

    /**
     * 检查用户是否已经完善资料，如果已经完善则完成任务
     * 地区编码位数问题或用户id为空
     *
     * @param user
     */
    private void checkCompleteInfo(UserInfoDTO user) {
        if (null != user.getWeixin()
                && null != user.getBirthday()
                && StringUtils.isNotEmpty(user.getAddress())
                && StringUtils.isNotEmpty(user.getHeadImg())
                && StringUtils.isNotEmpty(user.getNickname())) {
            logger.debug("用户信息:[{}]", JSON.toJSONString(user));
            missionIntegrationService.completeTask(user.getId(), TASK_IMPROVE_INFO, null);
        }

    }

    @Override
    public LocationUserInfoDTO getLocationUserInfo(Long userId) {
        return userMapper.getLocationUserInfo(userId);
    }

    @Override
    public String getDefaultHeadImgUrl() {
        return userProperties.getDefaultHeadImgUrl();
    }

    @Override
    public User selectByPrimaryKey(Long userId) {
        return this.userMapper.selectByPrimaryKey(userId);
    }

    @Override
    public Boolean checkUnionIdIsUser(String unionId) {
        UserAuth userAuth = userAuthMapper.selectByUserAuth((byte) 3, unionId);
        if (userAuth == null) {
            return false;
        }

        User user = userMapper.selectByPrimaryKey(userAuth.getUserId());
        if (user != null && user.getState() == AppConst.USER_STATE.NOT_ACTIVE) {
            return false;
        }

        return true;
    }

    @Override
    public User selectUserByUnionId(String unionId) {
        return this.userMapper.findByWeixin(unionId);
    }

    @Override
    public User selectUserByEquipment(String equipment) {
        return this.userMapper.findByEquipment(equipment);
    }

    @Override
    public Long checkExistEquipment(String equipment) {
        return this.userMapper.checkExistEquipment(equipment);
    }

    @Override
    public Boolean updateUserRecieveRedPacket(Long userId) {
        this.userMapper.updateUserReceiveRedPacket(userId);
        return Boolean.TRUE;
    }

    @Override
    public Boolean isRiskUser(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        return user.getRisk() == AppConst.USER_RISK.IS_RISK;
    }

    @Override
    public Boolean addPostReplyNum(Long userId, Integer type) {
        if (1 == type) {
            userMapper.addPostNum(userId);
        } else if (2 == type) {
            userMapper.minusPostNum(userId);
        } else if (3 == type) {
            userMapper.addReplyNum(userId);
        } else if (4 == type) {
            userMapper.minusReplyNum(userId);
        }

        return true;
    }

    @Override
    public Boolean isTempUser(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        return AppConst.USER_STATE.NOT_ACTIVE == user.getState();
    }

    @Override
    public Message createUser(TempUserParam userParam) {
        Message message = Message.build();
        if (null == userParam.getUnionId()) {
            return message;
        }

        logger.debug("开始创建临时用户：{}", JSON.toJSONString(userParam));
        User user = convertUser(userParam);
        UserAuth unionUserAuth = null;
        Long userId = null;
        Boolean isInsertInviteRecord = false;

        unionUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN, userParam.getUnionId());
        //是否已经存在该用户
        if (null == unionUserAuth || null == unionUserAuth.getUserId()) {
            //分布式锁保证不会重复插入
            if (distributedLock.lock(userParam.getUnionId(), userParam.getUnionId(), 1, TimeUnit.SECONDS)) {
                userMapper.insertSelective(user);
                userAccountService.createUserAccount(user.getId());
                userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN, user.getId(), userParam.getUnionId(), null);
                userId = user.getId();
                isInsertInviteRecord = true;
            }
        } else {
            userId = unionUserAuth.getUserId();
            User existedUser = userMapper.selectByUserId(userId);
            if (null != existedUser && existedUser.getState() == 2 && null == existedUser.getInviteUserId() && user.getInviteUserId() != null) {
                isInsertInviteRecord = true;
                user.setId(userId);
                user.setState(existedUser.getState());
                userMapper.updateByPrimaryKeySelective(user);
            }
        }

        if (null != userParam.getOpenId()) {
            if (InviteTypeEnum.APPLETDRAW.name().equals(userParam.getInviteType())) {
                UserAuth appletUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_OPENID, userParam.getOpenId());
                if (null == appletUserAuth || null == appletUserAuth.getUserId()) {
                    userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_OPENID, userId, userParam.getOpenId(), null);
                }
            } else {
                UserAuth weixinUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN_OPENID, userParam.getOpenId());
                if (null == weixinUserAuth || null == weixinUserAuth.getUserId()) {
                    userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN_OPENID, userId, userParam.getOpenId(), null);
                }
            }
        }

        if (userParam.getInviteUserId() != null && !userParam.getInviteUserId().equals(userId) && isInsertInviteRecord) {
            inviteRecordService.addInviteRecord(userParam.getInviteUserId(), userId, userParam.getInviteType());
        }

        return message.addParam("userId", userId);
    }

    /**
     * 创建临时用户
     *
     * @param tempUserParam
     * @return
     */
    private User convertUser(TempUserParam tempUserParam) {
        User user = new User();
        user.setWeixin(tempUserParam.getUnionId());
        user.setNickname(tempUserParam.getNickName());
        user.setHeadImg(tempUserParam.getHeadImg());
        user.setSex(tempUserParam.getSex());
        user.setRegIp(tempUserParam.getRegIp());
        user.setRegisterClient(tempUserParam.getPlatform());
        user.setRegisteredAddress(tempUserParam.getRegisteredAddress());
        user.setInviteUserId(tempUserParam.getInviteUserId());
        User inviteUser = userMapper.selectByPrimaryKey(tempUserParam.getInviteUserId());
        if (inviteUser == null) {
            logger.info("受邀用户[{}]的邀请人不存在", user.getId());
            user.setInviteLevel(1);
        } else {
            logger.info("创建邀请用户[{}]的受邀用户", inviteUser.getId());
            user.setInviteLevel(inviteUser.getInviteLevel() + 1);
        }
        user.setRegisterChannel(tempUserParam.getInviteType());
        //临时用户
        user.setState((byte) 2);
        user.setLevel(LevelEunm.INITIAL_USER.getType());
        user.setIsNew(AppConst.IS_NEW_USER.NEW_USER);
        user.setExpiretime(this.getExpireTime());
        Date now = new Date();
        user.setLastLoginTime(now);
        user.setIsTempNickName((byte) 1);
        user.setIsTempHeadImg((byte) 1);

        return user;
    }

    /**
     * 上传头像图片到oss，路径为avator/对用户进行10取模/
     *
     * @param userId
     * @param file
     * @return
     */
    private String upLoadHeadImage(Long userId, MultipartFile file) {
        int hashCode = (int) (userId % 10);
        String url = aliyunOSSService.upload(file, "avatar/" + hashCode + "/");
        if (!StringUtils.containsAny(url, "?")) {
            //添加oss的压缩样式
            url += "?x-oss-process=style/head";
        }
        return url;
    }

    /**
     * 账号绑定参数必要校验
     *
     * @param loginInfo
     * @return
     */
    private boolean bindingCheckParam(LoginInfo loginInfo) {
        //校验绑定类型
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
                && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }
        //登录标识不能为空
        if (StringUtils.isBlank(loginInfo.getLoginName())) {
            return false;
        }
        //绑定手机号标识
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
                && StringUtils.isEmpty(loginInfo.getCode())) {
            return false;
        }
        //绑定微信，则微信头像和昵称字段不能为空
        return AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
                || !StringUtils.isEmpty(loginInfo.getHeadImg())
                || !StringUtils.isEmpty(loginInfo.getNickname());
    }

    @Override
    public LocationDTO doPostingLocation(String code, Long userId, BasicParam basicParam) {
        if (StringUtils.isEmpty(code)) {
            logger.error("定位时无法提供地区编码(定位失败)!!!:code:{},userId:{}", code, userId);
        }
        if (StringUtils.isNotEmpty(code) && ((code.length() != 6 && code.length() != 12)) || null == userId) {
            logger.error("参数错误(地区编码位数问题或用户id为空):code:{},userId:{}", code, userId);
        }
        if (null == code) {
            code = bizConfigProperties.getLocationCode();
        }
        LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(code);

        if (appVersionIntegrationService.getPublishState(basicParam)) {
            locationDTO.setEnablePaidPromote((byte) 0);
            locationDTO.setEnableCommunityContent(0);
            locationDTO.setEnableIndexPublish(0);
        }
        UserInternalService userInternalService = SpringContextHolder.getBean(UserInternalService.class);
        userInternalService.updateUserLocation(locationDTO, userId, code);

        return locationDTO;
    }

    @Async
    @Override
    public void updateUserLocation(LocationDTO locationDTO, Long userId, String code) {
        //如果没有传用户id则不需要更新用户以及新增地理位置历史
        if (null != userId) {
            User user = this.userMapper.selectByPrimaryKey(userId);
            //测试环境在删除用户时，客户端还是有用户id的，当客户端传用户id时服务端查不到用户的信息会报服务器错误
            if (null != user) {
                user.setLocationCode(locationDTO.getCode());
                user.setLocationName(locationDTO.getName());
                //1.更新用户地理位置
                int i = this.userMapper.updateByPrimaryKeySelective(user);
                if (i > 0) {

                    //2.埋点日志
                    bizLogIntegrationService.changeArea(userId, code);

                    //3.更新用户信息
                    UserInfoDTO dto = this.getUserToDTO(user);
                    this.redisHashMapAdapter.put(RedisConfig.USER_INFO, userId + "", JSON.toJSONString(dto));

                    //4.清空用户首页推荐
                    UserLocationHistory lastUserLocation = userLocationHistoryMapper.getLastLocationByUser(userId);
                    if (lastUserLocation == null || !lastUserLocation.getLocationCode().equals(locationDTO.getCode())) {
                        logger.debug("[updateUserLocation]更新用户的地理位置，如果历史定位为空，或者与历史定位不一致，删除用户混合推荐缓存,原定位信息:{},现定位信息:{}",
                                JSONObject.toJSONString(lastUserLocation), locationDTO.getCode());
                        recommendIntegrationService.cleanMixRecommendCache(userId);
                    }

                    //5.位置历史表新增
                    UserLocationHistory userLocationHistory = new UserLocationHistory();
                    userLocationHistory.setAddTime(new Date());
                    userLocationHistory.setId(nextSequence());
                    userLocationHistory.setLocationCode(locationDTO.getCode());
                    userLocationHistory.setUserId(userId);
                    userLocationHistoryMapper.insert(userLocationHistory);
                }
            }
        }
    }

    @Override
    public void savePush(PushParam pushParam) {
        User userForPush = userMapper.findByPushToken(pushParam.getPushToken());
        if (null != userForPush) {
            userMapper.clearPushInfo(userForPush.getId());
        }
        User user = userMapper.selectByUserId(pushParam.getUserId());
        if (null != user) {
            PushInfo pushInfo = new PushInfo();
            BeanUtils.copyProperties(pushParam, pushInfo);
            userMapper.savePushInfo(pushInfo);
        }
    }
}
