package com.bxm.localnews.user.account.impl;

import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.account.impl.callback.DefaultAccountActionCallback;
import com.bxm.localnews.user.account.impl.callback.IAccountActionCallback;
import com.bxm.localnews.user.account.impl.context.AccountActionContext;
import com.bxm.localnews.user.account.impl.handler.DefaultAccountActionHandler;
import com.bxm.localnews.user.account.impl.handler.IAccountActionHandler;
import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.exception.UserAccountException;
import com.bxm.localnews.user.param.AccountActionParam;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;

/**
 * 账号处理代理类 封装原始请求，匹配对应的处理器 如非重大改版，此类不需要进行额外的处理.扩展流程如下：
 * <ol>
 * <li>定义账号处理动作 {@link AccountActionEnum}
 * <li>定义处理动作对应的Handler {@link IAccountActionHandler}
 * <li>给Handler绑定对应的处理规则 {@link com.bxm.localnews.user.account.impl.rule.IRule}
 * <li>如存在流程之外的逻辑，提供callback {@link IAccountActionCallback}
 * </ol>
 * @author liujia
 * @date 2020/05/04 11:08
 */
@Component
@Slf4j
@Transactional(rollbackFor = Exception.class)
public class AccountHandlerProxy {

    private UserAccountService userAccountService;

    private Map<AccountActionEnum, IAccountActionHandler> handlerMap;

    private IAccountActionHandler defaultHandler = new DefaultAccountActionHandler();

    public UserAccountService getUserAccountService() {
        if (userAccountService == null) {
            userAccountService = SpringContextHolder.getBean(UserAccountService.class);
        }
        return userAccountService;
    }

    @Autowired
    public AccountHandlerProxy(List<IAccountActionHandler> handlerList) {
        handlerMap = Maps.newHashMap();

        if (!CollectionUtils.isEmpty(handlerList)) {
            handlerList.forEach(handler -> handlerMap.put(handler.support(), handler));
        }
    }

    @Retryable(value = RetryException.class)
    public void handle(AccountActionParam param) {
        handler(param, null);
    }

    @Retryable(value = RetryException.class)
    public void handler(AccountActionParam param, IAccountActionCallback callback) {
        log.debug("进行账号操作，请求参数：{}", param);

        //封装上下文
        AccountActionContext context = new AccountActionContext(param);
        if (null == callback) {
            callback = new DefaultAccountActionCallback();
        }
        context.setCallback(callback);
        context.setAccount(getUserAccountService().getUserAccountDetail(context.getUserId()));

        try {
            //查找匹配的处理类
            IAccountActionHandler handler = handlerMap.getOrDefault(context.getAction(), defaultHandler);
            log.debug("账号操作的Handler：{}", handler.getClass().getSimpleName());

            handler.handle(context);
        } catch (UserAccountException e) {
            log.warn("处理账号发生业务异常，请求参数：{}，异常信息：{}:是否重试：{}",
                    context,
                    e.getMessage(),
                    e.isRetryable());

            //如果可以重试则抛出异常，触发重试
            if (e.isRetryable()) {
                callback.preRetry(context);
                throw new RetryException(e.getMessage(), e);
            } else {
                callback.exception(context);
                throw e;
            }
        } catch (Exception e) {
            callback.exception(context);
            log.error("账号处理发生未预期的错误,请求参数：{}", context);
            log.error(e.getMessage(), e);

            throw e;
        }

        log.debug("账号操作处理完成，请求参数：{}", param);
    }
}
