package com.bxm.localnews.user.account.impl;

import com.bxm.localnews.user.account.CashFlowService;
import com.bxm.localnews.user.account.UserWithdrawService;
import com.bxm.localnews.user.constant.CashFlowClassifyEnum;
import com.bxm.localnews.user.domain.CashFlowMapper;
import com.bxm.localnews.user.dto.AccountCashFlowDTO;
import com.bxm.localnews.user.dto.AccountIncomeTypeDTO;
import com.bxm.localnews.user.dto.CashFlowDTO;
import com.bxm.localnews.user.dto.UserDayCashDTO;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.WithdrawEnum;
import com.bxm.localnews.user.param.AccountCashFlowPageParam;
import com.bxm.localnews.user.param.CashFlowParam;
import com.bxm.localnews.user.vo.CashFlow;
import com.bxm.localnews.user.vo.WithdrawFlow;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.apache.commons.lang3.StringUtils.join;

@Service
@Slf4j
public class CashFlowServiceImpl extends BaseService implements CashFlowService {

    @Resource
    private CashFlowMapper cashFlowMapper;

    @Resource
    private UserWithdrawService userWithdrawService;

    @Resource
    private SequenceCreater sequenceCreater;

    private List<AccountIncomeTypeDTO> typeCache;

    @Override
    public List<UserDayCashDTO> getRecentSevenDayCashFlowByUser(Long userId) {
        return cashFlowMapper.querySevenDayCashFlows(userId);
    }

    @Override
    public List<CashFlowDTO> getUserAllCashFlow(CashFlowParam cashFlowParam) {
        if (StringUtils.isNotEmpty(cashFlowParam.getDate())) {
            Date date = DateUtils.parse(cashFlowParam.getDate());
            Date endDate = DateUtils.addField(date, Calendar.DATE, 1);
            cashFlowParam.setEndDate(DateUtils.formatAtWill(endDate, DateUtils.DATE_FORMAT));
            cashFlowParam.setType("income");
        } else {
            //查询所有的明细需要微信提现失败流水
            cashFlowParam.setType(null);
        }
        List<CashFlow> cashFlowList = cashFlowMapper.queryCashFlows(cashFlowParam);
        return generateCashFlowDTO(cashFlowList);
    }

    /**
     * 转换数据
     */
    private List<CashFlowDTO> generateCashFlowDTO(List<CashFlow> cashFlows) {
        if (CollectionUtils.isEmpty(cashFlows)) {
            return new ArrayList<>();
        }

        return cashFlows.stream().map(this::convertCashFlow).collect(Collectors.toList());
    }

    /**
     * 转换对象
     */
    private CashFlowDTO convertCashFlow(CashFlow cashFlow) {
        CashFlowDTO cashFlowDTO = new CashFlowDTO();
        cashFlow.setAmount(cashFlow.getAmount().setScale(2, BigDecimal.ROUND_HALF_UP));
        String amount;
        if (cashFlow.getAmount().compareTo(BigDecimal.ZERO) > 0) {
            amount = "+" + cashFlow.getAmount();
            cashFlowDTO.setPlus(Boolean.TRUE);
        } else {
            amount = cashFlow.getAmount().toString();
            cashFlowDTO.setPlus(Boolean.FALSE);
        }
        cashFlowDTO.setAmount(amount);
        cashFlowDTO.setCreateTime(cashFlow.getCreateTime());
        CashFlowTypeEnum cashFlowTypeEnum = CashFlowTypeEnum.valueOf(cashFlow.getType());
        cashFlowDTO.setTypeDesc(cashFlowTypeEnum.getDesc());

        if ((Objects.equals(CashFlowTypeEnum.WECHAT_WITHDRAWAL, cashFlowTypeEnum)
                || (Objects.equals(CashFlowTypeEnum.ALIPAY_WITHDRAWAL, cashFlowTypeEnum)))
                && cashFlow.getRelationId() != null) {

            WithdrawFlow withdrawFlow = userWithdrawService.getWithdrawFlowById(cashFlow.getRelationId());
            if (WithdrawEnum.SUCCESS_PAYMENT.getState().equals(withdrawFlow.getState())) {
                cashFlowDTO.setTypeDesc(join(cashFlowTypeEnum.getDesc(), "--已成功"));
            }

            if (WithdrawEnum.AUDIT.getState().equals(withdrawFlow.getState()) ||
                    WithdrawEnum.DELAY_AUDIT.getState().equals(withdrawFlow.getState())) {
                cashFlowDTO.setTypeDesc(join(cashFlowTypeEnum.getDesc(), "--审核中"));
            }

            if (WithdrawEnum.FAIL_PAYMENT.getState().equals(withdrawFlow.getState())) {
                cashFlowDTO.setTypeDesc(join(cashFlowTypeEnum.getDesc(), "-–失败已退回"));
            }
        }
        return cashFlowDTO;
    }

    @Override
    public int addCashFlow(CashFlow cashFlow) {
        logger.debug("新增流水：{}", cashFlow);

        if (null == cashFlow.getId()) {
            cashFlow.setId(sequenceCreater.nextLongId());
        }

        if (null == cashFlow.getCreateTime()) {
            cashFlow.setCreateTime(new Date());
        }

        return cashFlowMapper.addCashFlow(cashFlow);
    }

    @Override
    public BigDecimal getMySelfBounty(Long userId) {
        BigDecimal myBounty = this.cashFlowMapper.getMySelfBounty(userId);
        return myBounty == null ? BigDecimal.ZERO : myBounty;
    }

    @Override
    public BigDecimal getMyRedPacket(Long userId) {
        BigDecimal myRedPacket = this.cashFlowMapper.getMyRedPacket(userId);
        return myRedPacket == null ? BigDecimal.ZERO : myRedPacket;
    }

    @Override
    public boolean enableWithdraw(Long userId) {
        CashFlowParam cashFlowParam = new CashFlowParam();
        cashFlowParam.setUserId(userId);

        List<CashFlow> cashFlowList = cashFlowMapper.queryCashFlows(cashFlowParam);
        if (CollectionUtils.isEmpty(cashFlowList)) {
            log.info("未领取过新用户赏金，用户id：" + userId);
            return false;
        }
        return true;
    }

    @Override
    public PageWarper<AccountCashFlowDTO> queryCashByPage(AccountCashFlowPageParam param) {
        Preconditions.checkArgument(null != param);

        CashFlowClassifyEnum classify = CashFlowClassifyEnum.of(param.getCashFlowCode());
        if (classify == null) {

            classify = CashFlowClassifyEnum.ALL;
        }

        if (null != classify.getInclude()) {
            param.setIncludeTypeList(Stream.of(classify.getInclude())
                    .map(CashFlowTypeEnum::name)
                    .collect(Collectors.toList()));
        }
        if (null != classify.getExclude()) {
            param.setExcludeTypeList(Stream.of(classify.getExclude())
                    .map(CashFlowTypeEnum::name)
                    .collect(Collectors.toList()));
        }

        List<CashFlow> cashFlows = cashFlowMapper.queryCashFlowByPage(param);

        List<AccountCashFlowDTO> result = cashFlows.stream()
                .map(this::convert)
                .collect(Collectors.toList());

        PageWarper warper = new PageWarper(cashFlows);
        warper.setList(result);
        return warper;
    }

    private AccountCashFlowDTO convert(CashFlow cashFlow) {
        AccountCashFlowDTO cashFlowResult = AccountCashFlowDTO.builder()
                .id(cashFlow.getId())
                .amount(cashFlow.getAmount())
                .createTime(cashFlow.getCreateTime())
                .remark(cashFlow.getRemark())
                .plus(cashFlow.getAmount().floatValue() > 0)
                .build();

        if (StringUtils.isBlank(cashFlowResult.getRemark())) {
            if (StringUtils.isNotBlank(cashFlow.getType())) {
                CashFlowTypeEnum type = CashFlowTypeEnum.get(cashFlow.getType());
                cashFlowResult.setRemark(type.getDesc());
            }
        }

        return cashFlowResult;
    }
}
