package com.bxm.localnews.user.attribute.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.UserTokenParam;
import com.bxm.localnews.msg.facade.UserTokenFacadeService;
import com.bxm.localnews.user.attribute.UserAttributeService;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.integration.UserNewsIntegrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.PushParam;
import com.bxm.localnews.user.vo.UserLikeVo;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

@Service
@Slf4j
public class UserAttributeServiceImpl implements UserAttributeService {

    private final UserMapper userMapper;

    private final UserNewsIntegrationService userNewsIntegrationService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserService userService;

    private final UserTokenFacadeService userTokenFacadeService;

    @Autowired
    public UserAttributeServiceImpl(UserMapper userMapper,
                                    UserNewsIntegrationService userNewsIntegrationService,
                                    RedisHashMapAdapter redisHashMapAdapter,
                                    UserService userService,
                                    UserTokenFacadeService userTokenFacadeService) {
        this.userMapper = userMapper;
        this.userNewsIntegrationService = userNewsIntegrationService;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userService = userService;
        this.userTokenFacadeService = userTokenFacadeService;
    }

    @Override
    public Boolean updateUserLikeNumByUserId(Long userId) {
        int update = userMapper.updateLikeNumByUserId(userId);
        //更新用户缓存
        userService.loadUserToRedis(userId);
        return update > 0;
    }

    @Override
    public Boolean updateUserLikeNum(Long userId) {
        log.debug("开始同步用户点赞数:userId:{}", userId);
        List<Long> userids = userMapper.selectUserGreaterThanUserId(userId);
        if (!CollectionUtils.isEmpty(userids)) {
            List<UserLikeVo> userLikeVos = userNewsIntegrationService.getLikeNumByUserIds(userids);
            updateUserLike(userLikeVos);

            Optional<Long> maxUserId = userids.stream().max(Long::compareTo);
            if (maxUserId.isPresent()) {
                return updateUserLikeNum(maxUserId.get());
            }
        }
        //删除用户缓存
        userService.removeUserCache(userId);
        return true;
    }

    private void updateUserLike(List<UserLikeVo> userLikeVos) {
        if (!CollectionUtils.isEmpty(userLikeVos)) {
            MybatisBatchBuilder.create(UserMapper.class, userLikeVos).run(UserMapper::updateUserLikeNumByUserId);
        }
    }

    @Override
    public void addUserFollowCount(Long userId, Long targetUserId, boolean isAdd) {
        int addCount = isAdd ? 1 : -1;

        userMapper.addAttentionNum(userId, addCount);
        userMapper.addFanNum(targetUserId, addCount);

        userService.loadUserToRedis(userId);
        userService.loadUserToRedis(targetUserId);
    }

    @Override
    public Boolean updateUserReceiveRedPacket(Long userId) {
        this.userMapper.updateUserReceiveRedPacket(userId);
        return Boolean.TRUE;
    }

    @Override
    public Boolean addPostReplyNum(Long userId, Integer type) {
        if (1 == type) {
            userMapper.addPostNum(userId);
        } else if (2 == type) {
            userMapper.minusPostNum(userId);
        } else if (3 == type) {
            userMapper.addReplyNum(userId);
        } else if (4 == type) {
            userMapper.minusReplyNum(userId);
        }

        userService.loadUserToRedis(userId);
        return true;
    }

    @Override
    public void savePush(PushParam pushParam) {
        UserTokenParam param = new UserTokenParam();
        param.setToken(pushParam.getPushToken());

        for (PlatformTypeEnum typeEnum : PlatformTypeEnum.values()) {
            if (typeEnum.name().equalsIgnoreCase(pushParam.getPushPlatform())) {
                param.setTokenType(typeEnum);
            }
        }

        for (PlatformEnum platformEnum : PlatformEnum.values()) {
            if (Objects.equals(platformEnum.getCode(), pushParam.getPlatform())) {
                param.setPlatform(platformEnum);
            }
        }
        param.setUserId(pushParam.getUserId());

        userTokenFacadeService.addToken(param);
    }

    @Override
    public void addInviteNum(Long userId) {
        userMapper.incrementInviteNum(userId);
        // 重新加载用户缓存
        userService.loadUserToRedis(userId);
    }
}
