/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.user.attribute.impl;


import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.user.attribute.UserChannelService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.LocationChannelMapper;
import com.bxm.localnews.user.domain.LocationRelationChannelMapper;
import com.bxm.localnews.user.dto.LocationDTO;
import com.bxm.localnews.user.dto.UserAllChannelDTO;
import com.bxm.localnews.user.dto.UserChannelDTO;
import com.bxm.localnews.user.integration.LocationIntegrationService;
import com.bxm.localnews.user.param.UserChannelEditParam;
import com.bxm.localnews.user.param.UserChannelParam;
import com.bxm.localnews.user.vo.LocationRelationUser;
import com.bxm.localnews.user.vo.UserChannelVO;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import net.sf.json.regexp.RegexpMatcher;
import net.sf.json.regexp.RegexpUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.parseObject;
import static com.alibaba.fastjson.JSON.toJSONString;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;

/**
 * @author : jieli.xu
 * @date 2020/6/22 10:33
 * 用户栏目信息管理
 */
@Service
public class UserChannelServiceImpl implements UserChannelService {

    @Autowired
    private LocationChannelMapper locationChannelMapper;

    @Autowired
    private LocationRelationChannelMapper locationRelationChannelMapper;

    @Autowired
    private RedisStringAdapter redisStringAdapter;

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private LocationIntegrationService locationIntegrationService;

    private static final RegexpMatcher MATCHER = RegexpUtils.getMatcher("^\\{\\S+\\}\\S*$");


    /**
     * 获取所有的栏目信息
     *
     * @param userChannelParam ： 栏目信息参数
     * @return ： 所有的栏目
     */
    @Override
    public UserAllChannelDTO getAllUserChannel(UserChannelParam userChannelParam) {
        UserAllChannelDTO userAllChannelDTO = new UserAllChannelDTO();
        //获取最新版本号
        Long newVersion = this.getNewVersion(userChannelParam.getLocationCode());
        String value = redisStringAdapter.getString(this.buildUserChannelKey(userChannelParam.getUserId(), userChannelParam.getLocationCode()));
        if (isNotEmpty(value)) {
            userAllChannelDTO = parseObject(value, UserAllChannelDTO.class);
            if (newVersion.equals(userAllChannelDTO.getVersion())) {
                return processPublishButton(userChannelParam, userAllChannelDTO);
            }
        }
        userAllChannelDTO.setVersion(newVersion);
        //获取所有的栏目
        List<UserChannelVO> allChannelList = locationChannelMapper.getAllChannelList(userChannelParam.getLocationCode());
        //获取用户的栏目
        List<UserChannelVO> userChannelList = locationChannelMapper.getUserChannelList(userChannelParam.getLocationCode(), userChannelParam.getUserId());
        //处理用户保存的数据
        mergeUserChannelList(allChannelList, userChannelList, userChannelParam.getLocationCode());

        userAllChannelDTO.setAllChannelList(convertVoToDto(allChannelList, userChannelParam.getLocationCode()));
        userAllChannelDTO.setUserChannelList(convertVoToDto(userChannelList, userChannelParam.getLocationCode()));
        ///存入缓存
        redisStringAdapter.set(this.buildUserChannelKey(userChannelParam.getUserId(), userChannelParam.getLocationCode()), toJSONString(userAllChannelDTO));

        return processPublishButton(userChannelParam, userAllChannelDTO);
    }

    /**
     * 根据版本处理发帖按钮
     */
    private UserAllChannelDTO processPublishButton(UserChannelParam param, UserAllChannelDTO userAllChannelDTO) {

        // 3.5.0以上才需要处理
        if (StringUtils.compareVersion(param.getCurVer(), "3.5.0") >= 0) {
            if (!CollectionUtils.isEmpty(userAllChannelDTO.getAllChannelList())) {
                userAllChannelDTO.getAllChannelList().forEach(p -> {
                    if (isNotBlank(p.getChannelUrl())) {
                        p.setChannelUrl(p.getChannelUrl().replace("&publishBtn=1", ""));
                    }
                });
            }

            if (!CollectionUtils.isEmpty(userAllChannelDTO.getUserChannelList())) {
                userAllChannelDTO.getUserChannelList().forEach(p -> {
                    if (isNotBlank(p.getChannelUrl())) {
                        p.setChannelUrl(p.getChannelUrl().replace("&publishBtn=1", ""));
                    }
                });
            }
        }

        return userAllChannelDTO;
    }

    @Override
    public List<UserChannelDTO> getUserChannel(UserChannelParam userChannelParam) {
        UserAllChannelDTO userAllChannelDTO = this.getAllUserChannel(userChannelParam);
        return userAllChannelDTO.getUserChannelList();
    }


    @Override
    public Boolean updateUserChannel(UserChannelParam userChannelParam) {
        //真删除一些老的数据
        locationRelationChannelMapper.deleteByUidCode(userChannelParam.getUserId(), userChannelParam.getLocationCode());
        List<LocationRelationUser> list = new ArrayList<>();
        for (UserChannelEditParam entity : userChannelParam.getUserChannelEditParams()) {
            LocationRelationUser relationUser = convertParamToLru(entity);
            relationUser.setCode(userChannelParam.getLocationCode());
            relationUser.setUserId(userChannelParam.getUserId());
            relationUser.setStatus(1);
            list.add(relationUser);
        }
        //批量新增
        locationRelationChannelMapper.batchInsert(list);
        //删除缓存
        redisStringAdapter.remove(buildUserChannelKey(userChannelParam.getUserId(), userChannelParam.getLocationCode()));
        return true;
    }

    /**
     * 对用户栏目进行补充，所有栏目过滤掉用户栏目的
     *
     * @param allChannelList  ： 所有栏目
     * @param userChannelList ： 用户的保存的栏目
     * @param areaCode        ： 投放地区code
     */
    private void mergeUserChannelList(List<UserChannelVO> allChannelList, List<UserChannelVO> userChannelList, String areaCode) {
        if (allChannelList.isEmpty()) {
            return;
        }
        if (null == userChannelList) {
            userChannelList = new ArrayList<>();
        }
        List<UserChannelVO> needRemoveList = new ArrayList<>();
        for (UserChannelVO entity : allChannelList) {
            //1 代表查询出该地区，但是黑名单排除，就是反选的
            if (null != entity.getAreaType() && entity.getAreaType().equals(1) && null != entity.getCode() && entity.getCode().indexOf(areaCode) != -1) {
                needRemoveList.add(entity);
            } else {
                if (!userChannelList.contains(entity)) {
                    //默认配置或者默认置顶的需要配置给用户
                    if (entity.getIsFixed().equals(1) || entity.getFocusOn().equals(1)) {
                        userChannelList.add(entity);
                    }
                } else if (userChannelList.size() > 0) {
                    userChannelList.forEach(userChannelVO -> {
                        if (userChannelVO.getId().equals(entity.getId())) {
                            //补充LocationRelationUser这个表没有保存的数据
                            userChannelVO.setChannelUrl(entity.getChannelUrl());
                            userChannelVO.setFocusOn(entity.getFocusOn());
                            userChannelVO.setIsFixed(entity.getIsFixed());
                            if (entity.getIsFixed().equals(1)) {
                                userChannelVO.setSort(entity.getSort());
                            }
                            return;
                        }
                    });
                }
            }
        }
        //所有栏目去掉用户的和不再这个地区的
        allChannelList.removeAll(userChannelList);
        allChannelList.removeAll(needRemoveList);
        //用户如果已经保存了，也要删除need
        userChannelList.removeAll(needRemoveList);
        //排序
        userChannelList.sort(UserChannelVO::compareTo);
    }

    /**
     * 转换listVO为listDTO
     *
     * @param listVO
     */
    private List<UserChannelDTO> convertVoToDto(List<UserChannelVO> listVO, String locationCode) {
        LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(locationCode);
        AtomicReference<String> cityName = new AtomicReference("本地");
        if (Objects.nonNull(locationByGeocode) || isNotBlank(locationByGeocode.getName())) {
            cityName.set(locationByGeocode.getName());
        }
        return listVO.stream()
                .map(p -> {
                    UserChannelDTO userChannelDTO = new UserChannelDTO();
                    BeanUtils.copyProperties(p, userChannelDTO);
                    userChannelDTO.setChannelId(p.getId());

                    // 如果有占位符 进行替换
                    // 暂时只有一个 多了可以进行设计
                    if (isNotBlank(userChannelDTO.getChannelName())
                            && MATCHER.matches(userChannelDTO.getChannelName())) {
                        userChannelDTO.setChannelName(replace(userChannelDTO.getChannelName(), cityName.get()));
                    }
                    return userChannelDTO;
                }).collect(Collectors.toList());
    }

    private String replace(String channelName, String replaceStr) {
        int start = channelName.indexOf("{");
        int end = channelName.indexOf("}");
        String startStr = channelName.substring(0, start);
        String endStr = channelName.substring(end + 1);
        return StringUtils.join(startStr, replaceStr, endStr);
    }

    /**
     * 客户端传的参数改为数据库入参bean
     *
     * @param param
     * @return
     */
    private LocationRelationUser convertParamToLru(UserChannelEditParam param) {
        LocationRelationUser locationRelationUser = new LocationRelationUser();
        locationRelationUser.setLocationChannelId(param.getChannelId());
        locationRelationUser.setSort(param.getSort());
        return locationRelationUser;
    }

    /**
     * 获取当前地区下面的栏目的最新版本
     *
     * @param locationCode ： 区域编码
     * @return 获取最新版本
     */
    private Long getNewVersion(String locationCode) {
        List<Long> list = redisHashMapAdapter.multiGet(RedisConfig.LOCATION_VERSION_CACHE, Arrays.asList(new String[]{locationCode, "0"}), Long.class);
        AtomicReference<Long> newVersion = new AtomicReference<>(0L);
        list.forEach(version -> {
            if (null != version) {
                newVersion.updateAndGet(v -> v + version);
            }
        });
        return newVersion.get();
    }


    /**
     * 创建用户关联地区的key
     *
     * @param userId       : 用户id
     * @param locationCode ： 区域编码
     * @return
     */
    private KeyGenerator buildUserChannelKey(Long userId, String locationCode) {
        KeyGenerator keyGenerator = RedisConfig.USER_LOCATION_LIST_CACHE.copy().appendKey(userId).appendKey(locationCode);
        return keyGenerator;
    }
}
