package com.bxm.localnews.user.attribute.impl;

import com.bxm.localnews.mq.common.constant.UserEventEnum;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.localnews.user.attribute.UserAttributeService;
import com.bxm.localnews.user.attribute.UserFollowService;
import com.bxm.localnews.user.attribute.UserFunsService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserFollowMapper;
import com.bxm.localnews.user.domain.UserFunsMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.domain.VirtualUserMapper;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.UserFollowStatusEnum;
import com.bxm.localnews.user.integration.MessageUserIntegrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.UserFollowParam;
import com.bxm.localnews.user.properties.NativeUserProperties;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vo.UserFollow;
import com.bxm.localnews.user.vo.UserFollowRecord;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.Boolean.TRUE;
import static java.util.Objects.nonNull;

@Service
public class UserFollowServiceImpl extends BaseService implements UserFollowService {

    private static final String USER_FOLLOW = "USER_FOLLOW_";

    /**
     * 关注缓存消息过期时间
     */
    private static final Long FOLLOW_CACHE_EXPIRED = 60 * 60 * 24 * 15L;

    @Resource
    private UserMapper userMapper;

    @Resource
    private UserFollowMapper userFollowMapper;

    @Resource
    private UserFunsMapper userFunsMapper;

    @Resource
    private UserService userService;

    @Resource
    private UserAttributeService userAttributeService;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private UserFunsService userFunsService;

    @Resource
    private MessageUserIntegrationService messageUserIntegrationService;

    @Resource
    private VirtualUserMapper virtualUserMapper;

    @Resource
    private NativeUserProperties nativeUserProperties;

    @Resource
    private MessageSender messageSender;

    @Autowired
    private UserVipService userVipService;

    @Override
    public Boolean isFollowed(Long currentUserId, Long targetUserId) {
        KeyGenerator keyGenerator = getFollowRedisKey(currentUserId);
        if (!redisSetAdapter.hasKey(keyGenerator)) {
            //从数据库中取
            List<Long> targetUserIdList = userFollowMapper.getFollowedUserIdList(currentUserId);
            if (CollectionUtils.isEmpty(targetUserIdList)) {
                //查询为空时，放入特殊值，防止后续继续查询数据库
                redisSetAdapter.add(keyGenerator, -1L);
                return false;
            }
            redisSetAdapter.add(keyGenerator, targetUserIdList.toArray());
        }

        return redisSetAdapter.exists(keyGenerator, targetUserId);
    }

    @Override
    public List<Long> isFolloweds(Long currentUserId,List<Long> targetUserIds ){
        List<Long> result = new ArrayList<>();
        for (Long targetUserId : targetUserIds) {
            if(this.isFollowed(currentUserId,targetUserId)){
                result.add(targetUserId);
            }
        }
        return result;
    }

    @Override
    public Boolean hasFollowMsg(Long userId, Long targetUserId) {
        //是否存在目标用户曾经关注当前用户的记录
        KeyGenerator key = RedisConfig.COMSUME_FOLLOW_INFO.copy().appendKey(userId).appendKey(targetUserId);

        if (redisStringAdapter.hasKey(key)) {
            redisStringAdapter.remove(key);
            return true;
        }

        return false;
    }

    @Override
    public Boolean follow(Long userId, Long followUserId, Byte type) {
        if (notValidUser(userId) || notValidUser(followUserId)) {
            return false;
        }

        if (Objects.equals(userId, followUserId)) {
            logger.warn("用户: {} 自己关注自己", userId);
            return true;
        }

        String requestId = nextSequence().toString();
        String lockKey = USER_FOLLOW + userId + "_" + followUserId;

        //关注或者取消关注，需要使用分布式锁，需注意幂等操作
        if (distributedLock.lock(lockKey, requestId)) {
            //当前用户是否关注对方
            boolean currentFollow = isFollowed(userId, followUserId);
            //对方是否关注当前用户
            boolean targetFollow = isFollowed(followUserId, userId);

            //如果是关注，并且当前用户未关注目标用户(已关注则不进行处理)
            if (UserFollowStatusEnum.FOLLOW.getCode() == type && !currentFollow) {
                //如果目标用户也关注了当前用户，则更新对方的记录为相互关注
                if (targetFollow) {
                    //增加当前用户相互关注的记录
                    addFollow(userId, followUserId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                    //对方用户的关注变成相互关注
                    modifyFollow(followUserId, userId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                    //从当前用户的粉丝中设置为相互关注
                    userFunsService.setFuns(userId, followUserId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                    //对方用户的粉丝新增互相关注
                    userFunsService.addFuns(followUserId, userId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                } else {
                    //增加一条关注记录
                    addFollow(userId, followUserId, UserFollowStatusEnum.FOLLOW.getCode());
                    //给对方增加粉丝
                    userFunsService.addFuns(followUserId, userId, UserFollowStatusEnum.FOLLOW.getCode());
                }

                //更新用户的关注、粉丝信息
                userAttributeService.addUserFollowCount(userId, followUserId, true);
                //关注成功
                messageUserIntegrationService.addFollowMessage(followUserId);
            } else if (UserFollowStatusEnum.UNFOLLOW.getCode() == type && currentFollow) {
                //将当前用户从对方的粉丝中移除
                userFunsService.removeFuns(followUserId, userId);
                //删除当前用户的关注列表
                removeFollow(userId, followUserId);
                //如果对方关注了当前用户
                if (targetFollow) {
                    //更新对方的关注状态为关注
                    modifyFollow(followUserId, userId, UserFollowStatusEnum.FOLLOW.getCode());
                    //将当前用户粉丝中的目标用户更新为关注
                    userFunsService.setFuns(userId, followUserId, UserFollowStatusEnum.FOLLOW.getCode());
                }

                //扣除用户的关注、目标用户的粉丝数
                userAttributeService.addUserFollowCount(userId, followUserId, false);
            }
            //解除分布式锁
            distributedLock.unlock(lockKey, requestId);
        }

        return true;
    }

    /**
     * 增加一条关注记录
     *
     * @param userId   用户ID
     * @param followId 关注目标用户ID
     * @param type     关注类型
     */
    private void addFollow(Long userId, Long followId, byte type) {
        UserFollowRecord follow = new UserFollowRecord();
        follow.setUserId(userId);
        follow.setFollowedUserId(followId);
        follow.setType(type);

        int update = userFollowMapper.updateFollowedStatus(follow);
        if (0 == update) {
            follow.setId(nextId());
            userFollowMapper.insertFollowed(follow);
        }
        redisSetAdapter.add(getFollowRedisKey(userId), followId);

        //给被关注人增加一条用户事件，用于在私聊页面时实时显示相关关注关系
        PushPayloadInfo userEvent = PushPayloadInfo.build().setType(UserEventEnum.ADD_FOLLOW.type)
                .addExtend("userId", followId)
                .addExtend("followUserId", userId)
                .addExtend("followType", type);

        messageSender.sendUserEvent(userEvent);

        //给被关注人增加一条消费消息
        KeyGenerator comsumeKey = RedisConfig.COMSUME_FOLLOW_INFO.copy().appendKey(followId).appendKey(userId);
        redisStringAdapter.set(comsumeKey, StringUtils.EMPTY, FOLLOW_CACHE_EXPIRED);
    }

    /**
     * 变更关注记录类型
     *
     * @param userId   用户ID
     * @param followId 关注目标用户ID
     * @param type     关注类型
     */
    private void modifyFollow(Long userId, Long followId, byte type) {
        if (UserFollowStatusEnum.UNFOLLOW.getCode() == type) {
            removeFollow(userId, followId);
        } else {
            UserFollowRecord follow = new UserFollowRecord();
            follow.setUserId(userId);
            follow.setFollowedUserId(followId);
            follow.setType(type);

            userFollowMapper.updateFollowedStatus(follow);
        }
    }

    /**
     * 移除关注记录
     *
     * @param userId   用户ID
     * @param followId 关注目标用户ID
     */
    private void removeFollow(Long userId, Long followId) {
        UserFollowRecord follow = new UserFollowRecord();
        follow.setUserId(userId);
        follow.setFollowedUserId(followId);
        follow.setType(UserFollowStatusEnum.UNFOLLOW.getCode());

        userFollowMapper.updateFollowedStatus(follow);
        redisSetAdapter.remove(getFollowRedisKey(userId), followId);
    }

    @Override
    public List<UserFollow> mayInterested(Long userId, String areaCode) {
        List<Long> userIdList = userMapper.getLikeUserIdListByArea(areaCode);

        if (CollectionUtils.isEmpty(userIdList)) {
            return null;
        }

        List<Long> resultUserIdList = userIdList.stream().filter(e -> !isFollowed(userId, e) && !userId.equals(e)).limit(3)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(resultUserIdList) || resultUserIdList.size() < 3) {
            return null;
        }

        List<UserFollow> userFollowList = new ArrayList<>();
        resultUserIdList.forEach(e -> {
            UserFollow userFollow = new UserFollow();
            userFollow.setUserId(e);
            userFollow.setStatus((byte) 1);
            userFollowList.add(userFollow);
        });

        fillExtraInfo(userFollowList);

        userFollowList.forEach(item -> {
            item.setIsVip(userVipService.isVip(item.getUserId()) ? 1 : 0);
        });
        return userFollowList;
    }

    @Override
    public PageWarper<UserFollow> followList(UserFollowParam param) {
        PageWarper<UserFollow> resultPage = new PageWarper<>(userFollowMapper.queryFollowByPage(param));
        fillExtraInfo(resultPage.getList());

        return resultPage;
    }

    @Override
    public PageWarper<UserFollow> queryFunsByPage(UserFollowParam param) {
        PageWarper<UserFollow> resultPage = new PageWarper<>(userFunsMapper.queryFunsByPage(param));
        fillExtraInfo(resultPage.getList());
        return resultPage;
    }

    @Override
    public void virtualFollow(Long userId, Integer num) {
        List<Long> virtualUserIdList = virtualUserMapper.getRandomVirtualUserIdList(num);
        virtualUserIdList.forEach(e -> follow(e, userId, (byte) 0));
    }

    private KeyGenerator getFollowRedisKey(Long userId) {
        return RedisConfig.USER_FOLLOW_LIST.copy().appendKey(userId);
    }

    private void fillExtraInfo(List<UserFollow> resultList) {
        if (!CollectionUtils.isEmpty(resultList)) {
            List<UserInfoDTO> userInfoList = batchLoadFormCache(resultList);

            resultList.forEach(userFollow -> {
                Optional<UserInfoDTO> userInfoOptional = userInfoList.stream()
                        .filter(user -> user.getId().equals(userFollow.getUserId()))
                        .findFirst();
                if (userInfoOptional.isPresent()) {
                    UserInfoDTO userInfo = userInfoOptional.get();
                    userFollow.setIsVip(Objects.equals(userVipService.isVip(userInfo.getId()), TRUE) ? 1 : 0);
                    userFollow.setNickname(userInfo.getNickname());
                    userFollow.setHeadImg(userInfo.getHeadImg());
                    userFollow.setSex(userInfo.getSex());
                    if (Objects.equals(userInfo.getIsDefaultPersonalProfile(), TRUE)) {
                        userFollow.setPersonalProfile(nativeUserProperties.getDefaultPersonalProfile());
                    } else {
                        userFollow.setPersonalProfile(userInfo.getPersonalProfile());
                    }
                }
            });
        }
    }

    /**
     * 根据id列表批量取得用户的用户
     */
    private List<UserInfoDTO> batchLoadFormCache(List<UserFollow> resultList) {
        Set<Long> originUserIds = resultList.stream().map(UserFollow::getUserId).collect(Collectors.toSet());
        return userService.getBatchUserInfo(originUserIds);
    }

    private boolean notValidUser(Long userId) {
        UserInfoDTO userInfoDTO = userService.getUserCache(userId);
        if (null == userInfoDTO) {
            return true;
        }
        return nonNull(userInfoDTO.getState()) && !Objects.equals(userInfoDTO.getState(), AppConst.USER_STATE.NORMAL)
                && !Objects.equals(userInfoDTO.getState(), AppConst.USER_STATE.VIRTUAL);
    }

}
