package com.bxm.localnews.user.attribute.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.common.constant.InfoCombineStateEnum;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.msg.integration.SmsIntegrationService;
import com.bxm.localnews.user.attribute.UserPersonalInfoService;
import com.bxm.localnews.user.attribute.UserTagService;
import com.bxm.localnews.user.constant.RedisConfig;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.domain.UserTagMapper;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.UserPersonalInfoDTO;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import com.bxm.localnews.user.event.AttributeActionEvent;
import com.bxm.localnews.user.integration.UserNewsIntegrationService;
import com.bxm.localnews.user.login.ChuanglanFlashService;
import com.bxm.localnews.user.login.LoginService;
import com.bxm.localnews.user.login.UserAuthService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.model.PopCache;
import com.bxm.localnews.user.param.*;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.support.UserSyncService;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserInfoComplete;
import com.bxm.localnews.user.vo.UserTag;
import com.bxm.localnews.user.vo.UserTagRedis;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.*;
import com.bxm.newidea.component.vo.Message;
import com.google.common.eventbus.EventBus;
import lombok.AllArgsConstructor;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class UserPersonalInfoServiceImpl extends BaseService implements UserPersonalInfoService, ApplicationContextAware {

    /**
     * 引导资料完善弹窗key
     */
    private static final String KEY = "PERFECT_INFORMATION_POP";

    /**
     * 记录引导资料完善弹窗的时间
     */
    private static final String LAST_MODIFY_TIME = "lastModifyTime";

    /**
     * 正则表达式预编译[cq]
     */
    private static final Pattern WECHAT_NAME_PATTERN = Pattern.compile("^[a-zA-Z0-9_-]{6,20}$");

    private final UserMapper userMapper;

    private final UserService userService;

    private final UserSyncService userSyncService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserTagService userTagService;

    private final UserTagMapper userTagMapper;

    private final UserNewsIntegrationService userNewsIntegrationService;

    private final UserAuthService userAuthService;

    private final ChuanglanFlashService chuanglanFlashService;

    private final UserProperties userProperties;

    private final SmsIntegrationService smsIntegrationService;

    private ApplicationContext applicationContext;

    private EventBus userActionEventBus;

    public LoginService getLoginService() {
        return SpringContextHolder.getBean(LoginService.class);
    }

    /**
     * 前置操作：判断用户存在与否 && 用户是否处于正常状态
     *
     * @param userId 用户id
     */
    private Message preExecute(Long userId) {
        User user = this.userMapper.selectByPrimaryKey(userId);

        //判断用户是否存在
        if (null == user) {
            return Message.build(false, "用户不存在");
        }

        //判断用户登录状态
        if (AppConst.USER_STATE.NORMAL != user.getState()) {
            // 如果状态不正常 修改为正常状态
            logger.warn("出现暂未定位原因的异常数据，将其更新为正常状态，用户ID：{}", userId);

            user.setState(AppConst.USER_STATE.NOT_ACTIVE);
            getLoginService().updateUserActive(user, new BasicParam(), null);
        }

        return Message.build(true).addParam("user", user);
    }

    /**
     * 后置操作：更新用户冗余信息 & 判断是否完善个人资料 & 保存用户信息至redis中
     *
     * @param user               用户实体
     * @param isRedundancyChange 是否需要更新用户冗余信息
     * @param isTagChange        是否更改了用户的兴趣标签
     * @param userTagList        用户的兴趣标签，如果没有则没有值
     */
    private void afterExecute(
            User user,
            boolean isRedundancyChange,
            boolean isTagChange,
            boolean isPush,
            List<UserTag> userTagList) {

        //如果是改了兴趣的话，得到修改的兴趣标签并保存
        if (isTagChange) {
            UserTagRedis userTagRedis = UserTagRedis.buildUserTagRedis(user.getId(), userTagList);
            this.redisHashMapAdapter.put(RedisConfig.USER_TAG, String.valueOf(user.getId()), JSON.toJSONString(userTagRedis));
        }

        //更新redis中的缓存
        this.userService.putUserToRedis(user);

        //判断是否完成完善资料的任务
        userService.checkCompleteInfo(user);

        //异步执行处理不紧急的任务，比如冗余信息的同步
        applicationContext.getBean(UserPersonalInfoService.class).asyncProcessing(user, isRedundancyChange, isPush);
    }

    @Override
    @Async
    public void asyncProcessing(User user, boolean changeRedundancy, boolean changeRecommendAttr) {
        //将变更的用户信息同步到冗余的表设计字段中
        if (changeRedundancy) {
            UserSyncParam userSyncParam = new UserSyncParam();
            BeanUtils.copyProperties(user, userSyncParam);
            userSyncService.sync(userSyncParam);
        }

        //推送用户推荐信息
        if (changeRecommendAttr) {
            AttributeActionEvent attributeActionEvent = AttributeActionEvent.of()
                    .setHometownCode(user.getHometownCode())
                    .setSex(Objects.isNull(user.getSex()) ? 0 : Integer.valueOf(user.getSex()))
                    .setBirth(user.getBirthday())
                    .setJobCategory(user.getJobCategory());

            attributeActionEvent.setUserId(user.getId());
            if (Objects.nonNull(user.getInfoCompleteState())) {
                attributeActionEvent.setInfoCompletePercent(InfoCombineStateEnum.getCustomInfoCompleteStatus(user.getInfoCompleteState()));
            }

            userActionEventBus.post(attributeActionEvent);
        }
    }

    @Override
    public Message doUpdateHeadImg(UserHeadImgParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");
        userEntity.setHeadImg(user.getHeadImg());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex())) {
            count = userMapper.updateUserHeadImg(userEntity.getId(), userEntity.getHeadImg(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserHeadImg(userEntity.getId(), userEntity.getHeadImg(), infoCompleteStatus);
        }

        if (count <= 0) {
            return Message.build(false);
        }

        afterExecute(userEntity, true, false, false, null);

        return Message.build(true);
    }

    @Override
    public Message doUpdateNickname(UserNicknameParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        if (userNewsIntegrationService.verifySensitiveWord(user.getNickname())) {
            return Message.build(false, "昵称包含敏感词汇");
        }

        User userEntity = m.getParam("user");
        //3.0.0 [康文静]取消昵称修改次数限制
        //if (null != userEntity.getIsTempNickName() && userEntity.getIsTempNickName() == 0) {
        //return Message.build(false, "该用户修改昵称已达上限");
        //}
        //userEntity.setIsTempNickName((byte) 0);

        userEntity.setNickname(EmojiCodeParser.replaceSoftbankEmoji(user.getNickname()));
        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex())) {
            count = userMapper.updateUserNickname(userEntity.getId(), userEntity.getNickname(), userEntity.getIsTempNickName(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper
                    .updateUserNickname(userEntity.getId(), userEntity.getNickname(), userEntity.getIsTempNickName(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, true, false, false, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateBirth(UserBirthParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");
        userEntity.setBirthday(user.getBirthday());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex())) {
            count = userMapper.updateUserBirth(userEntity.getId(), userEntity.getBirthday(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserBirth(userEntity.getId(), userEntity.getBirthday(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, false, false, true, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateHometown(UserHomeTownParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");
        userEntity.setHometownCode(user.getHometownCode());
        userEntity.setHometownName(user.getHometownName());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex())) {
            count = userMapper.updateUserHometown(userEntity.getId(), userEntity.getHometownCode(), userEntity.getHometownName(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper
                    .updateUserHometown(userEntity.getId(), userEntity.getHometownCode(), userEntity.getHometownName(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, false, false, true, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateEmotion(UserEmotionParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");
        userEntity.setRelationshipStatus(user.getRelationshipStatus());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex())) {
            count = userMapper.updateUserRelationshipStatus(userEntity.getId(), userEntity.getRelationshipStatus(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserRelationshipStatus(userEntity.getId(), userEntity.getRelationshipStatus(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, false, false, false, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateJob(UserJobParam user) {
        logger.debug("更新职业,参数为:[{}]", JSONObject.toJSON(user));
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");
        userEntity.setCompany(user.getCompany());
        userEntity.setJobTitle(user.getJobTitle());
        userEntity.setJobCategory(user.getJobCategory());
        userEntity.setJobCategoryName(user.getJobCategoryName());
        userEntity.setJobType(user.getJobType());
        userEntity.setJobTypeName(user.getJobTypeName());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex())) {
            count = userMapper.updateUserJob(userEntity.getId(),
                    userEntity.getJobType(),
                    userEntity.getJobTypeName(),
                    userEntity.getJobCategoryName(),
                    userEntity.getJobCategory(),
                    userEntity.getCompany(),
                    userEntity.getJobTitle(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserJob(userEntity.getId(),
                    userEntity.getJobType(),
                    userEntity.getJobTypeName(),
                    userEntity.getJobCategoryName(),
                    userEntity.getJobCategory(),
                    userEntity.getCompany(),
                    userEntity.getJobTitle(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, false, false, true, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdatePersonalProfile(UserIntroductionParam user) {
        //判断标签是否包含敏感词汇
        if (userNewsIntegrationService.verifySensitiveWord(user.getPersonalProfile())) {
            return Message.build(false, "个人简介含有敏感词，请重新输入");
        }

        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");
        userEntity.setPersonalProfile(user.getPersonalProfile());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex())) {
            count = userMapper.updateUserPersonalProfile(userEntity.getId(), userEntity.getPersonalProfile(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil
                    .setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserPersonalProfile(userEntity.getId(), userEntity.getPersonalProfile(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, false, false, false, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateUserPhone(UserPhoneParam user) {
        logger.debug("更换手机号:[{}]", JSONObject.toJSON(user));

        //根据不同的验证方式得到手机号
        Message validMessage = validPhoneProxy(user);
        if (!validMessage.isSuccess()) {
            return validMessage;
        }

        //验证成功获得手机号
        String phone = validMessage.getParam("phone");

        //预处理
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");

        //如果已经绑定则暂不支持更换手机号
        if (isNotBlank(userEntity.getPhone())) {
            if (phone.equals(userEntity.getPhone())) {
                return Message.build(true).addParam("phone", phone);
            } else {
                return Message.build(false, "已绑定其他手机号，请重新登陆")
                        .addParam("code", RespCode.UNAUTHORIZED);
            }
        }

        //判断是否有新手机号的权限验证
        Message messageByCheckAuth = userAuthService.checkBindExist((byte) 1, phone);
        if (!messageByCheckAuth.isSuccess()) {
            return messageByCheckAuth;
        }

        userEntity.setPhone(phone);

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.PHONE_COMPLETE.getIndex())) {
            count = userMapper.updateUserPhone(userEntity.getId(), phone, null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PHONE_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserPhone(userEntity.getId(), phone, infoCompleteStatus);
        }

        if (count > 0) {
            //增加手机权限
            userAuthService.addUserAuth((byte) 1, user.getUserId(), phone, null);

            // 如果校验风险开关开启的 判断用户是否为风险用户
            if (Objects.equals(userProperties.getRiskCheckSwitch(), Boolean.TRUE)) {

                userService.isRisk(user.getUserId(), phone, user, user.getDeviceToken(), user.getIp());
            }

            afterExecute(userEntity, false, false, false, null);
        }

        return Message.build(true).addParam("phone", phone);
    }

    /**
     * 手机号验证分发器，现在有两种分发模式 1.根据验证码验证 2.闪验一键登录
     */
    private Message validPhoneProxy(UserPhoneParam userPhoneParam) {
        if (isNotBlank(userPhoneParam.getFlashToken())) {
            return validPhoneByFlashToken(userPhoneParam);
        } else if (isNotBlank(userPhoneParam.getPhone())
                && isNotBlank(userPhoneParam.getCode())) {
            return validPhoneByCode(userPhoneParam);
        }

        return Message.build(false, "参数错误");
    }

    /**
     * 根据创蓝闪验 验证手机号
     */
    private Message validPhoneByFlashToken(UserPhoneParam userPhoneParam) {
        ChuangLanLoginParam chuangLanLoginParam = new ChuangLanLoginParam();
        BeanUtils.copyProperties(userPhoneParam, chuangLanLoginParam);
        String phone = chuanglanFlashService.getPhone(chuangLanLoginParam);
        if (null == phone) {
            logger.error("一键登录获取手机号码失败，参数：[{}]", chuangLanLoginParam);
            return Message.build(false, "获取手机号失败");
        }

        return Message.build(true).addParam("phone", phone);
    }

    /**
     * 根据短信验证码验证手机号
     */
    private Message validPhoneByCode(UserPhoneParam userPhoneParam) {
        //验证码的验证
        if (isNotBlank(userPhoneParam.getCode()) &&
                !this.smsIntegrationService.verifyByType(AppConst.SMS_TPL.VCODE_TYPE_BIND_PHONE,
                        userPhoneParam.getPhone(), userPhoneParam.getCode())) {
            logger.info("用户输入验证码失败，请求参数：{}", JSON.toJSONString(userPhoneParam));

            return Message.build(false, "请输入正确的验证码");
        }

        return Message.build(true).addParam("phone", userPhoneParam.getPhone());
    }

    @Override
    public Message doUpdateUserSex(UserSexParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        //如果不传性别则不做改动
        if (null == user.getSex() || 0 == user.getSex()) {
            return Message.build(true);
        }

        User userEntity = m.getParam("user");
        userEntity.setSex(user.getSex());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex())) {
            count = userMapper.updateUserSex(userEntity.getId(), userEntity.getSex(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserSex(userEntity.getId(), userEntity.getSex(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, true, false, true, null);
        }

        return Message.build(true);
    }

    @Override
    public Message doUpdateUserWechat(UserWechatParam user) {
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        Message isValid = validWechat(user.getWechatId());
        if (!isValid.isSuccess()) {
            return isValid;
        }

        User userEntity = m.getParam("user");
        userEntity.setWechatId(user.getWechatId());

        //更新完善资料度
        long infoCompleteStatus = userEntity.getInfoCompleteState();

        int count;
        if (BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.WECHAT_COMPLETE.getIndex())) {
            count = userMapper.updateUserWechat(userEntity.getId(), userEntity.getWechatId(), null);
        } else {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.WECHAT_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            count = userMapper.updateUserWechat(userEntity.getId(), userEntity.getWechatId(), infoCompleteStatus);
        }

        if (count > 0) {
            afterExecute(userEntity, true, false, true, null);
        }

        return Message.build(true);
    }

    /**
     * 判断微信号是否合法
     */
    private Message validWechat(String qqorwx) {
        if (StringUtils.isEmpty(qqorwx) || qqorwx.length() < 6 || qqorwx.length() > 20) {
            return Message.build(false, "请输入6-20个字符，支持字母，数字，下划线和减号");
        }
        //QQ号最短5位，微信号最短是6位最长20位
        Matcher m = WECHAT_NAME_PATTERN.matcher(qqorwx);
        boolean isValid = m.matches();
        if (!isValid) {
            return Message.build(false, "请不要输入中文和其他特殊符号");
        }
        return Message.build(true);
    }

    @Override
    public Message doUpdateNewbieGuideInfo(NewbieGuideParam newbieGuideParam) {
        Message m = preExecute(newbieGuideParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        Message isValid = validNewbieGuideInfo(newbieGuideParam);
        if (!isValid.isSuccess()) {
            return isValid;
        }

        User userEntity = m.getParam("user");
        BeanUtils.copyProperties(newbieGuideParam, userEntity);

        //更新完善资料度
        long infoCompleteStatus = initUserInfoCompleteStatus(userEntity);
        userEntity.setInfoCompleteState(infoCompleteStatus);

        int count = userMapper.updateNewbieGuideInfo(userEntity);

        if (count > 0) {
            afterExecute(userEntity, false, false, true, null);
        }

        return Message.build(true).addParam("user", userService.convertUserToDTO(userEntity));
    }

    private Message validNewbieGuideInfo(NewbieGuideParam newbieGuideParam) {
        if (StringUtils.isBlank(newbieGuideParam.getNickname())) {
            return Message.build(false, "请完善昵称，使用真实昵称，方便大家认识你");
        } else if (StringUtils.isBlank(newbieGuideParam.getHeadImg())) {
            return Message.build(false, "请完善头像，使用真实头像，方便大家认识你");
        } else if (null == newbieGuideParam.getSex()) {
            return Message.build(false, "请选择性别");
        } else if (null == newbieGuideParam.getBirthday()) {
            return Message.build(false, "请填写生日，可以找到更多同龄人哦");
        }
        return Message.build(true);
    }

    @Override
    public Message doUpdateUserTag(UserTagParam user) {
        logger.debug("更新用户标签信息：[{}]", JSONObject.toJSON(user));
        Message m = preExecute(user.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        long count = user.getSpecificUserTagParamList().parallelStream().filter(e -> e.getChooseFlag() != 0).count();
        if (count > 15) {
            return Message.build(false, "标签选择个数限定15个");
        }

        User userEntity = m.getParam("user");

        //更新标签
        List<UserTag> userTagList = new ArrayList<>();
        Date date = new Date();
        int index = 0;
        for (UserTagParam.SpecificUserTagParam specificUserTagParam : user.getSpecificUserTagParamList()) {
            //如果未选中并且是默认的，则不保存
            if (0 == specificUserTagParam.getChooseFlag() && 1 == specificUserTagParam.getDefaultFlag()) {
                continue;
            }
            UserTag userTag = UserTag.builder()
                    .createTime(date)
                    .id(nextSequence())
                    .label(specificUserTagParam.getLabel())
                    .order(index)
                    .defaultFlag(specificUserTagParam.getDefaultFlag())
                    .userId(user.getUserId())
                    .deleteFlag((byte) 0).build();
            //如果未选中并且是自定义的，则按<删除状态>保存
            if (0 == specificUserTagParam.getChooseFlag() && 0 == specificUserTagParam.getDefaultFlag()) {
                userTag.setDeleteFlag((byte) 1);
                userTag.setDeleteTime(date);
            }
            userTagList.add(userTag);
            index++;
        }

        this.userTagMapper.deleteByUserId(user.getUserId());
        MybatisBatchBuilder.create(UserTagMapper.class, userTagList).run(UserTagMapper::insert);

        //补全用户完善资料程度
        long infoCompleteStatus = userEntity.getInfoCompleteState();
        if (!CollectionUtils.isEmpty(user.getSpecificUserTagParamList())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        } else {
            infoCompleteStatus = BitOperatorUtil.clearBit(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        }
        userEntity.setInfoCompleteState(infoCompleteStatus);
        userMapper.updateUserInfoCompleteStatus(new UserInfoComplete(user.getUserId(), infoCompleteStatus));

        afterExecute(userEntity, false, true, false, userTagList);

        return Message.build(true);
    }

    @Override
    public UserPersonalInfoDTO getUserPersonalInfo(Long userId) {

        //获取用户
        UserInfoDTO dto = userService.getUserCache(userId);

        if (null == dto) {
            return null;
        }

        //得到个人资料完善度百分比
        int infoCompletePercent = InfoCombineStateEnum.getInfoCompletePercent(
                dto.getInfoCompleteState() == null ? 0 : dto.getInfoCompleteState()
        );

        String infoCompleteTaskText = userProperties.getInfoCompleteTaskText();

        //获取用户标签列表
        List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(userId);

        //包装用户实体
        UserPersonalInfoDTO userPersonalInfoDTO = convertUserInfo2UserPersonalInfo(dto);
        userPersonalInfoDTO.setInfoCompletePercent(infoCompletePercent);
        userPersonalInfoDTO.setInfoCompleteTaskText(infoCompleteTaskText);

        //包装用户标签
        userPersonalInfoDTO.setUserTagList(userTagList.stream()
                .filter(e -> e.getDeleteFlag() == 0)
                .map(UserTag::getLabel).collect(Collectors.toList()));

        return userPersonalInfoDTO;

    }

    /**
     * 用途如其名
     */
    private UserPersonalInfoDTO convertUserInfo2UserPersonalInfo(UserInfoDTO dto) {
        //包装用户实体
        UserPersonalInfoDTO.IndustryInfo industryInfo = null;
        if (null != dto.getIndustry()) {
            industryInfo = UserPersonalInfoDTO.IndustryInfo.builder()
                    .company(dto.getCompany())
                    .jobCategory(dto.getJobCategory())
                    .jobCategoryName(dto.getJobCategoryName())
                    .jobTitle(dto.getJobTitle())
                    .jobType(dto.getJobType())
                    .jobTypeName(dto.getJobTypeName()).build();
        }

        return UserPersonalInfoDTO.builder()
                .personalProfile(dto.getPersonalProfile())
                .isDefaultPersonalProfile(dto.getIsDefaultPersonalProfile())
                .birthday(dto.getBirthday())
                .headImg(dto.getHeadImg())
                .industryInfo(industryInfo)
                .industry(dto.getIndustry())
                // 3.4.0版本不需要用户引导 志化说返回1就可以去掉引导
                .newbieGuideFlag((byte) 1)
                //3.0.0 取消临时昵称，允许修改
                .isTempNickName((byte) 1)
                .generation(getGeneration(dto))
                //.isTempNickName(dto.getIsTempNickName())
                .nickname(dto.getNickname())
                .phone(dto.getPhone())
                .id(dto.getId())
                .relationshipStatus(dto.getRelationshipStatus())
                .sex(dto.getSex())
                .wechatId(dto.getWechatId())
                .hometownCode(dto.getHometownCode())
                .hometownName(dto.getHometownName())
                .build();
    }

    @Override
    public Long initUserInfoCompleteStatus(User user) {
        long infoCompleteStatus = 0L;
        //用户在登录时可能携带 性别、昵称、头像等信息，需要将这些作为完善资料度的参考
        if (null != user.getSex() && 0 != user.getSex()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.SEX_COMPLETE.getIndex());
        }
        if (isNotBlank(user.getNickname())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.NICKNAME_COMPLETE.getIndex());
        }
        if (isNotBlank(user.getHeadImg())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HEAD_IMG_COMPLETE.getIndex());
        }
        if (null != user.getBirthday()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.BIRTHDAY_COMPLETE.getIndex());
        }
        if (null != user.getRelationshipStatus()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.RELATION_SHIP_COMPLETE.getIndex());
        }
        if (isNotBlank(user.getHometownCode())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.HOMETOWN_COMPLETE.getIndex());
        }
        if (null != user.getJobCategory()) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.JOB_COMPLETE.getIndex());
        }
        if (isNotBlank(user.getPersonalProfile())) {
            infoCompleteStatus = BitOperatorUtil
                    .setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PERSONAL_PROFILE_COMPLETE.getIndex());
        }
        if (isNotBlank(user.getPhone())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.PHONE_COMPLETE.getIndex());
        }
        if (isNotBlank(user.getWechatId())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.WECHAT_COMPLETE.getIndex());
        }
        List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(user.getId());
        if (!CollectionUtils.isEmpty(userTagList)) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
        }
        return infoCompleteStatus;
    }

    @Override
    public Boolean checkInfoCompleteStatus(Long userId) {
        UserInfoDTO dto = userService.getUserCache(userId);
        Long infoCompleteStatus = dto.getInfoCompleteState();
        if (null == infoCompleteStatus) {
            User user = userMapper.selectByPrimaryKey(userId);
            infoCompleteStatus = user.getInfoCompleteState();
            if (null == infoCompleteStatus) {
                return false;
            }
        }
        int completeCount = InfoCombineStateEnum.getInfoCompleteStatus(infoCompleteStatus);
        return 10 - completeCount < 2;
    }

    @Override
    public Message saveCustomTag(CustomTagParam customTagParam) {

        //标签长度检测
        if (CharUtil.getWordCount(customTagParam.getLabel()) > 10) {
            return Message.build(false, "标签超过5个长度");
        }

        //判断标签是否包含敏感词汇
        if (userNewsIntegrationService.verifySensitiveWord(customTagParam.getLabel())) {
            return Message.build(false, "标签含有敏感词词，请重新输入");
        }

        Message m = preExecute(customTagParam.getUserId());
        if (!m.isSuccess()) {
            return m;
        }

        User userEntity = m.getParam("user");

        //因为新创建的标签需要加在标签列表的最前面，所以需要获取标签的最小的顺序号
        List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(customTagParam.getUserId());

        //判断已选标签是否小于15，否则抛出异常“标签不能超过15个哦”
        long count = userTagList.stream().filter(e -> 0 == e.getDeleteFlag()).count();
        if (count >= 15) {
            return Message.build(false, "标签不能超过15个哦");
        }

        int minOder = 0;

        //获得最小的顺序号
        if (!CollectionUtils.isEmpty(userTagList)) {
            minOder = userTagList.get(0).getOrder();
        } else {
            userTagList = new ArrayList<>();
        }

        //构建标签实体并入库
        UserTag userTag = UserTag.buildUserTag(customTagParam.getUserId(), minOder - 1, customTagParam.getLabel(), nextSequence());
        this.userTagMapper.insert(userTag);

        userTagList.add(0, userTag);

        //补全用户完善资料程度
        long infoCompleteStatus = userEntity.getInfoCompleteState();
        if (!BitOperatorUtil.getBitAsBoolean(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex())) {
            infoCompleteStatus = BitOperatorUtil.setBitToLong(infoCompleteStatus, InfoCombineStateEnum.USER_TAG_COMPLETE.getIndex());
            userEntity.setInfoCompleteState(infoCompleteStatus);
            userMapper.updateUserInfoCompleteStatus(new UserInfoComplete(customTagParam.getUserId(), infoCompleteStatus));
        }

        afterExecute(userEntity, false, true, false, userTagList);

        //返回新增标签的实体，方便客户端回显
        return Message.build(true).addParam("tag", userTagService.convertUserTag2UserTagDto(userTag));
    }

    @Override
    public Boolean getCompleteInformationWindow(Long userId) {
        logger.debug("获取用户完善个人资料弹窗，userId:[{}]", userId);
        //用户资料完成度
        UserService userService = SpringContextHolder.getBean(UserService.class);
        UserInfoDTO userFromRedisDB = userService.getUserCache(userId);
        int status = InfoCombineStateEnum.getInfoUnCompleteStatus(userFromRedisDB.getInfoCompleteState());
        //当用户资料未完成项>=2时 ,需要弹出该弹窗
        if (status >= 2) {
            KeyGenerator cacheKey = RedisConfig.USER_CACHE_POPUP.copy().appendKey(userId);
            //初始化缓存map，比对长度后增量
            Map<String, PopCache> cacheMap = redisHashMapAdapter.entries(cacheKey, PopCache.class);
            PopCache popCache = cacheMap.get(KEY);
            if (Objects.isNull(popCache)) {
                popCache = new PopCache();
            }
            //上次弹窗的时间
            Long popTime = popCache.getParam(LAST_MODIFY_TIME);
            long timeMillis = System.currentTimeMillis();

            //缓存时间为空或者当前时间与上次弹窗时间相隔7天,需弹窗
            if (popTime == null || (timeMillis - popTime) > DateUtils.WEEK_MILLISECOND) {
                popCache.setPopUp(Boolean.TRUE);
                popCache.addParam(LAST_MODIFY_TIME, System.currentTimeMillis());
                //兼容首页次数弹窗
                Integer num = popCache.getParam("num");
                if (Objects.isNull(num)) {
                    num = 1;
                } else {
                    num += 1;
                }
                popCache.addParam("num", num);
                redisHashMapAdapter.put(cacheKey, KEY, popCache);
                logger.debug("弹出个人资料引导弹窗:[{}]", JSONObject.toJSONString(popCache));
                return Boolean.TRUE;
            } else {
                return Boolean.FALSE;
            }
        }
        return Boolean.FALSE;
    }

    @Override
    public Boolean orderPrivacyChange(OrderPrivacyChangeParam param) {
        userService.updateUserJudgeMarker(param.getUserId(), LocalNewsUserJudgeMarkerEnum.ORDER_PRIVACY,
                // 如果change type 为0则是关闭隐私开关
                Objects.equals(param.getChangeType(), 0));
        return true;
    }

    /**
     * 年龄段组装
     */
    private String getGeneration(UserInfoDTO user) {
        String generationStr = "";
        if (null != user.getBirthday()) {
            LocalDate birthday = user.getBirthday().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            String gen = userService.getGeneration(birthday);
            if (!org.springframework.util.StringUtils.isEmpty(gen)) {
                generationStr = gen;
            }
        }
        return generationStr;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
