package com.bxm.localnews.user.constant;

import com.bxm.localnews.user.dto.UserAccountDTO;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;

import java.math.BigDecimal;

import static org.apache.commons.lang.StringUtils.isBlank;

/**
 * 现金流水归类枚举
 * 此枚举定义的类型在用户收益列表中使用，通过include和exclude控制该汇总类型
 * 通过AccountField控制该分类显示的金额对应账号中的金额
 *
 * @author liujia
 * @date 2020/05/09 15:47
 */
public enum CashFlowClassifyEnum {
    ALL("全部收益",
            null,
            null,
            UserAccountDTO::getSettledCash),

    REBATE("购物收益",
            new CashFlowTypeEnum[]{CashFlowTypeEnum.REBATE_CASH},
            null,
            UserAccountDTO::getRebateCash),

    INVITE_VIP("开卡收益",
            new CashFlowTypeEnum[]{CashFlowTypeEnum.INVITE_VIP},
            null,
            UserAccountDTO::getInviteVipCash),

    OTHER("其他收益",
            null,
            new CashFlowTypeEnum[]{CashFlowTypeEnum.REBATE_CASH, CashFlowTypeEnum.INVITE_VIP},
            UserAccountDTO::getOtherCash),
    ;

    /**
     * 分类显示的标签名字
     */
    private String label;

    /**
     * 当前归类包含哪几种现金收益类型
     */
    private CashFlowTypeEnum[] include;

    /**
     * 当前归类不包含哪几种现金收益类型
     */
    private CashFlowTypeEnum[] exclude;

    /**
     * 当前分类需要从账号的某一个或某几个字段中获取值（可能会有字段之间的计算）
     */
    private AccountField accountField;

    CashFlowClassifyEnum(String label,
                         CashFlowTypeEnum[] include,
                         CashFlowTypeEnum[] exclude,
                         AccountField accountField) {
        this.label = label;
        this.include = include;
        this.exclude = exclude;
        this.accountField = accountField;
    }

    public BigDecimal getAccountField(UserAccountDTO userAccount) {
        return this.accountField.get(userAccount);
    }

    public String getLabel() {
        return label;
    }

    public CashFlowTypeEnum[] getInclude() {
        return include;
    }

    public CashFlowTypeEnum[] getExclude() {
        return exclude;
    }

    public static CashFlowClassifyEnum of(String name) {
        if (isBlank(name)) {
            return null;
        }

        for (CashFlowClassifyEnum value : CashFlowClassifyEnum.values()) {
            if (value.name().equalsIgnoreCase(name)) {
                return value;
            }
        }
        return null;
    }

    interface AccountField {
        BigDecimal get(UserAccountDTO userAccount);
    }
}
