package com.bxm.localnews.user.invite.processor;

import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.attribute.UserAttributeService;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.UserInviteBindDTO;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.integration.BizLogIntegrationService;
import com.bxm.localnews.user.invite.InviteProcessorContext;
import com.bxm.localnews.user.invite.InviteRecordService;
import com.bxm.localnews.user.invite.InviteTypeProcessor;
import com.bxm.localnews.user.invite.UserInviteService;
import com.bxm.localnews.user.invite.bind.BindInviteManager;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.AccountCashParam;
import com.bxm.localnews.user.vo.InviteRecord;
import com.bxm.localnews.user.vo.UserInviteHistoryBean;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;

/**
 * 抽象的邀请类型处理器
 *
 * @author liujia
 * @date 2020/05/11 15:37
 */
@Slf4j
public abstract class AbstractInviteTypeProcessor implements InviteTypeProcessor {
    /**
     * 显示标题的默认长度，防止超长，标题在客户端显示不正确
     */
    int defaultColumnWidth = 80;

    String defaultInviteMsg = "的其他分享";

    @Resource
    UserService userService;

    @Resource
    UserAccountService userAccountService;

    @Resource
    InviteRecordService inviteRecordService;

    @Resource
    BizLogIntegrationService bizLogIntegrationService;

    @Resource
    UserInviteService userInviteService;

    @Resource
    UserAttributeService userAttributeService;

    @Autowired
    private BindInviteManager bindInviteManager;

    @Override
    public final Message afterInvited(InviteProcessorContext context) {
        log.info("邀请成功，邀请记录信息：{}", context);
        fillContext(context);

        //邀请结果写入到邀请成功记录表--可能存在已经绑定，所以会save返回false
        UserInviteBindDTO bean = convertInviteBind(context);
        Message message = Message.build(true);
        //绑定邀请关系
        if(null == bindInviteManager.bindInvite(bean)){
            message.setSuccess(false);
        }
        if (!message.isSuccess()) {
            return message;
        }
        //触发不同邀请的后续处理回调逻辑
        message.append(callback(context));

        //如果邀请会提供现金奖励，则进行邀请现金奖励状态的变更
        //[3.4.0] 暂时用不上，后续可能还会用上，但是逻辑需要修改，所以这里暂时注释掉。
//        if (AwardTypeEnum.CASH.name().equals(context.getRecord().getAwardType())) {
//            changeInviteAwardState(context);
//        }
        return message;
    }


    private void fillContext(InviteProcessorContext context) {
        if (context.getInvitedUser() == null) {
            UserInfoDTO inviteUser = userService.getUserCache(context.getRecord().getInvitedUserId());
            context.setInvitedUser(inviteUser);
        }
    }

    /**
     * 更新邀请人的赏金状态
     */
    private void changeInviteAwardState(InviteProcessorContext context) {
        InviteRecord inviteRecord = context.getRecord();

        //邀请关系被解除
        if (!InviteRecordStateEnum.ACCEPT_INVITE.name().equals(inviteRecord.getInviteState())) {
            log.info("当前邀请人[{}]与受邀人[{}]的状态处于[{}]------不予处理",
                    inviteRecord.getUserId(),
                    inviteRecord.getInvitedUserId(),
                    inviteRecord.getInviteState());
        }

        if (context.isEffective()) {
            //如果是有效邀请，则将邀请人的临时金额转变为永久金额
            AccountCashParam param = new AccountCashParam(
                    inviteRecord.getUserId(),
                    CashEnum.STABILIZE_CASH.name(),
                    false,
                    inviteRecord.getAward(),
                    inviteRecord.getId(), CashFlowTypeEnum.PROMOTE_BOUNTY.name(),
                    CashFlowConvertTypeEnum.TEMP_TO_STABILIZE.name());
            userAccountService.transferCash(param);

            //更新邀请状态为已登陆
            inviteRecord.setInviteState(InviteRecordStateEnum.LOGIN_APP.getName());
            inviteRecord.setEffectTime(null);
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());

            //增加邀请成功日志(历史逻辑迁移)
            Integer platform = context.getBasicParam() == null ? null : context.getBasicParam().getPlatform();

            bizLogIntegrationService.inviteSuccessed(inviteRecord.getUserId(),
                    inviteRecord.getInvitedUserId(),
                    platform);
        } else {
            //如果是无效邀请，则将临时赏金更新为无效赏金
            AccountCashParam accountCashParam = new AccountCashParam();
            accountCashParam.setUserId(inviteRecord.getUserId());
            accountCashParam.setCashType(CashEnum.TEMP_CASH.name());
            accountCashParam.setAddTotal(false);
            accountCashParam.setCashFlowType(CashFlowTypeEnum.INVALID_BOUNTY.name());
            accountCashParam.setCash(inviteRecord.getAward().negate());
            accountCashParam.setRelationId(inviteRecord.getId());
            userAccountService.addCash(accountCashParam);

            //更新邀请状态为无效
            inviteRecord.setInviteState(InviteRecordStateEnum.OVERDUE_INVALID.name());
            inviteRecord.setAwardType(AwardTypeEnum.CASH.name());
            inviteRecord.setStatus(InviteRecordStatusEnum.INVALID.getStatus());
        }
        inviteRecordService.updateInviteRecord(inviteRecord);
    }


    private UserInviteBindDTO convertInviteBind(InviteProcessorContext context) {
        UserInviteBindDTO bean = new UserInviteBindDTO();
        bean.setUserId(context.getRecord().getInvitedUserId());
        if (context.getInvitedUser() != null) {
            bean.setInviteUserId(context.getInvitedUser().getInviteUserId());
        } else {
            bean.setInviteUserId(context.getRecord().getUserId());
        }
        bean.setInviteBindMethodEnum(InviteBindMethodEnum.LOGIN_APP);
        return bean;
    }

    private UserInviteHistoryBean convert(InviteProcessorContext context) {
        UserInviteHistoryBean history = new UserInviteHistoryBean();
        history.setUserId(context.getRecord().getInvitedUserId());
        history.setCreateTime(context.getRecord().getCreateTime());

        if (context.getInvitedUser() != null) {
            history.setInviteUserId(context.getInvitedUser().getInviteUserId());
            history.setInviteUserNickname(context.getRecord().getInvitedUserName());
            history.setInviteHeadImg(context.getRecord().getInvitedUserImg());
            history.setRelationId(context.getInvitedUser().getRelationId());
            if (null != context.getInvitedUser().getIsVip()) {
                history.setActiveVip(context.getInvitedUser().getIsVip().byteValue());
            }
            if (null != context.getRecord()) {
                history.setType(context.getRecord().getType());
            }
        } else {
            history.setInviteUserId(context.getRecord().getUserId());
            history.setInviteUserNickname(context.getRecord().getInvitedUserName());
            history.setInviteHeadImg(context.getRecord().getInvitedUserImg());
            history.setType(context.getRecord().getType());
        }

        //从实现类获取具体的标题，不同邀请类型文案和获取方式不同
        String title = obtainTitle(context);
        history.setRelationTitle(StringUtils.substring(title, 0, 200));

        return history;
    }

    /**
     * 不同邀请类型邀请成功后的回调处理逻辑
     *
     * @param context 邀请记录上下文信息
     * @return 邀请结果
     */
    abstract Message callback(InviteProcessorContext context);
}
