package com.bxm.localnews.user.login;

import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.vo.LoginInfo;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserAuth;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import org.springframework.stereotype.Service;

@Service
public class UserParamCheckService {

    /**
     * 账号绑定参数必要校验
     */
    public boolean bindingCheckParam(LoginInfo loginInfo) {
        //校验绑定类型
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
            && AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
            && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
            && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }
        //登录标识不能为空
        if (StringUtils.isBlank(loginInfo.getLoginName())) {
            return false;
        }
        //绑定手机号标识
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
            && StringUtils.isEmpty(loginInfo.getCode())) {
            return false;
        }
        //绑定微信，则微信头像和昵称字段不能为空
        return AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
            || !StringUtils.isEmpty(loginInfo.getHeadImg())
            || !StringUtils.isEmpty(loginInfo.getNickname());
    }

    /**
     * 检查定位接口参数
     */
    public Message checkPostLocationParam(String code, Long userId) {
        if (StringUtils.isEmpty(code)) {
            return Message.build(false, "定位时无法提供地区编码，将提供默认定位北京(定位失败)!!!:code:{" + code + "},userId:{" + userId + "}");
        }
        if ((code.length() != 6 && code.length() != 12) || null == userId) {
            return Message.build(false, "定位时参数错误(地区编码位数问题或用户id为空):code:{" + code + "},userId:{" + userId + "}");
        }
        return Message.build(true);
    }

    /**
     * 只检查了当前要绑定第三方的用户有没有重复绑定，但是忽视了别的用户也可能已经绑定了第三方
     */
    public String checkBindRepeat(Byte loginInfoType, User user) {
        String message = "";
        if (AppConst.LOGIN_TYPE.PHONE == loginInfoType && StringUtils.isNotBlank(user.getPhone())) {
            message = "请勿重复绑定手机号";
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfoType && StringUtils.isNotBlank(user.getWeixin())) {
            message = "请勿重复绑定微信账号";
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfoType && StringUtils.isNotBlank(user.getWeibo())) {
            message = "请勿重复绑定微博账号";
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfoType && StringUtils.isNotBlank(user.getQq())) {
            message = "请勿重复绑定QQ账号";
        }
        return message;
    }

    /**
     * 检查H5注册参数
     */
    public boolean checkH5RegisterParam(LoginInfo loginInfo) {
        return StringUtils.isNotBlank(loginInfo.getLoginName())
            && StringUtils.isNotBlank(loginInfo.getCode())
            && AppConst.LOGIN_TYPE.PHONE == loginInfo.getType();
    }

    /**
     * 判断是否是密码登录
     */
    public boolean checkVerifyPwd(LoginInfo loginInfo) {
        return AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
            && StringUtils.isNotBlank(loginInfo.getPassword())
            && !loginInfo.getResetPwd();
    }

    /**
     * 验证账号密码登录
     */
    public Message verifyPwd(String password, UserAuth userAuth) {
        String encodePwd = MD5Util.hgmd5(password);
        if (null == userAuth) {
            //若用户账号不存在
            return Message.build(false, "手机号码未注册，请前往注册");
        }
        if (!userAuth.getCredential().equals(encodePwd)) {
            //密码不一致则直接返回
            return Message.build(false, "账号或密码错误");
        }
        return Message.build(true);
    }

    /**
     * 检查第三方是否已经被绑定
     */
    public Message checkRepeatBind(User user, LoginInfo loginInfo) {
        if (null != user) {
            if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType() && StringUtils.isNotEmpty(user.getPhone())) {
                return Message.build(false, "手机号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType() && StringUtils.isNotEmpty(user.getWeixin())) {
                return Message.build(false, "微信号已被绑定");
            } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType() && StringUtils.isNotEmpty(user.getWeibo())) {
                return Message.build(false, "微博已被绑定");
            } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType() && StringUtils.isNotEmpty(user.getQq())) {
                return Message.build(false, "QQ号已被绑定");
            }
        }
        return Message.build(true);
    }

    /**
     * 登录参数验证
     */
    public boolean checkParam(LoginInfo loginInfo) {
        //校验登录类型
        if (AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
            && AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
            && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
            && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }

        //登录标识不能为空
        if (StringUtils.isBlank(loginInfo.getLoginName())) {
            return false;
        }

        //手机号标识登录，则验证码和密码不能为空
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()
            && StringUtils.isEmpty(loginInfo.getPassword())
            && StringUtils.isEmpty(loginInfo.getCode())) {
            return false;
        }

        //微信登录，则微信头像和昵称字段不能为空
        return true;
    }

    /**
     * 注册绑定参数验证
     */
    public boolean bindingAndRegisterCheckParam(LoginInfo loginInfo) {
        //校验登录类型
        if (AppConst.LOGIN_TYPE.WEIXIN != loginInfo.getType()
            && AppConst.LOGIN_TYPE.QQ != loginInfo.getType()
            && AppConst.LOGIN_TYPE.PHONE != loginInfo.getType()
            && AppConst.LOGIN_TYPE.WEIBO != loginInfo.getType()) {
            return false;
        }

        //验证必填地段
        if (StringUtils.isBlank(loginInfo.getLoginName())
            && StringUtils.isBlank(loginInfo.getPhone())
            && StringUtils.isBlank(loginInfo.getCode())
            && StringUtils.isBlank(loginInfo.getPassword())) {
            return false;
        }

        //微信登录，则微信头像和昵称字段不能为空
        return true;
    }

}
