package com.bxm.localnews.user.login.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.attribute.NoviceTaskRecordService;
import com.bxm.localnews.user.attribute.UserPersonalInfoService;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.EquipmentDTO;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.LevelEnum;
import com.bxm.localnews.user.enums.TaskEnum;
import com.bxm.localnews.user.exception.UserRepeatCreateException;
import com.bxm.localnews.user.integration.*;
import com.bxm.localnews.user.login.UserAuthService;
import com.bxm.localnews.user.login.UserRegistrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.properties.BizConfigProperties;
import com.bxm.localnews.user.support.ChannelService;
import com.bxm.localnews.user.support.ExtendedService;
import com.bxm.localnews.user.vo.LoginInfo;
import com.bxm.localnews.user.vo.LoginMeta;
import com.bxm.localnews.user.vo.User;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.service.BaseService;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Service
public class UserRegistrationServiceImpl extends BaseService implements UserRegistrationService {

    @Resource
    private UserAuthService userAuthService;

    @Resource
    private NoviceTaskRecordService noviceTaskRecordService;

    @Resource
    private UserAccountService userAccountService;

    @Resource
    private UserMapper userMapper;

    @Resource
    private BizConfigProperties bizConfigProperties;

    @Resource
    private ChannelService channelService;

    @Resource
    private MissionIntegrationService missionIntegrationService;

    @Resource
    private BizLogIntegrationService bizLogIntegrationService;

    @Resource
    private UserNewsIntegrationService userNewsIntegrationService;

    @Resource
    private AdvertIntegrationService advertIntegrationService;

    @Resource
    private UserService userService;

    @Resource
    private ExtendedService extendedService;

    @Resource
    private UserPersonalInfoService userPersonalInfoService;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private BizIntegrationService bizIntegrationService;

    @Resource
    private ScheduledThreadPoolExecutor scheduledThreadPoolExecutor;

    @Override
    @Retryable(value = RetryException.class, backoff = @Backoff(300))
    public Json<LoginMeta> register(LoginInfo loginInfo, BasicParam basicParam, String firstOpenType) {
        // 根据设备号查询
        EquipmentDTO equipmentDTO = bizIntegrationService.getDevice(basicParam.getDevcId());

        if (equipmentDTO == null || equipmentDTO.getId() == null) {
            return ResultUtil.genFailedResult("无法获取设备号，请重试");
        }

        basicParam.setDevcId(equipmentDTO.getId());
        User user = this.userMapper.findByEquipment(equipmentDTO.getId());

        // 没有该设备号的用户
        if (null == user) {
            //如果为空，则创建一条游客用户
            try {
                user = this.addUser(loginInfo, basicParam, AppConst.USER_STATE.VISITOR);
            } catch (UserRepeatCreateException e) {
                logger.warn("重复创建用户，进行重试，请求参数：{}", loginInfo);
                throw new RetryException("用户重复创建，重新请求一次");
            }

            //初始化用户默认分类
            this.logger.debug(LogMarker.BIZ, "用户静默注册生成默认频道列表，loginInfo:[{}]", loginInfo);
            this.userNewsIntegrationService.initUserNewsImage(user.getId(), null);
        }

        user.setSex(user.getSex() == null ? 0 : user.getSex());
        UserInfoDTO dto = userService.putUserToRedis(user);
        return ResultUtil.genSuccessResult(new LoginMeta(dto));
    }

    @Override
    public User addUser(LoginInfo loginInfo, BasicParam basicParam, Byte userType) throws UserRepeatCreateException {
        //新增用户信息
        logger.info("添加新用户信息:[{}]", JSON.toJSONString(loginInfo));

        String lockKey = nextSequence().toString();
        String lockResource = buildLockResource(loginInfo);

        if (distributedLock.lock(lockResource, lockKey)) {
            User user = this.saveUser(loginInfo, basicParam, userType);

            //判断用户是否是游客
            if (AppConst.USER_STATE.VISITOR != userType) {
                //添加授权登录认证信息(同步)
                this.userAuthService.saveAuth(user.getId(), loginInfo);
                //生成账户(同步)
                this.userAccountService.createUserAccount(user.getId());
                //生成新手任务(同步)
                this.noviceTaskRecordService.batchAdd(user.getId());

                //延迟3s
                scheduledThreadPoolExecutor.schedule(() -> {
                    //完成首次登陆任务(异步)
                    this.missionIntegrationService.asyncCompleteTask(user.getId(), TaskEnum.TASK_FIRST_LOGIN, String.valueOf(user.getId()));
                    //如果是邀请注册,则添加注册来源,并完成任务(异步)
                    this.saveRegSourceInfo(loginInfo, user, basicParam.getChnl());
                    //回调广告接口(异步)
                    this.advertIntegrationService.triggerAndroidId(loginInfo, basicParam);
                    //数据埋点(异步)
                    this.bizLogIntegrationService.createUser(user, basicParam, null);
                }, 3, TimeUnit.SECONDS);

            }
            logger.debug("添加用户成功：[{}]", JSON.toJSONString(user));

            distributedLock.unlock(lockResource, lockKey);

            return user;
        }

        throw new UserRepeatCreateException();
    }

    private String buildLockResource(LoginInfo loginInfo) {
        if (null == loginInfo || null == loginInfo.getType()) {
            return String.valueOf(nextId());
        }

        if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            return loginInfo.getLoginName();
        } else if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            return loginInfo.getPhone();
        }
        return loginInfo.getLoginName();
    }

    /**
     * 生成用户并入库
     *
     * @param loginInfo  登录信息
     * @param basicParam 基础参数
     * @param userType   游客 or 正常用户
     */
    private User saveUser(LoginInfo loginInfo, BasicParam basicParam, Byte userType) {
        User user = new User();
        user.setRegisterClient(String.valueOf(basicParam.getPlatform()));
        //不维护年龄字段，因为这个字段每年都在变
        user.setAge(0);
        user.setSex(loginInfo.getSex() == null ? 0 : loginInfo.getSex());
        user.setEquipment(basicParam.getDevcId());
        user.setPhone(loginInfo.getPhone());
        user.setRegisteredAddress(loginInfo.getRegisteredaddress());
        user.setState(userType);
        user.setLevel(LevelEnum.INITIAL_USER.getType());
        user.setIsNew(AppConst.IS_NEW_USER.NEW_USER);
        user.setIsTempNickName((byte) 1);
        user.setIsTempHeadImg((byte) 1);
        user.setChannelId(this.channelService.getChannelId(basicParam.getChnl()));
        user.setRegIp(loginInfo.getRegIp());
        user.setLastLoginIp(loginInfo.getLastLoginIp());
        user.setNewbieGuideFlag((byte) 0);
        user.setCreateTime(new Date());

        if (AppConst.USER_STATE.VISITOR != userType) {
            this.setLoginName(user, loginInfo);
            user.setInfoCompleteState(userPersonalInfoService.initUserInfoCompleteStatus(user));
        }

        int count = this.userService.createUser(user, basicParam);

        if (count > 0) {
            //获取到用户id才能设置token
            User loginUser = new User();
            loginUser.setId(user.getId());
            loginUser.setToken(JwtTokenUtil.generateToken(user.getId()));
            loginUser.setRefreshtoken(JwtTokenUtil.generateToken(user.getId()));
            loginUser.setExpiretime(JwtTokenUtil.generateExpirationDate().getTime());

            user.setToken(loginUser.getToken());
            user.setRefreshtoken(loginUser.getRefreshtoken());
            user.setExpiretime(loginInfo.getExpiretime());
            this.userMapper.updateByPrimaryKeySelective(loginUser);
        }

        return user;
    }

    /**
     * 更改地理位置
     */
    private void changeUserLocation(LoginInfo loginInfo, User user, UserMapper userMapper) {
        if (null != loginInfo.getId()) {
            User originalUser = userMapper.selectByPrimaryKey(loginInfo.getId());
            if (null != originalUser && null != originalUser.getLocationCode() && null != originalUser.getLocationName()) {
                user.setLocationCode(originalUser.getLocationCode());
                user.setLocationName(originalUser.getLocationName());
            }
        }
    }

    /**
     * 设置登陆名称
     */
    private void setLoginName(User user, LoginInfo loginInfo) {
        if (AppConst.LOGIN_TYPE.PHONE == loginInfo.getType()) {
            user.setPhone(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.QQ == loginInfo.getType()) {
            user.setQq(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIBO == loginInfo.getType()) {
            user.setWeibo(loginInfo.getLoginName());
        } else if (AppConst.LOGIN_TYPE.WEIXIN == loginInfo.getType()) {
            user.setWeixin(loginInfo.getLoginName());
            user.setNickname(
                    user.getNickname() == null ? EmojiCodeParser.replaceSoftbankEmoji(loginInfo.getNickname()) : user.getNickname());
            user.setHeadImg(
                    user.getHeadImg() == null ? extendedService.uploadHeadImg(loginInfo.getHeadImg(), user.getId()) : user.getHeadImg());
        } else {
            throw new RuntimeException("参数错误");
        }
    }

    /**
     * 如果是邀请注册,则添加注册来源,并完成任务
     */
    private void saveRegSourceInfo(LoginInfo loginInfo, User user, String chnl) {
        Long inviteCode = loginInfo.getInviteCode();
        if (null == inviteCode) {
            return;
        }

        this.logger.debug(LogMarker.BIZ, "用户邀请记录,loginInfo:[{}],user:[{}],chnl:[{}]", loginInfo, user, chnl);

        bizLogIntegrationService.inviteSuccessed(inviteCode, user.getId(),
                null == user.getRegisterClient() ? null : Integer.valueOf(user.getRegisterClient()));

        missionIntegrationService.asyncCompleteTask(inviteCode, TaskEnum.TASK_INVITED_FRIEND, user.getId().toString());
    }
}
