package com.bxm.localnews.user.login.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.InfoCombineStateEnum;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.user.account.UserAccountService;
import com.bxm.localnews.user.domain.UserAuthMapper;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.dto.*;
import com.bxm.localnews.user.enums.*;
import com.bxm.localnews.user.integration.BizIntegrationService;
import com.bxm.localnews.user.integration.LocationIntegrationService;
import com.bxm.localnews.user.integration.MissionIntegrationService;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.invite.InviteRecordService;
import com.bxm.localnews.user.invite.UserInviteService;
import com.bxm.localnews.user.invite.bind.BindInviteManager;
import com.bxm.localnews.user.login.UserAuthService;
import com.bxm.localnews.user.login.UserLoginHistoryService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.login.protocal.ProtocalProcesser;
import com.bxm.localnews.user.model.MerchantUserDTO;
import com.bxm.localnews.user.param.*;
import com.bxm.localnews.user.properties.HomePageConfigProperties;
import com.bxm.localnews.user.properties.UserProperties;
import com.bxm.localnews.user.support.ExtendedService;
import com.bxm.localnews.user.support.impl.RiskRecordServiceImpl;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vip.relation.MerchantRelationService;
import com.bxm.localnews.user.vo.*;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.tools.Validater;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.time.LocalDate;
import java.util.*;
import java.util.concurrent.TimeUnit;

import static com.bxm.localnews.user.constant.RedisConfig.USER_CACHE_INFO;
import static com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum.VER_3_5_0;
import static com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum.VER_3_6_0;
import static com.bxm.localnews.user.enums.TaskEnum.TASK_IMPROVE_INFO;
import static java.util.Objects.nonNull;
import static org.apache.commons.lang3.StringUtils.*;
import static org.springframework.util.CollectionUtils.isEmpty;

@Service("userService")
@RefreshScope
public class UserServiceImpl extends BaseService implements UserService {

    @Resource
    private UserAuthService userAuthService;

    @Resource
    private UserAccountService userAccountService;

    @Resource
    private UserMapper userMapper;

    @Resource
    private UserVipService userVipService;

    @Resource
    private UserAuthMapper userAuthMapper;

    @Resource
    private UserProperties userProperties;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private MissionIntegrationService missionIntegrationService;

    @Resource
    private InviteRecordService inviteRecordService;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private UserInviteService userInviteService;

    @Resource
    private PushMsgIntegrationService pushMsgIntegrationService;

    @Resource
    private HomePageConfigProperties homePageConfigProperties;

    @Resource
    private ExtendedService extendedService;

    @Resource
    private RiskRecordServiceImpl riskRecordService;

    @Resource
    private BizIntegrationService bizIntegrationService;

    @Autowired
    private ProtocalProcesser protocalProcesser;

    @Autowired
    private MerchantRelationService merchantRelationService;

    private KeyGenerator buildUserCacheKey(Long userId) {
        return USER_CACHE_INFO.copy().appendKey(userId);
    }

    @Autowired
    private BindInviteManager bindInviteManager;

    @Autowired
    private UserLoginHistoryService userLoginHistoryService;


    @Override
    public int createUser(User user, BasicParam basicParam) {

        if (Objects.isNull(user.getJudgeMarker())) {
            user.setJudgeMarker(0L);
        }

        // 3.5.0增加标志位
        user.setJudgeMarker(BitOperatorUtil.setBitToLong(user.getJudgeMarker(), VER_3_5_0.getIndex()));
        // 3.6.0 标志位
        user.setJudgeMarker(BitOperatorUtil.setBitToLong(user.getJudgeMarker(), VER_3_6_0.getIndex()));

        return this.userMapper.insertSelective(user);
    }

    @Override
    public UserInfoDTO getUserCache(Long userId) {
        UserInfoDTO userDTO = this.redisStringAdapter.get(buildUserCacheKey(userId), UserInfoDTO.class);

        if (null == userDTO) {
            userDTO = this.loadUserToRedis(userId);
        } else if (UserInfoDTO.CURRENT_VERSION > userDTO.getV()) {
            if (logger.isDebugEnabled()) {
                logger.debug("缓存对象是历史版本，加载新的用户缓存，用户ID：{},缓存中的版本:{},当前版本:{}",
                        userId,
                        userDTO.getV(),
                        UserInfoDTO.CURRENT_VERSION);
            }
            userDTO = this.loadUserToRedis(userId);
        }

        return userDTO;
    }

    @Override
    public void removeUserCache(Long userId) {
        redisStringAdapter.remove(buildUserCacheKey(userId));
    }

    /**
     * 从数据库中查找用户信息，并更新至redis中
     */
    @Override
    public UserInfoDTO loadUserToRedis(Long userId) {
        //从数据库中查找用户
        User user = this.userMapper.selectByPrimaryKey(userId);
        if (null == user) {
            UserInfoDTO emptyUser = new UserInfoDTO();
            emptyUser.setId(userId);

            this.redisStringAdapter.set(buildUserCacheKey(userId), emptyUser, 3600);
            return null;
        }

        return putUserToRedis(user);
    }

    @Override
    public UserInfoDTO putUserToRedis(User user) {
        logger.debug("更新用户缓存：[{}]", JSONObject.toJSON(user));
        //封装返回实体
        UserInfoDTO dto = this.convertUserToDTO(user);

        //处理职业信息的展示
        dto.setIndustry(getIndustryInfo(user));

        //设置默认个人简介
        if (isNotEmpty(user.getPersonalProfile())) {
            dto.setIsDefaultPersonalProfile(Boolean.FALSE);
        } else {
            dto.setIsDefaultPersonalProfile(Boolean.TRUE);
            dto.setPersonalProfile(homePageConfigProperties.getDefaultPersonalProfile());
        }

        //家乡填充
        if (null != user.getHometownCode()) {
            LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(user.getHometownCode());
            if (null != locationByGeocode) {
                dto.setHometownName(locationByGeocode.getFullName());
            }
        }

        //当前居住城市填充
        if (null != user.getLocationCode()) {
            LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(user.getLocationCode());
            if (null != locationByGeocode) {
                dto.setLocationName(locationByGeocode.getFullName());
            }
        }

        //添加vip
        dto.setIsVip(userVipService.isVip(user.getId()) ? 1 : 0);

        this.redisStringAdapter.set(buildUserCacheKey(user.getId()), dto);

        return dto;
    }

    private String getIndustryInfo(User user) {
        //处理职业信息的展示
        StringBuilder industryBuilder = new StringBuilder();
        //公司和职位不为空（这几个判断有点蠢，待优化）
        if (StringUtils.isNotBlank(user.getJobTitle()) && StringUtils.isNotBlank(user.getCompany())) {
            industryBuilder.append(user.getCompany()).append(user.getJobTitle());
        }
        //公司和职位为空
        if (isBlank(user.getJobTitle()) && isBlank(user.getCompany()) && StringUtils
                .isNotBlank(user.getJobCategoryName())) {
            industryBuilder.append(user.getJobCategoryName());
        }
        //职业不为空，公司为空
        if (StringUtils.isNotBlank(user.getJobTitle()) && isBlank(user.getCompany()) && StringUtils
                .isNotBlank(user.getJobCategoryName())) {
            industryBuilder.append(user.getJobTitle());
        }
        //职业为空，公司不为空
        if (isBlank(user.getJobTitle()) && StringUtils.isNotBlank(user.getCompany())) {
            industryBuilder.append(user.getCompany());
        }
        if (industryBuilder.length() > 0) {
            return industryBuilder.toString();
        }
        return null;
    }

    @Override
    public boolean checkUserExistByPhone(String phone, Integer flag) {
        User user = userMapper.findByPhone(phone);

        if (null == user) {
            return false;
        } else {
            // 如果当前用户的手机号码和想绑定的手机号码一致，也允许绑定
            // 解决3.4.0版本之前安卓本地存储失效导致的错误提示用户绑定弹窗出现
            // 通过此逻辑，实现用户手机号码的重复绑定
            if (StringUtils.equals(user.getPhone(), phone)) {
                logger.warn("用户尝试绑定相同的手机号码，用户ID:{}，手机号码：{}", user.getId(), phone);
                return false;
            }

            if (flag == 1) {
                UserAuth userAuth = userAuthMapper.selectByUserId((byte) 3, user.getId());
                return userAuth != null;
            } else {
                return true;
            }
        }
    }

    @Override
    public boolean checkUserExistByPhone(String phone) {
        return nonNull(userMapper.findByPhone(phone));
    }

    @Override
    public UserInfoDTO convertUserToDTO(User user) {
        UserInfoDTO dto = new UserInfoDTO();
        BeanUtils.copyProperties(user, dto);
        if (isBlank(dto.getNickname())) {
            dto.setNickname(StringUtils.hideMobile(user.getPhone()));
        }
        // 设置简称
        if (isNotBlank(user.getLocationName())) {
            dto.setLocationShortName(user.getLocationName());
        }
        if (isBlank(dto.getHeadImg())) {
            dto.setHeadImg(userProperties.getDefaultHeadImgUrl());
        }
        //3.0.0 [康文静]取消昵称修改次数限制，这里强制设置为可以编辑
        dto.setIsTempNickName((byte) 1);
        return dto;
    }

    @Override
    public Json<UserInfoDTO> refreshToken(Long userId, String refreshToken) {
        User user = this.userMapper.getUserByRefreshToken(userId, refreshToken);
        if (null == user) {
            return ResultUtil.genFailedResult(RespCode.UNAUTHORIZED, "登录失效");
        }
        user.setExpiretime(JwtTokenUtil.generateExpirationDate().getTime());
        user.setToken(JwtTokenUtil.generateToken(userId));
        this.userMapper.updateByPrimaryKeySelective(user);
        UserInfoDTO dto = this.convertUserToDTO(user);
        return ResultUtil.genSuccessResult(dto);
    }

    @Override
    public void checkCompleteInfo(User user) {
        long infoCompleteStatus = user.getInfoCompleteState();
        logger.debug("用户完善资料程度信息:[{}]", infoCompleteStatus);

        if (InfoCombineStateEnum.values().length == InfoCombineStateEnum.getInfoCompleteStatus(infoCompleteStatus)) {
            logger.debug("用户已完善资料，调用完成资料接口:[{}]", infoCompleteStatus);
            missionIntegrationService.completeTask(user.getId(), TASK_IMPROVE_INFO, null);
        }
    }

    @Override
    public Message getWxUserInfo(TempUserParam tempUserParam, HttpServletRequest request) {
        Message message = createTempUser(tempUserParam);

        if (message.isSuccess()) {
            WxUserInfo wxUserInfo = new WxUserInfo();
            wxUserInfo.setUserId((long) message.getParamMap().get("userId"));
            wxUserInfo.setHeadImg(tempUserParam.getHeadImg());
            wxUserInfo.setUnionId(tempUserParam.getUnionId());
            wxUserInfo.setNickname(tempUserParam.getNickName());

            logger.info("创建临时用户成功:[{}]", JSONObject.toJSON(wxUserInfo));

            return Message.build(true).addParam("wxUserInfo", wxUserInfo);
        }
        return message;
    }

    @Override
    public LocationUserInfoDTO getLocationUserInfo(Long userId) {
        return userMapper.getLocationUserInfo(userId);
    }

    @Override
    public User selectByPrimaryKey(Long userId) {
        return this.userMapper.selectByPrimaryKey(userId);
    }

    @Override
    public Boolean checkUnionIdIsUser(String unionId) {
        UserAuth userAuth = userAuthMapper.selectByUserAuth((byte) 3, unionId);
        if (userAuth == null) {
            return false;
        }

        User user = userMapper.selectByPrimaryKey(userAuth.getUserId());
        return user == null || user.getState() != AppConst.USER_STATE.NOT_ACTIVE;
    }

    @Override
    public Boolean isRiskUser(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        return user.getRisk() == AppConst.USER_RISK.IS_RISK;
    }

    @Override
    public Long addWSTAppletUser() {
        User user = this.saveWSTUser();
        return user.getId();
    }

    /**
     * 创建万事通小程序默认用户
     */
    private User saveWSTUser() {
        User user = new User();
        user.setRegisterClient("5");
        //不维护年龄字段，因为这个字段每年都在变
        user.setAge(0);
        user.setSex((byte) 0);
        user.setState(AppConst.USER_STATE.NOT_ACTIVE);
        user.setLevel(LevelEnum.INITIAL_USER.getType());
        user.setIsNew(AppConst.IS_NEW_USER.NEW_USER);
        user.setIsTempNickName((byte) 1);
        user.setIsTempHeadImg((byte) 1);
        user.setNewbieGuideFlag((byte) 0);
        int count = createUser(user, null);

        if (count > 0) {
            User updateUser = new User();
            updateUser.setToken(JwtTokenUtil.generateToken(user.getId()));
            updateUser.setRefreshtoken(JwtTokenUtil.generateToken(user.getId()));
            updateUser.setExpiretime(JwtTokenUtil.generateExpirationDate().getTime());
            this.userMapper.updateByPrimaryKeySelective(updateUser);
            this.loadUserToRedis(user.getId());
        }

        return user;
    }

    @Override
    public Boolean isTempUser(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        return AppConst.USER_STATE.NOT_ACTIVE == user.getState();
    }

    @Override
    public Boolean isNormalUser(Long userId) {
        User user = userMapper.selectByPrimaryKey(userId);
        return null != user && AppConst.USER_STATE.NORMAL == user.getState();
    }

    @Override
    public Message createTempUser(TempUserParam userParam) {
        if (isBlank(userParam.getUnionId()) && null == userParam.getUserId()) {
            logger.error("创建临时用户出现错误，参数非法:[{}]", JSONObject.toJSON(userParam));
            return Message.build(false, "参数为空");
        }

        logger.info("开始创建临时用户，参数:[{}]", JSON.toJSONString(userParam));

        //1.转换实体（小弟不才，只能优化至此，望有识之士能将此流程优化清楚）
        User user = new User();
        convertTempUser(userParam, user);

        //2.但凡传了unionId的，都去查一下相应的记录，为后续操作做准备
        UserAuth unionUserAuth = null;
        if (StringUtils.isNotBlank(userParam.getUnionId())) {
            unionUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN, userParam.getUnionId());
        }

        //3.如果前端传了userId，说明用户已经授权过了，所以需要更新用户的最新的邀请关系
        User existedUser = null;
        Long userId = null;
        if (null != userParam.getUserId()) {
            existedUser = userMapper.selectByPrimaryKey(userParam.getUserId());
        } else if (StringUtils.isNotBlank(userParam.getUnionId()) && null != unionUserAuth) {
            existedUser = userMapper.selectByPrimaryKey(unionUserAuth.getUserId());
        }

        //如果用户存在则设置id
        if (null != existedUser) {
            userId = existedUser.getId();
        }

        //4.核心操作
        //如果能查到用户，且是临时用户的话
        if (null == existedUser && StringUtils.isNotBlank(userParam.getUnionId())) {
            logger.debug("根据用户id或者unionId无法查询到用户，创建新用户");
            //分布式锁保证不会重复插入
            if (distributedLock.lock(userParam.getUnionId(), userParam.getUnionId(), 1, TimeUnit.SECONDS)) {
                createUser(user, userParam);
                userId = user.getId();
                userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN, userId, userParam.getUnionId(), null);
                userAccountService.createUserAccount(userId);

            }
        }

        //存入redis库中
        this.loadUserToRedis(userId);

        //openid不为空的话，将相应的公众号标识和大转盘小程序存入库中
        afterProcessing(userParam, userId);

        //绑定邀请关系
        UserInviteBindDTO inviteBindDTO = convertUserToInviteBind(user);
        if (null == inviteBindDTO.getUserId()) {
            inviteBindDTO.setUserId(userId);
        }
        bindInviteManager.bindInvite(inviteBindDTO);

        return Message.build(true).addParam("userId", userId);
    }

    /**
     * openid不为空的话，将相应的公众号标识和大转盘小程序存入库中
     */
    private void afterProcessing(TempUserParam userParam, Long userId) {
        //openid不为空的话，将相应的公众号标识和大转盘小程序存入库中
        if (null != userParam.getOpenId()) {
            if (InviteTypeEnum.APPLETDRAW.getName().equals(userParam.getInviteType())) {
                UserAuth appletUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_OPENID, userParam.getOpenId());
                if (null == appletUserAuth || null == appletUserAuth.getUserId()) {
                    userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_OPENID, userId, userParam.getOpenId(), null);
                }
            } else {
                UserAuth weixinUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN_OPENID, userParam.getOpenId());
                if (null == weixinUserAuth || null == weixinUserAuth.getUserId()) {
                    userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN_OPENID, userId, userParam.getOpenId(), null);
                }
            }
        }
    }

    /**
     * 用户信息转换为绑定师徒关系需要的入参
     *
     * @param user ： 用户信息
     * @return ： 绑定师徒关系所需要的入参
     */
    private UserInviteBindDTO convertUserToInviteBind(User user) {
        UserInviteBindDTO userInviteBindDTO = new UserInviteBindDTO();
        userInviteBindDTO.setUser(user);
        userInviteBindDTO.setUserId(user.getId());
        userInviteBindDTO.setInviteUserId(user.getInviteUserId());
        userInviteBindDTO.setRelationId(user.getRelationId());
        userInviteBindDTO.setInviteBindMethodEnum(InviteBindMethodEnum.CREATE_TEMP_USER);
        return userInviteBindDTO;
    }

    @Override
    public Message updateTempUser(TempUserParam userParam) {
        Message message = Message.build();
        if (isBlank(userParam.getUnionId())) {
            return message;
        }

        //只有万事通小程序才会用这个
        UserAuth openUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_WST_OPENID, userParam.getOpenId());
        if (null == openUserAuth) {
            logger.warn("当前用户没有进行静默登陆，禁止创建用户");
            return message;
        }

        logger.debug("开始更新【万事通小程序】临时用户：{}", JSON.toJSONString(userParam));
        User user = userMapper.selectByUserId(openUserAuth.getUserId());
        Long userId = null;
        boolean isInsertInviteRecord = false;

        UserAuth unionUserAuth = userAuthMapper.selectByUserAuth(AppConst.LOGIN_TYPE.WEIXIN, userParam.getUnionId());
        if (null == unionUserAuth || null == unionUserAuth.getUserId()) {
            logger.info("当前用户未进入APP--------直接绑定用户, unionId: " + userParam.getUnionId());
            //分布式锁保证不会重复插入
            if (distributedLock.lock(userParam.getUnionId(), userParam.getUnionId(), 1, TimeUnit.SECONDS)) {
                //更新用户信息
                convertTempUser(userParam, user);
                user.setState(AppConst.USER_STATE.NORMAL);
                userMapper.updateByPrimaryKeySelective(user);
                userId = user.getId();
                userAuthService.addUserAuth(AppConst.LOGIN_TYPE.WEIXIN, userId, userParam.getUnionId(), null);
                userAccountService.createUserAccount(userId);
                //邀请新用户增加金币并推送
                if (userParam.getInviteUserId() != null && userParam.getTaskInvited()) {
                    addInviteGoldAndPushMsg(userParam, 0);
                }
                missionIntegrationService.asyncCompleteTask(userId, TaskEnum.TASK_FIRST_LOGIN, String.valueOf(userId));
                isInsertInviteRecord = true;
            }
        } else {
            userId = unionUserAuth.getUserId();
            updateUserAuth(openUserAuth, userId);
            User existedUser = userMapper.selectByUserId(userId);

            //邀请老用户增加金币并推送
            if (userParam.getInviteUserId() != null && existedUser != null) {
                addInviteGoldAndPushMsg(userParam, 1);
            }
        }
        this.loadUserToRedis(userId);
        logger.debug("小程序授权登陆移除用户[{}]在redis的缓存", userId);

        addInviteRecord(userParam, userId, isInsertInviteRecord);
        return message.addParam("userId", userId);
    }

    /**
     * 增加邀请金币并推送
     */
    private void addInviteGoldAndPushMsg(TempUserParam userParam, Integer userType) {
        Integer userGold;
        if (0 == userType) {
            userGold = userProperties.getNewUserGold();
        } else if (1 == userType) {
            userGold = userProperties.getOldUserGold();
        } else {
            return;
        }

        userAccountService.addGold(new AccountGoldParam(userParam.getInviteUserId(),
                GoldEnum.USABLE_GOLD.name(),
                true,
                userGold,
                null,
                GoldFlowTypeEnum.TASK_INVITE_FRIEND.name()));

        pushMsgIntegrationService
                .pushInviteMsg(userParam.getInviteUserId(), userParam.getHeadImg(), userParam.getNickName(), userGold, userType);
    }

    /**
     * 修改用户授权信息
     */
    private void updateUserAuth(UserAuth openUserAuth, Long userId) {
        if (!userId.equals(openUserAuth.getUserId())) {
            // 删除旧的授权信息
            userAuthMapper.removeAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_WST_OPENID, userId);
            userAuthMapper.removeAuth(AppConst.LOGIN_TYPE.WEIXIN_APPLET_WST_OPENID, openUserAuth.getUserId());

            //保存新的授权信息
            openUserAuth.setId(nextId());
            openUserAuth.setUserId(userId);
            userAuthMapper.insertSelective(openUserAuth);
        }
    }

    /**
     * 创建邀请关系
     */
    private void addInviteRecord(TempUserParam userParam, Long userId, Boolean isInsertInviteRecord) {
        if (isInsertInviteRecord && userParam.getInviteUserId() != null && !userParam.getInviteUserId().equals(userId)) {
            inviteRecordService.addInviteRecord(userParam.getInviteUserId(), userId, userParam.getInviteType());
            //从任何渠道进来的用户都可以作为vip邀请的好友人数：[2019-04-28]
            RecordWechatParam recordWechatParam = new RecordWechatParam();
            recordWechatParam.setUserId(userParam.getInviteUserId());
            recordWechatParam.setWechatUid(userParam.getUnionId());
            recordWechatParam.setWechatImg(userParam.getHeadImg());
            recordWechatParam.setWechatName(userParam.getNickName());

            logger.debug("添加VIP邀请记录: 邀请人：[{}], 受邀人：[{}]", userParam.getInviteUserId(), userParam.getUnionId());
            //vipIntegrationService.saveTempViewRecord(recordWechatParam);
        }
    }

    /**
     * 创建临时用户
     * TempUserParam用户信息转为user   tempUserParam  -》  user
     */
    private void convertTempUser(TempUserParam tempUserParam, User user) {
        user.setWeixin(tempUserParam.getUnionId());
        user.setNickname(EmojiCodeParser.replaceSoftbankEmoji(tempUserParam.getNickName()));
        user.setHeadImg(tempUserParam.getHeadImg());

        //不知道用户id---随机生成一个id来存到对应图片路径中
        Long userId;
        if (user.getId() == null) {
            userId = nextId();
        } else {
            userId = user.getId();
        }

        String headImg = extendedService.uploadHeadImg(tempUserParam.getHeadImg(), userId);
        if (isNotEmpty(headImg)) {
            user.setHeadImg(headImg);
        }
        user.setSex(tempUserParam.getSex());
        user.setRegIp(tempUserParam.getRegIp());
        user.setRegisterClient(Objects.toString(tempUserParam.getPlatform()));
        user.setRegisteredAddress(tempUserParam.getRegisteredAddress());
        user.setLocationCode(tempUserParam.getLocationCode());
        user.setLocationName(tempUserParam.getLocationName());

        // 如果没有传城市编码
        if (isBlank(tempUserParam.getLocationCode())) {
            // 判断用户是否有邀请人有的话使用邀请人的定位信息
            if (nonNull(tempUserParam.getInviteUserId())) {
                UserLocationInfoDTO infoDTO = getUserLocationCodeById(tempUserParam.getInviteUserId());
                if (nonNull(infoDTO)) {
                    user.setLocationCode(infoDTO.getLocationCode());
                    user.setLocationName(infoDTO.getLocationShortName());
                }
            }
        }

        user.setRelationId(tempUserParam.getRelationId());
        user.setInviteUserId(tempUserParam.getInviteUserId());
        user.setRegisterChannel(tempUserParam.getInviteType());

        User inviteUser = userMapper.selectByPrimaryKey(tempUserParam.getInviteUserId());

        if (inviteUser == null) {
            logger.info("受邀用户[{}]的邀请人不存在，请求参数:[{}]", user.getId(), tempUserParam);
            user.setInviteLevel(1);
        } else {
            logger.info("创建邀请用户[{}]的受邀用户", inviteUser.getId());
            user.setInviteLevel(inviteUser.getInviteLevel() + 1);
        }

        //临时用户
        user.setState(AppConst.USER_STATE.NOT_ACTIVE);
        user.setLevel(LevelEnum.INITIAL_USER.getType());
        user.setIsNew(AppConst.IS_NEW_USER.NEW_USER);
        user.setIsTempNickName((byte) 1);
        user.setIsTempHeadImg((byte) 1);
        user.setNewbieGuideFlag((byte) 0);

    }

    @Override
    public BaseUserInfo getNickNameAndHeadImg(Long userId) {
        BaseUserInfo baseUserInfo = new BaseUserInfo();

        UserInfoDTO user = getUserCache(userId);
        if (null != user) {
            BeanUtils.copyProperties(user, baseUserInfo);
        }

        baseUserInfo.setMerchantStatus(0);

        //添加vip和关联商家
        MerchantUserDTO merchantUserDTO = merchantRelationService.getRelationMerchant(userId);
        if (null != merchantUserDTO) {
            BeanUtils.copyProperties(merchantUserDTO, baseUserInfo);
        }

        if (isBlank(baseUserInfo.getHeadImg())) {
            baseUserInfo.setHeadImg(userProperties.getDefaultHeadImgUrl());
        }

        return baseUserInfo;
    }

    @Override
    public List<UserInfoDTO> getBatchUserInfo(Collection<Long> userIds) {
        List<UserInfoDTO> userInfos = Lists.newArrayList();

        if (!isEmpty(userIds)) {
            userIds.forEach(userId -> {
                UserInfoDTO userInfoDTO = getUserCache(userId);
                if (null != userInfoDTO) {
                    userInfos.add(userInfoDTO);
                }
            });
        }
        return userInfos;
    }

    /**
     * 判断用户是否为风险用户
     */
    @Override
    @Async
    public void isRisk(Long userId, String phone, BasicParam basicParam, String deviceToken, String ip) {
        //判断当前手机号是否存在风险
        boolean isRiskNum = isNotEmpty(phone) &&
                (phone.startsWith("170"))
                || phone.startsWith("171");
        if (isRiskNum) {
            this.userMapper.updateUserToRisk(userId);
        } else {
            if (checkRisk(phone, basicParam, deviceToken, ip)) {
                this.userMapper.updateUserToRisk(userId);
            }
        }
    }

    @Override
    public String getGeneration(LocalDate birthday) {
        String generationStr = null;
        if (null != birthday) {
            int year = birthday.getYear();
            int decade = year / 10;
            String s = String.valueOf(decade);
            generationStr = s.substring(s.length() - 1) + "0后";
        }
        return generationStr;
    }

    @Override
    public UserTbkInfo selectUserFromCacheByRelationId(String relationId) {
        return build(userMapper.selectUserByRelationId(relationId));
    }

    @Override
    public void updateTbkInfo(Long userId, String relationId, String specialId) {
        userMapper.updateTbkInfo(userId, relationId, specialId);
    }

    @Override
    public UserTbkInfo selectUserFromCacheByUserId(Long userId) {
        return build(userMapper.selectUserFromCacheByUserId(userId));
    }


    @Override
    public List<UserBaseInfo> selectUserByIdName(Long userId, String userName) {
        if (null != userId) {
            return userMapper.selectUserById(userId);
        } else if (isNotEmpty(userName)) {
            return userMapper.selectUserByName(userName);
        }
        return new ArrayList<>();
    }

    private UserTbkInfo build(UserTbkInfoBean infoBean) {
        if (Objects.isNull(infoBean)) {
            return null;
        }
        UserTbkInfo info = new UserTbkInfo();
        info.setId(infoBean.getId());
        info.setNickname(infoBean.getNickname());
        info.setIsVip(userVipService.isVip(infoBean.getId()));
        // 是否新人
        info.setIsNovice(!BitOperatorUtil.getBitAsBoolean(infoBean.getJudgeMarker(),
                LocalNewsUserJudgeMarkerEnum.HAS_ORDERED.getIndex()));
        info.setJudgeMarker(infoBean.getJudgeMarker());
        info.setTbkSpecialId(infoBean.getTbkSpecialId());
        info.setTbkRelationId(infoBean.getTbkRelationId());

        UserInviteHistoryBean userInviteHistoryBean = userInviteService.selectByUserId(infoBean.getId());

        if (nonNull(userInviteHistoryBean)) {
            // 上级id
            info.setSuperiorUserId(userInviteHistoryBean.getInviteUserId());
            // 上上级id
            info.setTopUserId(userInviteHistoryBean.getInviteSuperUserId());
            // 内容上级id
            info.setContentSuperiorUserId(userInviteHistoryBean.getCreatedUserId());
            // 内容上上级id
            info.setContentTopUserId(userInviteHistoryBean.getInviteCreatedUserId());
        }
        return info;
    }

    /**
     * 检测手机账号风险系数
     */
    private Boolean checkRisk(String phone, BasicParam basicParam, String deviceToken, String ip) {
        String operateSource = StringUtils.EMPTY;
        if (basicParam.getPlatform() == 1) {
            operateSource = "ANDROID";
        } else if (basicParam.getPlatform() == 2) {
            operateSource = "IOS";
        }
        RiskParam riskParam = new RiskParam();
        riskParam.setDeviceToken(deviceToken);
        riskParam.setIp(ip);
        riskParam.setOperateSource(operateSource);
        riskParam.setPhone(phone);

        EquipmentDTO equipmentDTO = bizIntegrationService.getDevice(basicParam.getDevcId());
        if (null != equipmentDTO && StringUtils.isNoneEmpty(equipmentDTO.getId())) {
            riskParam.setEquiment(equipmentDTO.getId());
            riskParam.setOperateSystem(equipmentDTO.getPhoneModel());
        }

        return riskRecordService.checkRiskDevice(riskParam);
    }

    @Override
    public UserBindPhoneVO checkUserAndPhoneExists(UserBindPhoneParam phoneParam) {
        UserBindPhoneVO result = new UserBindPhoneVO();
        User user = null;
        if (null == phoneParam.getUserId() && StringUtils.isEmpty(phoneParam.getPhone())) {
            result.setStatus(0);
            result.setMsg("手机号和用户id必传一个");
            return result;
        }
        if (null != phoneParam.getUserId()) {
            user = this.userMapper.selectByPrimaryKey(phoneParam.getUserId());
            if (null == user) {
                result.setStatus(6);
                result.setMsg("用户不存在");
                return result;
            } else if (isNotEmpty(user.getPhone())) {
                result.setStatus(5);
                result.setMsg("用户已经绑定手机号");
                return result;
            }
        }
        if (isNotEmpty(phoneParam.getPhone())) {
            //手机号码是否合法
            if (!Validater.checkPhone(phoneParam.getPhone())) {
                result.setStatus(4);
                result.setMsg("手机号码格式错误");
                return result;
            }
            user = userMapper.findByPhone(phoneParam.getPhone());
            if (null != user && isNotEmpty(user.getPhone())) {
                result.setStatus(3);
                result.setMsg("手机号已被使用");
                return result;
            }
        }
        return result;
    }

    @Override
    public UserBindPhoneVO bindUserPhone(UserBindPhoneParam phoneParam) {
        UserBindPhoneVO result = new UserBindPhoneVO();
        if (null == phoneParam.getUserId() || StringUtils.isEmpty(phoneParam.getPhone())) {
            result.setStatus(0);
            result.setMsg("手机号和用户id必传");
            return result;
        }
        result = this.checkUserAndPhoneExists(phoneParam);
        if (result.getStatus() != 1) {
            return result;
        }
        User user = userMapper.selectByPrimaryKey(phoneParam.getUserId());
        if (isNotEmpty(user.getPhone())) {
            result.setStatus(5);
            result.setMsg("用户已绑定手机号");
            return result;
        }
        //更新手机号码
        this.userMapper.updateUserPhone(phoneParam.getUserId(), phoneParam.getPhone(), null);
        user.setPhone(phoneParam.getPhone());
        //更新redis中的缓存
        this.putUserToRedis(user);
        return result;
    }

    @Override
    public void updateUserJudgeMarker(Long userId, LocalNewsUserJudgeMarkerEnum judgeMarker, boolean add) {
        if (logger.isDebugEnabled()) {
            logger.debug("userId: {} {} 标志位judgeMarker: {} ", userId, add ? "新增" : "移除", judgeMarker);
        }

        User user = selectByPrimaryKey(userId);
        if (Objects.isNull(user)) {
            logger.warn("userId: {} 查找不到对应的用户 无法更新标志位judgeMarker: {}", user, judgeMarker);
            return;
        }

        User update = new User();
        update.setId(userId);
        update.setUpdateTime(new Date());

        if (add) {
            update.setJudgeMarker(BitOperatorUtil.setBitToLong(user.getJudgeMarker(), judgeMarker.getIndex()));
        } else {
            update.setJudgeMarker(BitOperatorUtil.clearBit(user.getJudgeMarker(), judgeMarker.getIndex()));
        }


        // 更新
        userMapper.updateByPrimaryKeySelective(update);
        // 清除缓存
        removeUserCache(userId);
    }

    @Override
    public List<InteractRankInfo> getInteractRankInfo(String areaCode, int limitNum, List<Long> userIds) {
        return userMapper.selectInteractRankInfo(areaCode, limitNum, userIds);
    }

    @Override
    public Integer replyNumForUser(Long userId) {
        Integer replyNum = userMapper.getReplyNumById(userId);
        return replyNum;
    }

    @Override
    public Boolean clipboardUpload(UserClipboardParam param) {
        protocalProcesser.execProtocal(param);
        return true;
    }

    @Override
    public List<UserPhoneDTO> getUserInfoBySensitivePhone(String sensitivePhone) {
        //手机号码的长度
        int phoneLength = 11;
        //如果脱敏的手机号为null或者长度不等于11位，则直接返回空的list
        if (sensitivePhone == null || sensitivePhone.length() != phoneLength) {
            return Collections.emptyList();
        }
        //手机号码3位前缀
        String phonePrefix = substring(sensitivePhone, 0, 3);
        //手机号码的4位后缀
        String phoneSuffix = substring(sensitivePhone, 7);

        return userMapper.selectUserBySensitivePhone(phonePrefix, phoneSuffix);
    }

    @Override
    public UserLocationInfoDTO getUserLocationCodeById(Long userId) {
        UserInfoDTO userCache = getUserCache(userId);
        return Objects.isNull(userCache) ? null : convert(userCache);
    }

    @Override
    public UserPhoneDTO getUserInfoByPhone(String phone) {
        return userMapper.selectUserByPhone(phone);
    }

    @Override
    public UserRegisterInfoDTO userRegisterInfo(Long userId) {

        // 可以把当前这个信息缓存一下 避免每次查询
        UserInfoDTO userInfoDTO = loadUserToRedis(userId);
        if (Objects.isNull(userInfoDTO)) {
            return null;
        }

        UserRegisterInfoDTO registerInfoDTO = new UserRegisterInfoDTO();
        BeanUtils.copyProperties(userInfoDTO, registerInfoDTO);
        registerInfoDTO.setUserId(userId);

        UserLoginHistory userLoginHistory = userLoginHistoryService.selectFirst(userId);
        if (nonNull(userLoginHistory)) {
            registerInfoDTO.setFirstLoginTime(userLoginHistory.getLastLoginTime());
            registerInfoDTO.setIntoAppDays(DateUtils.getDiffDays(userLoginHistory.getLastLoginTime(),
                    new Date(), true) + 1);
        }
        return registerInfoDTO;
    }

    @Override
    public Boolean delUserCacheBatch(List<Long> userIds) {
        if (Objects.equals(CollectionUtils.isEmpty(userIds), Boolean.FALSE)) {
            for (Long userId : userIds) {
                redisStringAdapter.remove(buildUserCacheKey(userId));
            }
        }
        return true;
    }

    private UserLocationInfoDTO convert(UserInfoDTO userInfoDTO) {
        UserLocationInfoDTO locationInfoDTO = new UserLocationInfoDTO();
        locationInfoDTO.setId(userInfoDTO.getId());
        locationInfoDTO.setLocationCode(userInfoDTO.getLocationCode());
        locationInfoDTO.setLocationName(userInfoDTO.getLocationName());
        locationInfoDTO.setLocationShortName(userInfoDTO.getLocationShortName());

        return locationInfoDTO;
    }

}
