/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.user.vip.activation;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.constant.UserActiveVipEnum;
import com.bxm.localnews.common.rediskey.UserRedisKey;
import com.bxm.localnews.user.constant.VipActiveCode;
import com.bxm.localnews.user.domain.UserInviteHistoryMapper;
import com.bxm.localnews.user.dto.UserInfoDTO;
import com.bxm.localnews.user.dto.UserInviteBindDTO;
import com.bxm.localnews.user.enums.ActivationVipEnum;
import com.bxm.localnews.user.enums.ActivationVipStatusEnum;
import com.bxm.localnews.user.enums.InviteBindMethodEnum;
import com.bxm.localnews.user.invite.UserInviteService;
import com.bxm.localnews.user.invite.bind.BindInviteManager;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.ActivationUserVipParam;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vo.UserInviteHistoryBean;
import com.bxm.localnews.user.vo.UserVip;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Objects;

import static org.apache.commons.lang3.StringUtils.isEmpty;

/**
 * 不同的激活策略
 *
 * @author : jieli.xu
 * @date 2020/5/14 16:57
 */
@Component
@Slf4j
public abstract class AbstractActivationVipStrategy {

    @Autowired
    private RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    private UserService userService;


    @Autowired
    private UserInviteService userInviteService;

    @Autowired
    private BindInviteManager bindInviteManager;

    private UserVipService userVipService;

    @Autowired
    private UserInviteHistoryMapper userInviteHistoryMapper;

    @Autowired
    private DistributedLock distributedLock;

    @Autowired
    private SequenceCreater sequenceCreater;

    private KeyGenerator activeVip = DefaultKeyGenerator.build("user", "lock", "activeVip");

    public UserVipService getUserVipService() {
        // 解除递归依赖
        if (userVipService == null) {
            userVipService = SpringContextHolder.getBean(UserVipService.class);
        }

        return userVipService;
    }

    /**
     * 前置检查,判断是否可以完成激活处理
     *
     * @param param 激活VIP的相关参数
     * @return 执行结果
     */
    protected abstract Message preCheck(ActivationUserVipParam param);

    /**
     * 激活成功后的处理逻辑
     *
     * @param param VIP激活参数
     */
    protected abstract void afterSuccess(ActivationUserVipParam param);

    /**
     * VIP激活失败后的处理逻辑，用于回退数据等操作
     * 默认不需要，如子类有需要可进行覆盖
     *
     * @param param VIP激活参数
     */
    protected void afterFailed(ActivationUserVipParam param) {
    }

    /**
     * 执行具体的激活vip流程
     *
     * @param param 激活参数
     * @return 激活结果
     */
    public final Message execActivationVip(ActivationUserVipParam param) {
        log.debug("执行VIP激活流程，当前激活类型：{},请求参数：{}",
                getType().name(),
                param);

        String requestId = sequenceCreater.nextStringId();
        String resource = activeVip.copy().appendKey(param.getUserId()).gen();
        if (!distributedLock.lock(resource, requestId)) {
            log.info("重复点击激活VIP，请求参数：{}", JSON.toJSONString(param));
            return Message.build(false, "请勿重复点击");
        }

        // 执行VIP激活的主体业务逻辑
        Message message = this.preCheck(param);

        //后置方法-发送推送，发放奖励等-子类实现具体的方法
        if (message.isSuccess()) {
            try {
                //执行VIP激活
                message.append(upgradeVip(param));

                if (message.isSuccess()) {
                    //激活完成的后置处理
                    afterSuccess(param);
                }
            } catch (Exception e) {
                log.error("VIP激活操作失败，请求参数为：{}", param);
                log.error(e.getMessage(), e);

                message = Message.build(false, "VIP激活失败");
            }

            if (!message.isSuccess()) {
                // 激活失败的后置处理，用于还原数据
                afterFailed(param);
            }
        }

        log.debug("VIP激活完成，激活类型：{},请求参数：{},处理结果：{}",
                getType().name(),
                param,
                message);

        if (message.getParam(VipActiveCode.KEY) == null) {
            message.addParam(VipActiveCode.KEY, VipActiveCode.SUCCESS);
        }

        distributedLock.unlock(resource, requestId);

        return message;
    }

    /**
     * 执行具体的升级VIP逻辑
     * 1.判断用户当前的VIP状态，如果非VIP则激活，如果已经是VIP则续约
     * 2.如果是站外开通的VIP，则设置首页弹窗，在进入VIP时进行承接逻辑
     */
    protected Message upgradeVip(ActivationUserVipParam param) {
        UserVip userVip = getUserVipService().getUserVipByUid(param.getUserId());

        // 设置开通时长
        param.setDuration(getDuration(param));

        Message message;
        // 只有处于待激活状态的VIP记录才创建，其他情况均作为续约处理
        if (Objects.equals(userVip.getStatus(), ActivationVipStatusEnum.WAIT_USE.getStatus())) {
            // 创建VIP，VIP的相关操作统一归集在VIP服务接口
            message = userVipService.createVip(param);
            if (message.isSuccess()) {
                // 初次创建VIP时，初始化师徒信息，建立上下级关系
                initMasterInfo(param);
            }
        } else {
            // VIP续约
            message = userVipService.execRenew(param);
        }

        //站外首次激活设置弹窗
        if (param.getPlatformEnum() != (PlatformEnum.ANDROID.getCode()) && param.getPlatformEnum() != (PlatformEnum.IOS.getCode())) {
            //设置首次弹窗
            if (isEmpty(redisHashMapAdapter.get(UserRedisKey.HASH_USER_ACTIVE_VIP_FIRST, String.valueOf(param.getUserId()), String.class))) {
                // 0：不需要弹窗 1: 站外开卡 需要弹出对应的弹窗 2: 站内开卡 需要弹出对应的弹窗
                redisHashMapAdapter.put(UserRedisKey.HASH_USER_ACTIVE_VIP_FIRST, String.valueOf(param.getUserId()), "1");
            }
        } else {
            // 站内激活 设置首次弹窗
            if (isEmpty(redisHashMapAdapter.get(UserRedisKey.HASH_USER_ACTIVE_VIP_FIRST, String.valueOf(param.getUserId()), String.class))) {
                // 0：不需要弹窗 1: 站外开卡 需要弹出对应的弹窗 2: 站内开卡 需要弹出对应的弹窗
                redisHashMapAdapter.put(UserRedisKey.HASH_USER_ACTIVE_VIP_FIRST, String.valueOf(param.getUserId()), "2");
            }
        }
        return message;
    }

    /**
     * 用户开通VIP后初始化师傅的相关信息
     * 包括更新师傅信息，补充师傅id等
     *
     * @param param
     */
    public void initMasterInfo(ActivationUserVipParam param) {
        log.debug("建立师徒关系，请求参数：{}", param);

        UserInfoDTO userInfoDTO = userService.getUserCache(param.getUserId());
        param.setUserName(userInfoDTO.getNickname());

        // 使用激活码的情况
        // [3.2.0]正常用户（已登录app)不修改上级
        // [3.3.0]正常用户（已登录app)修改上级 - 记住产品罪恶的需求
        if (ActivationVipEnum.ACTIVATION_CODE.equals(getType())) {
            UserInviteBindDTO userInviteBindDTO = new UserInviteBindDTO();
            userInviteBindDTO.setInviteBindMethodEnum(InviteBindMethodEnum.ACTIVE_CODE);
            userInviteBindDTO.setUserId(param.getUserId());
            userInviteBindDTO.setInviteUserId(param.getCodeUserId());
            userInviteBindDTO.setActiveCodeHasTime(true);
            UserInviteHistoryBean userInviteHistoryBean = bindInviteManager.bindInvite(userInviteBindDTO);
            if (null != userInviteHistoryBean) {
                param.setMasterId(userInviteHistoryBean.getInviteUserId());
            }
        } else if (ActivationVipEnum.PAY.equals(getType())) {
            UserInviteBindDTO userInviteBindDTO = new UserInviteBindDTO();
            userInviteBindDTO.setInviteBindMethodEnum(InviteBindMethodEnum.WELFARE_VIP);
            userInviteBindDTO.setUserId(param.getUserId());
            userInviteBindDTO.setIsCallback(true);
            UserInviteHistoryBean userInviteHistoryBean = bindInviteManager.bindInvite(userInviteBindDTO);
            log.info(">>>>>>>>>>2---preCheckCommon 【{}】", JSONObject.toJSONString(userInviteHistoryBean));
            //支付的情况下，判断是否有邀请关系，有的话则进行赋值masterId
            if (null != userInviteHistoryBean) {
                param.setMasterId(userInviteHistoryBean.getInviteUserId());
            }
        }
    }

    /**
     * 如果建立邀请关系，当前实现类对应的邀请类型
     *
     * @return 邀请类型信息
     */
    abstract InviteTypeEnum getInviteType();

    /**
     * 设置类属性,每个类对应唯一属性，作为key值
     *
     * @return 当前激活方式支持的类型
     */
    abstract ActivationVipEnum getType();

    /**
     * 根据VIP激活参数获取开通的时长
     *
     * @param param 激活参数
     * @return VIP的开通时长
     */
    abstract int getDuration(ActivationUserVipParam param);
}
