/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.user.vip.activation;

import com.bxm.localnews.common.constant.InviteTypeEnum;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.user.constant.VipActivationChannel;
import com.bxm.localnews.user.constant.VipActiveCode;
import com.bxm.localnews.user.domain.vip.UserActiveRelationMapper;
import com.bxm.localnews.user.dto.LocationUserInfoDTO;
import com.bxm.localnews.user.enums.ActivationVipEnum;
import com.bxm.localnews.user.integration.BizLogIntegrationService;
import com.bxm.localnews.user.integration.PushMsgIntegrationService;
import com.bxm.localnews.user.login.UserService;
import com.bxm.localnews.user.param.ActivationUserVipParam;
import com.bxm.localnews.user.vip.ActiveCodeService;
import com.bxm.localnews.user.vo.UserVip;
import com.bxm.localnews.user.vo.vip.UserActiveCodeBean;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Component;

import java.time.LocalDate;
import java.util.regex.Pattern;

/**
 * 通过激活码激活VIP的执行策略
 *
 * @author : jieli.xu
 * @date 2020/5/14 16:57
 */
@Component
@Slf4j
@Primary
public class CodeActivationStrategy extends AbstractActivationVipStrategy {

    private final PushMsgIntegrationService pushMsgIntegrationService;

    private final UserService userService;

    private final BizLogIntegrationService bizLogIntegrationService;

    private final ActiveCodeService activeCodeService;

    private final UserActiveRelationMapper userActiveRelationMapper;

    private Pattern pattern = Pattern.compile("[0-9a-zA-Z]{1,}");

    @Autowired
    public CodeActivationStrategy(PushMsgIntegrationService pushMsgIntegrationService,
                                  UserService userService,
                                  BizLogIntegrationService bizLogIntegrationService,
                                  ActiveCodeService activeCodeService,
                                  UserActiveRelationMapper userActiveRelationMapper) {
        this.pushMsgIntegrationService = pushMsgIntegrationService;
        this.userService = userService;
        this.bizLogIntegrationService = bizLogIntegrationService;
        this.activeCodeService = activeCodeService;
        this.userActiveRelationMapper = userActiveRelationMapper;
    }

    @Override
    public Message preCheck(ActivationUserVipParam param) {
        //检查激活码是否为数字或字母
        if (!pattern.matcher(param.getCode()).matches()) {
            log.info("输入的激活码格式错误，激活码为：[{}]", param.getCode());
            return Message.build(false, "激活码格式错误").addParam(VipActiveCode.KEY, VipActiveCode.NOT_FOUND);
        }

        UserActiveCodeBean activeCodeInfo = activeCodeService.getUsableActiveCode(param.getCode());

        if (null == activeCodeInfo) {
            log.info("用户输入的激活码不存在，激活码为：[{}]", param.getCode());
            return Message.build(false, "激活码不存在或已失效").addParam(VipActiveCode.KEY, VipActiveCode.NOT_FOUND);
        }
        if(null == activeCodeInfo.getMaxTimes() || activeCodeInfo.getMaxTimes() <= 0){
            return Message.build(false, "激活码次数已用完").addParam(VipActiveCode.KEY, VipActiveCode.NO_TIMES);
        }
        //用户是否已经是VIP
        UserVip openUserVip = getUserVipService().getUserVipByUid(param.getUserId());
        if (openUserVip.getUsable()) {
            log.info("{}已经是VIP,重复进行了激活", param.getUserId());
            return Message.build(false, "你已经是VIP,不需要重复激活.").addParam(VipActiveCode.KEY, VipActiveCode.EXISTS);
        }

        // 提前扣除一次激活码的使用次数，如果后续发生异常或失败，则还原激活码使用次数
        Message message = activeCodeService.decrementWithRetry(param.getCode());

        if (message.isSuccess()) {
            //补充设置激活码归属人的id和设置默认的师傅id
            param.setCodeUserId(activeCodeInfo.getUserId());
        } else {
            message.addParam(VipActiveCode.KEY, VipActiveCode.NO_TIMES);
        }

        if (StringUtils.isBlank(param.getChannel())) {
            param.setChannel(VipActivationChannel.CODE);
        }

        return message;
    }

    @Override
    protected void afterFailed(ActivationUserVipParam param) {
        // 还原激活次数
        activeCodeService.incrementWithRetry(param.getCode());
        // 删除激活记录
        userActiveRelationMapper.removeRelation(param.getCode(), param.getUserId());
    }

    @Override
    protected void afterSuccess(ActivationUserVipParam param) {
        if (null != param.getCodeUserId()) {
            // 上报数据埋点
            PointReportParam reportParam = PointReportParam.build(param)
                    .e("3034")
                    .ev("119." + param.getCodeUserId())
                    .put("uid", String.valueOf(param.getUserId()));

            if (param.getPlatformEnum() != null) {
                reportParam.dtype(param.getPlatformEnum().toString());
            }
            bizLogIntegrationService.point(reportParam);

            // 删除激活码的所属人的缓存信息
            getUserVipService().removeCache(param.getCodeUserId());
        }

        if (null != param.getMasterId()) {
            //给师傅发送消息推送
            LocationUserInfoDTO location = userService.getLocationUserInfo(param.getMasterId());

            if (location != null) {
                pushMsgIntegrationService.pushMasterReceiveMessage(param.getMasterId(), param.getUserName(), location);
            } else {
                log.error("用户[{}]无所属地区信息，不发送推送", param.getMasterId());
            }

        }
    }

    @Override
    InviteTypeEnum getInviteType() {
        return InviteTypeEnum.ACTIVE_CODE;
    }

    @Override
    public ActivationVipEnum getType() {
        return ActivationVipEnum.ACTIVATION_CODE;
    }

    @Override
    int getDuration(ActivationUserVipParam param) {
        LocalDate current = LocalDate.now();
        LocalDate nextYear = LocalDate.now().plusYears(1);

        return (int) (nextYear.toEpochDay() - current.toEpochDay());
    }
}
