package com.bxm.localnews.user.vip.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.user.domain.UserMapper;
import com.bxm.localnews.user.domain.vip.UserActiveCodeMapper;
import com.bxm.localnews.user.domain.vip.UserActiveRelationMapper;
import com.bxm.localnews.user.dto.UserInviteBindDTO;
import com.bxm.localnews.user.dto.vip.ActiveCodeDTO;
import com.bxm.localnews.user.enums.ActivationVipEnum;
import com.bxm.localnews.user.enums.ActiveCodeTypeEnum;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.invite.UserInviteService;
import com.bxm.localnews.user.invite.bind.BindInviteManager;
import com.bxm.localnews.user.model.ResultDTO;
import com.bxm.localnews.user.param.ActivationUserVipParam;
import com.bxm.localnews.user.param.AddTimesParam;
import com.bxm.localnews.user.param.OfflineBindRelationParam;
import com.bxm.localnews.user.properties.UserVipProperties;
import com.bxm.localnews.user.utils.NumberConvertUtils;
import com.bxm.localnews.user.vip.ActiveCodeService;
import com.bxm.localnews.user.vip.UserVipService;
import com.bxm.localnews.user.vip.VipCardService;
import com.bxm.localnews.user.vip.activation.ActivationVipManager;
import com.bxm.localnews.user.vip.code.ActiveCodeProcessContext;
import com.bxm.localnews.user.vip.code.IActiveCodeProcessStrategy;
import com.bxm.localnews.user.vo.User;
import com.bxm.localnews.user.vo.UserInviteOrderVO;
import com.bxm.localnews.user.vo.vip.UserActiveCodeBean;
import com.bxm.localnews.user.vo.vip.UserActiveRelationBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;

import static com.bxm.localnews.user.constant.RedisConfig.VIP_CODE_KEY;
import static com.bxm.localnews.user.enums.ActiveCodeTypeEnum.MANUAL;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang3.RandomUtils.nextInt;

@Service
@Slf4j
public class ActiveCodeServiceImpl implements ActiveCodeService {

    private Map<ActiveCodeTypeEnum, IActiveCodeProcessStrategy> strategyMap;

    private final UserActiveCodeMapper userActiveCodeMapper;

    private final UserActiveRelationMapper userActiveRelationMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final VipCardService vipCardService;

    private final SequenceCreater sequenceCreater;

    private final UserVipProperties userVipProperties;

    private final UserMapper userMapper;

    private final static Long START_NUM = 10000000L;

    private final UserInviteService userInviteService;

    private UserVipService userVipService;

    private final BindInviteManager bindInviteManager;

    private ActivationVipManager activationVipManager;

    @Autowired
    public ActiveCodeServiceImpl(List<IActiveCodeProcessStrategy> strategyList,
                                 UserActiveCodeMapper userActiveCodeMapper,
                                 UserActiveRelationMapper userActiveRelationMapper,
                                 RedisStringAdapter redisStringAdapter,
                                 VipCardService vipCardService,
                                 SequenceCreater sequenceCreater,
                                 UserVipProperties userVipProperties,
                                 UserMapper userMapper,
                                 UserInviteService userInviteService, BindInviteManager bindInviteManager) {
        this.userActiveCodeMapper = userActiveCodeMapper;
        this.userActiveRelationMapper = userActiveRelationMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.vipCardService = vipCardService;
        this.sequenceCreater = sequenceCreater;
        this.userVipProperties = userVipProperties;
        this.userMapper = userMapper;
        this.userInviteService = userInviteService;
        this.bindInviteManager = bindInviteManager;
        strategyMap = Maps.newHashMap();

        strategyList.forEach(strategy -> {
            strategyMap.put(strategy.match(), strategy);
        });
    }

    private UserVipService getUserVipService() {
        if (userVipService == null) {
            userVipService = SpringContextHolder.getBean(UserVipService.class);
        }
        return userVipService;
    }

    private ActivationVipManager getActivationVipManager() {
        if (activationVipManager == null) {
            activationVipManager = SpringContextHolder.getBean(ActivationVipManager.class);
        }
        return activationVipManager;
    }

    @Override
    public UserActiveCodeBean getUsableActiveCode(String activeCode) {
        UserActiveCodeBean entity = userActiveCodeMapper.selectByCode(activeCode);
        if (null == entity ) {
            return null;
        }
        return entity;
    }

    @Override
    public UserActiveCodeBean getActiveCode(String activeCode) {
        return userActiveCodeMapper.selectByCode(activeCode);
    }

    @Override
    @Retryable(RetryException.class)
    public Message decrementWithRetry(String activeCode) {
        UserActiveCodeBean currentActiveCode = userActiveCodeMapper.selectByCode(activeCode);
        if (currentActiveCode.getMaxTimes() <= 0) {
            return Message.build(false, "激活码无可用次数");
        }

        currentActiveCode.setMaxTimes(currentActiveCode.getMaxTimes() - 1);
        currentActiveCode.setActiveTimes(currentActiveCode.getActiveTimes() + 1);

        if (currentActiveCode.getMaxTimes() <= 0) {
            currentActiveCode.setStatus(AppConst.DISALBE);
        } else {
            currentActiveCode.setStatus(AppConst.ENABLE);
        }

        int result = userActiveCodeMapper.updateTimes(currentActiveCode);
        if (result == 0) {
            log.error("激活码扣除失败，激活码为：{}", activeCode);
            throw new RetryException("扣除激活码可用次数更新失败，重试");
        }
        return Message.build(result);
    }

    @Override
    @Retryable(RetryException.class)
    public Message incrementWithRetry(String activeCode) {
        UserActiveCodeBean currentActiveCode = userActiveCodeMapper.selectByCode(activeCode);

        currentActiveCode.setStatus(AppConst.ENABLE);
        currentActiveCode.setMaxTimes(currentActiveCode.getMaxTimes() + 1);
        if (currentActiveCode.getMaxTimes() <= 0) {
            currentActiveCode.setMaxTimes(1);
        }
        if (currentActiveCode.getActiveTimes() > 0) {
            currentActiveCode.setActiveTimes(currentActiveCode.getActiveTimes() - 1);
        }

        int result = userActiveCodeMapper.updateTimes(currentActiveCode);
        if (result == 0) {
            log.error("激活码更新失败，激活码为：{}", activeCode);
            throw new RetryException("增加激活码可用次数更新失败，重试");
        }
        return Message.build(result);
    }

    @Override
    public Message addActiveTimes(AddTimesParam param) {
        Boolean vip = getUserVipService().isVip(param.getUserId());

        // 如果用户还不是VIP，则手工激活VIP
        if (Boolean.FALSE.equals(vip)) {
            log.info("用户[{}]不是VIP，添加激活码时，手工激活VIP", param.getUserId());

            ActivationUserVipParam vipParam = new ActivationUserVipParam();
            vipParam.setUserId(param.getUserId());
            vipParam.setActivationVipEnum(ActivationVipEnum.MANUAL);

            Message message = getActivationVipManager().activationVip(vipParam);
            if (Boolean.FALSE.equals(message.isSuccess())) {
                return message;
            }
        }

        UserActiveCodeBean activeCode = userActiveCodeMapper.selectByUserId(param.getUserId());

        if (null == activeCode) {
            return Message.build(false, "用户未开通VIP或无激活码");
        }

        if (param.getTimes() <= 0 && param.getTimes() > 99999) {
            return Message.build(false, "激活码设置范围超过限制范围");
        }

        activeCode.setMaxTimes(param.getTimes());
        activeCode.setStatus(AppConst.ENABLE);


        int total = userActiveCodeMapper.addTimes(activeCode);

        if (total > 0) {
            getUserVipService().removeCache(param.getUserId());
        }

        return Message.build(total);
    }

    @Override
    public ActiveCodeDTO process(ActiveCodeProcessContext context) {

        if (ActivationVipEnum.PAY.equals(context.getActivationVipType())) {
            context.setActiveCodeType(ActiveCodeTypeEnum.PAYMENT);
        } else if (isNotBlank(context.getCode())) {
            // 获取原始的激活码信息
            UserActiveCodeBean sourceCodeInfo = getActiveCode(context.getCode());
            context.setSourceCodeInfo(sourceCodeInfo);

            if (Objects.equals(ActiveCodeTypeEnum.CHANNEL.getCode(), sourceCodeInfo.getCardType())) {
                context.setActiveCodeType(ActiveCodeTypeEnum.CHANNEL);
            } else {
                context.setActiveCodeType(ActiveCodeTypeEnum.SUB);
            }
        } else {
            context.setActiveCodeType(MANUAL);
        }

        return strategyMap.get(context.getActiveCodeType()).process(context);
    }

    @Override
    public String nextCode() {
        return nextCode(1).get(0);
    }


    @Override
    public List<String> nextCode(Integer limit) {
        int randomKey = nextInt(1, 21);

        KeyGenerator key = VIP_CODE_KEY.copy().appendKey(randomKey);

        if (!redisStringAdapter.hasKey(key)) {
            redisStringAdapter.set(key, START_NUM * randomKey);
        }
        long afterIncrementNum = redisStringAdapter.increment(key, limit);

        List<String> codeList = new ArrayList<>(limit);

        for (int i = limit - 1; i >= 0; i--) {
            codeList.add(NumberConvertUtils.convertToShortCode(afterIncrementNum - i, 6));
        }

        return codeList;
    }

    @Override
    public List<ActiveCodeDTO> execBatchExport(String areaCode, int limit) {
        List<String> cardNoList = vipCardService.nextCard(areaCode, limit);

        List<ActiveCodeDTO> result = new ArrayList<>(limit);

        List<UserActiveCodeBean> saveList = new ArrayList<>(limit);

        List<String> phaseCodeList = Lists.newArrayList();

        for (int i = 0; i < limit; i++) {

            if (phaseCodeList == null || phaseCodeList.size() <= 0) {
                phaseCodeList = nextCode(nextInt(50, 300));
            }

            String code = phaseCodeList.remove(0);

            ActiveCodeDTO activeCode = ActiveCodeDTO.builder()
                    .activeCode(code)
                    .cardNo(cardNoList.get(i))
                    .build();


            UserActiveCodeBean userActiveCode = new UserActiveCodeBean();
            userActiveCode.setId(sequenceCreater.nextLongId());
            userActiveCode.setCardType(ActiveCodeTypeEnum.CHANNEL.getCode());
            userActiveCode.setActiveCode(code);
            userActiveCode.setCardNo(cardNoList.get(i));
            userActiveCode.setMaxTimes(userVipProperties.getChannelCodeNum());
            userActiveCode.setActiveTimes(0);
            userActiveCode.setAreaCode(areaCode);
            userActiveCode.setStatus(AppConst.ENABLE);
            userActiveCode.setCreateTime(new Date());

            result.add(activeCode);
            saveList.add(userActiveCode);
        }

        MybatisBatchBuilder.create(UserActiveCodeMapper.class, saveList).run((mapper, item) -> {
            return mapper.insert(item);
        });

        return result;
    }

    @Override
    public List<ResultDTO> execBatchBind(List<OfflineBindRelationParam> bindRelationParams) {
        if (bindRelationParams == null) {
            return Lists.newArrayList();
        }

        Map<String, User> tempUserMap = Maps.newHashMap();

        List<ResultDTO> resultDTOS = Lists.newArrayList();
        int index = 0;

        for (OfflineBindRelationParam bindRelationParam : bindRelationParams) {
            //根据手机号码获取用户
            User user = tempUserMap.get(bindRelationParam.getPhoneNo());
            if (null == user) {
                user = userMapper.findByPhone(bindRelationParam.getPhoneNo());
                tempUserMap.put(bindRelationParam.getPhoneNo(), user);
            }

            ResultDTO result = ResultDTO.builder().success(true).build();

            if (null == user) {
                result.setSuccess(false);
                result.appendResult("手机号码[" + bindRelationParam.getPhoneNo() + "]对应的用户不存在");
            }

            //根据卡号获取对应信息
            UserActiveCodeBean activeCode = userActiveCodeMapper.selectByCardNo(bindRelationParam.getCardNo());
            if (null == activeCode) {
                result.setSuccess(false);
                result.appendResult("激活码[" + bindRelationParam.getActiveCode() + "]不存在");
            }

            if (result.getSuccess()) {
                //此处不做任何更改师徒关系的操作，故注释掉
//                modifyRecord(activeCode, user);
            } else {
                result.setTitle(bindRelationParam.getPhoneNo());
                resultDTOS.add(result);
            }
            index++;
        }

        return resultDTOS;
    }

    /**
     * 更新激活码的所属人信息
     * 如果激活码已经被使用，则设置使用激活码的用户的上级为激活码所属人
     * 如果激活码的使用用户已经有上级，则不做处理
     *
     * @param activeCode 激活码信息
     * @param user       激活码的所属用户
     */
    private void modifyRecord(UserActiveCodeBean activeCode, User user) {
        // 设置激活码的所属用户
        activeCode.setUserId(user.getId());
        userActiveCodeMapper.updateUserId(activeCode);

        // 如果激活码已被使用，获取对应的用户，设置其上级用户为当前用户
        if (Objects.equals(activeCode.getStatus(), AppConst.DISALBE)) {
            List<UserActiveRelationBean> relations = userActiveRelationMapper.selectByCode(activeCode.getActiveCode());

            if (CollectionUtils.isEmpty(relations)) {
                log.error("激活码[{}]已使用，但是未发现对应的激活记录", activeCode.getActiveCode());
            } else {
                relations.forEach(relation -> {
                    if (null != relation.getActiveUserId()) {
                        UserInviteOrderVO inviteInfo = userInviteService.getInviteNumByUserId(relation.getActiveUserId());

                        // 如果激活的用户无上级则设置激活码所属用户未上级
                        if (inviteInfo == null) {
                            UserInviteBindDTO userInviteBindDTO = new UserInviteBindDTO();
//                            userInviteBindDTO.setIsCallback(false);
//                            userInviteBindDTO.setUserId(relation.getActiveUserId());
//                            userInviteBindDTO.setInviteUserId(user.getId());
//                            userInviteBindDTO.setInviteBindMethodEnum(InviteBindMethodEnum.ACTIVE_CODE);
//                            userInviteBindDTO.setActiveCodeHasTime(true);
                            bindInviteManager.bindInvite(userInviteBindDTO);
                        }
                    }
                });
            }
        }
    }
}




















