package com.bxm.openlog.extension.client;

import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Lists;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.springframework.http.MediaType;
import org.springframework.util.ClassUtils;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * 基于 HttpClient 的实现。
 *
 * @author allen
 * @date 2021-10-27
 * @since 1.0
 * @see Builder
 */
@Slf4j
public class HttpClientOpenLogClient implements OpenLogClient {

    private final ThreadPoolExecutor executor;
    private final HttpClient httpClient;

    private Timer timer;
    private Counter success;
    private Counter fail;

    private HttpClientOpenLogClient(ThreadPoolExecutor executor, int maxTotal, int defaultMaxPerRoute, int connectionRequestTimeout, int connectTimeout, int socketTimeout) {
        this.executor = executor;
        this.httpClient = HttpClientUtils.createHttpClient(maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        Class<?> clazz = ClassUtils.getUserClass(this);
        registerGauge(registry, this, clazz);
        registerTimer(registry, this, clazz);
        registerCounter(registry, this, clazz);
    }

    @Override
    public boolean request(String url) throws IOException {
        return this.request(url, HttpMethod.GET);
    }

    @Override
    public boolean request(String url, HttpMethod httpMethod) throws IOException {
        long start = System.nanoTime();
        HttpResponse response = null;
        try {
            HttpUriRequest request = null;
            switch (httpMethod) {
                case GET:
                    request = new HttpGet(url);
                    break;
                case POST:
                    UriComponents build = UriComponentsBuilder.fromUriString(url).build();
                    MultiValueMap<String, String> params = build.getQueryParams();
                    List<NameValuePair> pairs = Lists.newArrayListWithCapacity(params.size());
                    for (Map.Entry<String, List<String>> e : params.entrySet()) {
                        String key = e.getKey();
                        List<String> value = e.getValue();
                        if (CollectionUtils.isNotEmpty(value)) {
                            String str = value.get(0);
                            pairs.add(new BasicNameValuePair(key, UrlHelper.urlDecode(str)));
                        }
                    }
                    String nonParamUrl = UriComponentsBuilder.fromUriString(url)
                            .replaceQuery(null)
                            .build()
                            .toString();
                    HttpPost post = new HttpPost(nonParamUrl);
                    post.setEntity(new UrlEncodedFormEntity(pairs));
                    post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_FORM_URLENCODED_VALUE);
                    request = post;
                    break;
                default:
                    throw new UnsupportedOperationException("Unsupported httpMethod: " + httpMethod);
            }
            response = httpClient.execute(request);
            return response.getStatusLine().getStatusCode() == HttpStatus.SC_OK;
        } finally {
            if (Objects.nonNull(response)) {
                EntityUtils.consume(response.getEntity());
            }

            if (Objects.nonNull(timer)) {
                timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
            }
        }
    }

    @Override
    public void asyncRequest(String url, Consumer<Fallback> fallback) {
        this.asyncRequest(url, fallback, HttpMethod.GET);
    }

    @Override
    public void asyncRequest(String url, Consumer<Fallback> fallback, HttpMethod httpMethod) {
        this.executor.execute(() -> {
            try {
                if (!request(url, httpMethod)) {
                    fail.increment();
                    if (Objects.nonNull(fallback)) {
                        fallback.accept(Fallback.builder().url(url).build());
                    }
                } else {
                    success.increment();
                }
            } catch (IOException e) {
                fail.increment();

                if (Objects.nonNull(fallback)) {
                    fallback.accept(Fallback.builder().url(url).exception(e).build());
                }
            }
        });
    }

    @Override
    public void asyncRequest(String url) {
        this.asyncRequest(url, HttpMethod.GET);
    }

    @Override
    public void asyncRequest(String url, HttpMethod httpMethod) {
        this.asyncRequest(url, null, httpMethod);
    }

    @Override
    public int queueSize() {
        return executor.getQueue().size();
    }

    @Override
    public void close() {
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(30, TimeUnit.SECONDS)) {
                log.warn("This executor was forced terminated!");
            }
        } catch (InterruptedException e) {
            log.error("awaitTermination: ", e);
        }
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }

    public ThreadPoolExecutor getExecutor() {
        return executor;
    }

    private void registerGauge(MeterRegistry registry, OpenLogClient client, Class<?> clazz) {
        Gauge.builder("OpenLogClient.queue", 0, value -> client.queueSize())
                .tags("name", clazz.getName())
                .register(registry);
    }

    private void registerTimer(MeterRegistry registry, OpenLogClient client, Class<?> clazz) {
        this.timer = Timer.builder("OpenLogClient.timer")
                .tag("name", clazz.getName())
                .register(registry);
        log.info("Registering OpenLogClient {} timer meter successful.", client);
    }

    private void registerCounter(MeterRegistry registry, OpenLogClient client, Class<?> clazz) {
        this.success = Counter.builder("OpenLogClient.counter")
                .tag("name", "success")
                .register(registry);
        this.fail = Counter.builder("OpenLogClient.counter")
                .tag("name", "fail")
                .register(registry);
        log.info("Registering OpenLogClient {} counter meter successful.", client);
    }

    public static class Builder {

        private int maxTotal = 200;
        private int defaultMaxPerRoute = 20;
        private int connectionRequestTimeout = 100;
        private int connectTimeout = 200;
        private int socketTimeout = 500;
        private ThreadPoolExecutor executor;

        {
            int processors = Runtime.getRuntime().availableProcessors();
            int max = 10000;
            executor = new ThreadPoolExecutor(processors, processors, 0, TimeUnit.SECONDS,
                    new LinkedBlockingQueue<>(max), new NamedThreadFactory("openlog-client"));
        }

        private Builder() {}

        public HttpClientOpenLogClient build() {
            return new HttpClientOpenLogClient(executor, maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
        }

        public Builder setMaxTotal(int maxTotal) {
            this.maxTotal = maxTotal;
            return this;
        }

        public Builder setDefaultMaxPerRoute(int defaultMaxPerRoute) {
            this.defaultMaxPerRoute = defaultMaxPerRoute;
            return this;
        }

        public Builder setConnectionRequestTimeout(int connectionRequestTimeout) {
            this.connectionRequestTimeout = connectionRequestTimeout;
            return this;
        }

        public Builder setConnectTimeout(int connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public Builder setSocketTimeout(int socketTimeout) {
            this.socketTimeout = socketTimeout;
            return this;
        }

        public Builder setExecutor(ThreadPoolExecutor executor) {
            this.executor = executor;
            return this;
        }
    }
}
