package com.bxm.openlog.extension.client;

import io.micrometer.core.instrument.binder.MeterBinder;

import java.io.Closeable;
import java.io.IOException;
import java.util.function.Consumer;

/**
 * OpenLog 客户端，发送埋点请求。
 * <p>
 * 如果没有使用 {@code try}-with-resources 构造，那么在必要情况下执行 {@link #close()} 方法，确保异步队列里的任务能够执行完。
 *
 * <pre>
 * 简单的示例：
 * try (OpenLogClient client = HttpClientOpenLogClient.Builder.builder().build()) {
 *     String url = "http://127.0.0.1:8090/inads/s.gif?device=000&mt=100";
 *     client.asyncRequest(url, new Consumer&lt;Fallback&gt;() {
 *
 *         public void accept(Fallback fallback) {
 *             // do something...
 *         }
 *     });
 * }
 * </pre>
 *
 * <p>使用 {@link HttpMethod#POST} 时，需要注意：</p>
 * <li>默认会将 {@code url} 的参数转成 application/x-www-form-urlencoded 使用</li>
 * <li>如果参数值为空，则不会传递这个参数</li>
 * <li>如果参数值有多个，只会传递第 1 个</li>
 *
 * @author allen
 * @date 2021-10-27
 * @see HttpClientOpenLogClient
 * @since 1.0
 */
public interface OpenLogClient extends MeterBinder, Closeable {

    /**
     * 发送一个同步的日志请求
     *
     * @param url 完整的URL
     * @return 埋点服务器是否成功接收
     * @throws IOException 网络异常
     */
    boolean request(String url) throws IOException;

    /**
     * 发送一个同步的日志请求
     *
     * @param url            完整的URL
     * @param httpMethod 请求类型
     * @return 埋点服务器是否成功接收
     * @throws IOException 网络异常
     */
    boolean request(String url, HttpMethod httpMethod) throws IOException;

    /**
     * 发送一个异步的日志请求
     *
     * @param url 完整的URL
     */
    void asyncRequest(String url);

    /**
     * 发送一个异步的日志请求(根据httpMethodEnum)
     *
     * @param url 完整的URL
     * @param httpMethod 请求类型
     */
    void asyncRequest(String url, HttpMethod httpMethod);

    /**
     * 发送一个异步的日志请求
     *
     * @param url      完整的URL
     * @param fallback 埋点服务器接收失败后的回调
     */
    void asyncRequest(String url, Consumer<Fallback> fallback);

    /**
     * 发送一个异步的Post日志请求
     *
     * @param url      完整的URL
     * @param fallback 埋点服务器接收失败后的回调
     * @param httpMethod 请求类型
     */
    void asyncRequest(String url, Consumer<Fallback> fallback, HttpMethod httpMethod);

    /**
     * 返回异步队列当前长度
     *
     * @return 等待执行的数量
     */
    int queueSize();

    @Override
    void close();
}
