package com.bxm.openlog.extension.client.ws;

import com.bxm.openlog.extension.client.Fallback;
import com.bxm.openlog.extension.client.HttpMethod;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.util.ClassUtils;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * @author zhangdong
 * @date 2025/2/13
 */
@Slf4j
@ConditionalOnProperty(value = WsOpenLogClientProperties.PROPERTY_ENABLE, havingValue = "true")
@EnableConfigurationProperties(WsOpenLogClientProperties.class)
public class WsOpenLogClient implements OpenLogClient {

    private Timer timer;
    private Counter success;
    private Counter fail;
    private final ThreadPoolExecutor executor;
    private final AtomicInteger count = new AtomicInteger(0);
    private final List<WsOpenLogClientHandler> handlers = new ArrayList<>();

    public WsOpenLogClient(WsOpenLogClientProperties properties) {
        Integer socketSize = properties.getSocketSize();
        Preconditions.checkNotNull(socketSize, "socketSize must be not null");
        Preconditions.checkArgument(socketSize > 0, "socketSize must be greater than 0");
        WsOpenLogClientProperties.ThreadPoolConfig threadPoolConfig = properties.getThreadPoolConfig();
        for (int i = 0; i < socketSize; i++) {
            handlers.add(new WsOpenLogClientHandler(properties));
        }
        this.executor = new ThreadPoolExecutor(threadPoolConfig.getCoreSize(), threadPoolConfig.getMaxSize(), threadPoolConfig.getKeepAlive(), TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(threadPoolConfig.getQueueSize()), new NamedThreadFactory(threadPoolConfig.getThreadPrefix()));
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        Gauge.builder("ws.openLogClient.queue", 0, value -> queueSize())
                .register(registry);
        this.timer = Timer.builder("ws.openLogClient.timer")
                .register(registry);
        this.success = Counter.builder("ws.openLogClient.counter")
                .tag("name", "success")
                .register(registry);
        this.fail = Counter.builder("ws.openLogClient.counter")
                .tag("name", "fail")
                .register(registry);

    }

    public String parseUrl(String url) {
        UriComponents build = UriComponentsBuilder.fromUriString(url).build();
        String path = build.getPath();
        String query = build.getQuery();
        // 需要考虑使用KeyValueMap构建的URL，会包含p参数。
        MultiValueMap<String, String> queryParams = build.getQueryParams();
        String productParamName = "p";
        if (queryParams.containsKey(productParamName)) {
            return query;
        }
        String p = "";
        if (path != null && path.contains("/")) {
            String[] split = path.split("/");
            if (split.length > 1) {
                p = split[1];
            }
        }
        return productParamName + "=" + p + "&" + query;
    }

    @Override
    public boolean request(String url) throws IOException {
        long start = System.nanoTime();
        String query = parseUrl(url);
        try {
            WsOpenLogClientHandler handler = handlers.get(Math.abs(count.addAndGet(1)) % handlers.size());
            handler.sendMessage(query);
            success.increment();
            return true;
        } catch (Exception e) {
            fail.increment();
            throw new IOException();
        } finally {
            if (Objects.nonNull(timer)) {
                timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
            }
        }
    }

    @Override
    public boolean request(String url, HttpMethod httpMethod) throws IOException {
        return request(url);
    }

    @Override
    public void asyncRequest(String url, Consumer<Fallback> fallback) {
        this.executor.execute(() -> {
            try {
                if (!request(url)) {
                    if (Objects.nonNull(fallback)) {
                        fallback.accept(Fallback.builder().url(url).build());
                    }
                }
            } catch (IOException e) {
                if (Objects.nonNull(fallback)) {
                    fallback.accept(Fallback.builder().url(url).exception(e).build());
                }
            }
        });
    }

    @Override
    public void asyncRequest(String url, Consumer<Fallback> fallback, HttpMethod httpMethod) {
        asyncRequest(url, fallback);
    }

    @Override
    public void asyncRequest(String url) {
        this.asyncRequest(url, HttpMethod.GET);
    }

    @Override
    public void asyncRequest(String url, HttpMethod httpMethod) {
        this.asyncRequest(url, null, httpMethod);
    }


    @Override
    public int queueSize() {
        return executor.getQueue().size();
    }

    @Override
    public void close() {
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(30, TimeUnit.SECONDS)) {
                log.warn("This executor was forced terminated!");
            }
        } catch (InterruptedException e) {
            log.error("awaitTermination: ", e);
        }
    }

}
