package com.bxm.openlog.extension.client.ws;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.curator.shaded.com.google.common.collect.Lists;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.PingMessage;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.client.WebSocketConnectionManager;
import org.springframework.web.socket.client.standard.StandardWebSocketClient;
import org.springframework.web.socket.handler.ConcurrentWebSocketSessionDecorator;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author zhangdong
 * @date 2025/2/13
 */
@Slf4j
public class WsOpenLogClientHandler extends TextWebSocketHandler {

    private WebSocketSession session;
    private final ScheduledExecutorService scheduler = new ScheduledThreadPoolExecutor(1);
    private final List<CloseStatus> reConnectionStatus = Lists.newArrayList(
            CloseStatus.NOT_ACCEPTABLE,
            CloseStatus.BAD_DATA,
            CloseStatus.SERVER_ERROR,
            CloseStatus.TOO_BIG_TO_PROCESS,
            CloseStatus.SERVICE_OVERLOAD);
    private final WsOpenLogClientProperties properties;
    private final WebSocketConnectionManager manager;

    public WsOpenLogClientHandler(WsOpenLogClientProperties properties) {
        this.properties = properties;
        String endpoint = properties.getEndpoint();
        Preconditions.checkArgument(StringUtils.isNotBlank(endpoint), "endpoint is null or empty");
        StandardWebSocketClient socketClient = new StandardWebSocketClient();
        WebSocketConnectionManager manager = new WebSocketConnectionManager(socketClient, this, endpoint);
        manager.start();
        this.manager = manager;
    }

    @Override
    public void afterConnectionEstablished(WebSocketSession session) throws Exception {
        this.session = new ConcurrentWebSocketSessionDecorator(session, Integer.MAX_VALUE, Integer.MAX_VALUE);
        startHeartbeat();
    }

    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) throws Exception {
        if (reConnectionStatus.contains(status)) {
            int failSize = 0;
            while (failSize < 3) {
                try {
                    manager.stop();
                    scheduler.shutdownNow();
                    manager.start();
                    return;
                } catch (Exception e) {
                    failSize++;
                    log.error("reConnection error count:{}", failSize, e);
                }
            }
            throw new RuntimeException("reConnection error");
        }
    }

    public void sendMessage(String data) throws Exception {
        session.sendMessage(new TextMessage(data));
    }

    private void startHeartbeat() {
        scheduler.scheduleAtFixedRate(() -> {
            if (session != null && session.isOpen()) {
                try {
                    PingMessage pingMessage = new PingMessage();
                    session.sendMessage(pingMessage);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }, 0, properties.getHeartCycle(), TimeUnit.MILLISECONDS);
    }
}
