package com.bxm.openlog.sdk;

import com.bxm.openlog.sdk.params.PublicParam;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Function;

/**
 * 参数名和值的映射对象。它实际就是一个 Map。但是值是一个集合。
 *
 * @author allen
 * @date 2021-09-27
 * @since 1.0
 */
@Slf4j
public final class KeyValueMap extends LinkedMultiValueMap<String, String> {

    private static final String[] IMMUTABLE_NAMES = new String[] {
            PublicParam.P,
            PublicParam.MT
    };

    public KeyValueMap() {
    }

    public KeyValueMap(int initialCapacity) {
        super(initialCapacity);
    }

    public KeyValueMap(Map<String, List<String>> otherMap) {
        super(otherMap);
    }

    /**
     * 使用当前对象生成一个 OpenLog 的埋点请求地址。
     * @param rDomain 请求域。填写完整的域：scheme://host:port
     * @return 请求地址。
     * @see #createOpenLogRequestUri(String, KeyValueMap)
     */
    public String createOpenLogRequestUri(String rDomain) {
        return this.createOpenLogRequestUri(rDomain, null);
    }

    /**
     * 使用当前对象生成一个 OpenLog 的埋点请求地址。
     * 还可通过指定 {@code customizedParams} 来替换当前对象已存在的参数。
     *
     * @param rDomain 请求域。填写完整的域：scheme://host:port
     * @param customizeParams 自定义参数
     * @return 请求地址。类似这样的：http://openlog-inner.bianxianmao.com/ads/s.gif?device=123&p=pangu
     */
    public String createOpenLogRequestUri(String rDomain, KeyValueMap customizeParams) {
        String production = getProduction();
        if (StringUtils.isBlank(production)) {
            throw new NullPointerException("Value is blank of 'production', please set to value like this \"this.put(PublicParam.P, Production.PANGU.getName())\"");
        }
        String url = UriComponentsBuilder.fromUriString(rDomain)
                .pathSegment(production, "s.gif")
                .build()
                .toString();
        return this.toRequestUri(url, customizeParams);
    }

    /**
     * 在 {@code url} 基础上加上当前对象的参数键值对，生成一个新的 GET 埋点请求地址。
     * 还可通过指定 {@code customizedParams} 来替换当前对象已存在的参数。
     *
     * @param url 基础地址
     * @param customizeParams 自定义参数
     * @return 请求地址。
     */
    private String toRequestUri(String url, KeyValueMap customizeParams) {
        UriComponentsBuilder builder = UriComponentsBuilder.fromUriString(url).replaceQueryParams(this);
        if (Objects.nonNull(customizeParams)) {
            customizeParams.forEach(new BiConsumer<String, List<String>>() {
                @Override
                public void accept(String s, List<String> strings) {
                    if (Objects.nonNull(strings)) {
                        String[] array = strings.toArray(new String[0]);
                        builder.replaceQueryParam(s, array);
                    }
                }
            });
        }
        return builder.build().toString();
    }

    /**
     * 转成简单的对象。如果遇到某个参数值是 {@link List}，那么只会取第一个值赋值。
     *
     * @param clazz 类
     * @param <T> 对象类型
     * @return 对象实体
     */
    public <T> T toSimpleObject(Class<T> clazz) {
        return toSimpleObject(new Function<Map<String, String>, T>() {
            @Override
            public T apply(Map<String, String> stringStringMap) {
                return JsonHelper.convert(JsonHelper.convert2bytes(stringStringMap), clazz);
            }
        });
    }

    private <T> T toSimpleObject(Function<Map<String, String>, T> converter) {
        Map<String, String> single = toSingleValueMap();
        return converter.apply(single);
    }

    /**
     * 返回 IP 地址。
     * @return IP
     */
    public String getIp() {
        return getFirst(PublicParam.IP);
    }

    /**
     * 返回 User-Agent。
     * @return User-Agent
     */
    public String getUa() {
        return getFirst(PublicParam.UA);
    }

    /**
     * 返回请求来源。
     * @return Referrer
     */
    public String getRef() {
        return getFirst(PublicParam.REFER);
    }

    /**
     * 返回业务产品。
     * @return Business Product
     */
    public String getProduction() {
        return getFirst(PublicParam.P);
    }

    /**
     * 返回行为类型。
     * @return Model type
     */
    public String getMt() {
        return getFirst(PublicParam.MT);
    }

    /**
     * 返回版本号。
     * @return Version
     */
    public String getVersion() {
        return getFirst(PublicParam.VER);
    }

    /**
     * 返回传入的 {@code key} 在当前映射里的 第一个 {@code value} 值，如果 {@code value} 是空格、空 ("") 或null ，则返回 {@code defaultValue} 值。
     *
     * @param key 映射的 key
     * @param defaultValue 如果 {@code value} 为空格、空 ("") 或null返回的默认字符串，可能为 null
     * @return 传入的字符串，或默认值
     */
    public String defaultIfBlank(String key, String defaultValue) {
        return StringUtils.defaultIfBlank(getFirst(key), defaultValue);
    }

    /**
     * 返回传入的 {@code key} 在当前映射里的 第一个 {@code value} 值，如果 {@code value} 是空（""）或null ，则返回 {@code defaultValue} 值。
     *
     * @param key 映射的 key
     * @param defaultValue 如果 {@code value} 为空 ("") 或null返回的默认字符串，可能为 null
     * @return 传入的字符串，或默认值
     */
    public String defaultIfEmpty(String key, String defaultValue) {
        return StringUtils.defaultIfEmpty(getFirst(key), defaultValue);
    }

    /**
     * 创建这个 {@code KeyValueMap} 的深层副本。
     * @return 此 Map 的副本，包括沿 KeyValueMap.addAll 语义行的每个持有值的 List 条目的副本（始终为每个条目使用独立的可修改 LinkedList）
     */
    @Override
    public KeyValueMap deepCopy() {
        return this.deepCopy(new String[0]);
    }

    /**
     * 创建这个 {@code KeyValueMap} 的深层副本。
     * @param withoutKey 复制时排除的 Key
     * @return 此 Map 的副本，包括沿 KeyValueMap.addAll 语义行的每个持有值的 List 条目的副本（始终为每个条目使用独立的可修改 LinkedList）
     */
    public KeyValueMap deepCopy(String...withoutKey) {
        KeyValueMap map = new KeyValueMap(size());
        forEach((key, values) -> {
            if (Objects.isNull(withoutKey) || !ArrayUtils.contains(withoutKey, key)) {
                map.put(key, new LinkedList<>(values));
            }
        });
        return map;
    }


    /**
     * 调用 {@code value} 的 {@code toString} 方法将返回值添加到 {@code key} 键中。
     * 如果映射先前包含键的映射，则旧值将替换为指定值。
     * @param key 键
     * @param value 值，如果传入 {@code null}，那么会赋值 {@code null}
     * @return 与key关联的先前值，如果没有key 的映射，则为null 。
     * @see #put(String, String)
     */
    public List<String> put(String key, Object value) {
        return this.put(key, TypeHelper.castToString(value));
    }

    /**
     * 将 {@code value} 添加到 {@code key} 键中。如果映射先前包含键的映射，则旧值将替换为指定值。
     * @param key 键
     * @param value 值
     * @return 与key关联的先前值，如果没有key 的映射，则为null 。
     */
    public List<String> put(String key, String value) {
        return this.put(key, Lists.newArrayList(value));
    }

    @Override
    public List<String> put(String key, List<String> value) {
        if (containsKey(key) && ArrayUtils.contains(IMMUTABLE_NAMES, key)) {
            throw new UnsupportedOperationException("Not allowed modify key: " + key);
        }
        return super.put(key, value);
    }

    @Override
    public void putAll(Map<? extends String, ? extends List<String>> map) {
        if (Objects.isNull(map)) {
            return;
        }
        Set<? extends String> keySet = map.keySet();
        for (String key : keySet) {
            if (containsKey(key) && ArrayUtils.contains(IMMUTABLE_NAMES, key)) {
                map.remove(key);
                if (log.isWarnEnabled()) {
                    log.warn("illegal key '{}', cause not allowed!", key);
                }
            }
        }
        super.putAll(map);
    }

    @Override
    public List<String> remove(Object key) {
        if (ArrayUtils.contains(IMMUTABLE_NAMES, key)) {
            throw new UnsupportedOperationException("Not allowed remove key: " + key);
        }
        return super.remove(key);
    }

    @Override
    public boolean remove(Object key, Object value) {
        if (ArrayUtils.contains(IMMUTABLE_NAMES, key)) {
            throw new UnsupportedOperationException("Not allowed remove key: " + key);
        }
        return super.remove(key, value);
    }
}
