package com.bxm.adsmanager.timer.qualify;

import com.alibaba.fastjson.JSON;
import com.bxm.adsmanager.dal.mapper.advertiser.ext.TblAdQualifyMapperExt;
import com.bxm.adsmanager.integration.message.MessagerServiceIntegration;
import com.bxm.adsmanager.model.dao.advertiser.TblAdQualify;
import com.bxm.adsmanager.model.ro.QualifyContentRo;
import com.bxm.adsmanager.service.advertiser.AdvertiserQualifyService;
import com.bxm.adsmanager.timer.media.MediaPositonTestJob;
import com.bxm.adsmanager.utils.DateUtils;
import com.bxm.messager.facade.dto.MessageInfoDTO;
import com.bxm.util.DateUtil;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.bxm.warcar.utils.DateHelper;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 *
 * 定时任务 查看资质过期并推送消息给广告主
 *
 * @Author: pf.w
 * @Date: 2022/2/16 13:29
 */

@Component
public class QualifyCHeckJob {
    private static final Logger LOGGER = LoggerFactory.getLogger(QualifyCHeckJob.class);

    @Qualifier("jedisUpdater")
    @Autowired
    private JedisUpdater updater;
    @Qualifier("jedisFetcher")
    @Autowired
    private JedisFetcher fetcher;

    @Autowired
    TblAdQualifyMapperExt tblAdQualifyMapperExt;

    @Autowired
    MessagerServiceIntegration messagerServiceIntegration;

    @Autowired
    AdvertiserQualifyService advertiserQualifyService;

    /**
     * 资质即将过期判断阀值
     */
    public static Integer LIMIT_DAYS = 3;

    /* 1：已过期
     * 2：未过期
     * 3：即将过期   资质到期前三天
     */
    public static Integer EXPIRE_STATUS_YES = 1;
    public static Integer EXPIRE_STATUS_NO = 2;
    public static Integer EXPIRE_STATUS_SOON = 3;

    //到期标志
    public static String EXPIRE__YES = "EXPIRE__YES";
    //即将到期标志
    public static String EXPIRE__SOON = "EXPIRE__SOON";


    public void checkQualify(){
        //获取所有资质内容列表
        List<TblAdQualify> qualifyList = tblAdQualifyMapperExt.qualifyList();

        if(CollectionUtils.isEmpty(qualifyList)){
            return;
        }

        for (TblAdQualify tblAdQualify : qualifyList) {
            String qualifyContentJsonStr = tblAdQualify.getQualifyContent();
            //判空
            if(Objects.isNull(qualifyContentJsonStr)){
                continue;
            }

            List<QualifyContentRo> qualifyContentRo = JSON.parseArray(qualifyContentJsonStr,QualifyContentRo.class);
            //判空
            if(CollectionUtils.isEmpty(qualifyContentRo)){
                continue;
            }

            //资质到期判断flag
            boolean expireStatusYesFlag = false;
            //资质未到期判断flag
            boolean expireStatusNoFlag = false;
            //资质即将到期判断flag
            boolean expireStatusSoonFlag = false;

            for (QualifyContentRo contentRo : qualifyContentRo) {

                if(contentRo.isLongTime()){
                    expireStatusNoFlag = true;
                    continue;
                }
                //判空
                if(Objects.isNull(contentRo.getExpireTime()) || StringUtils.isBlank(contentRo.getExpireTime())){
                    continue;
                }

                //判断到期相差天数
                int diffDays =  DateUtil.countDiffDateDays(new Date(),DateHelper.parse(contentRo.getExpireTime(),DateHelper.PATTERN_STR10));

                //资质到期时间前 满足阀值则推送消息告知广告主资质要过期
                if( diffDays>=0 && diffDays <= LIMIT_DAYS  ){
                    //发送过期通知
                    sendMessage(contentRo,tblAdQualify,EXPIRE__SOON);

                    expireStatusSoonFlag = true;
                }

                //如果资质到期 资质状态变更为审核拒绝  拒绝原因为：资质到期
                if(diffDays < 0){
                    //发送过期通知
                    sendMessage(contentRo,tblAdQualify,EXPIRE__YES);

                    expireStatusYesFlag = true;
                }
                //资质未到期
                if(diffDays > LIMIT_DAYS){
                    expireStatusNoFlag = true;
                }
            }

            //变更资质到期状态
            this.updateExpireStatusOperate(tblAdQualify,expireStatusYesFlag,expireStatusNoFlag,expireStatusSoonFlag);
        }
    }

    /**
     * 变更资质是否过期状态
     */
    private void updateExpireStatusOperate(TblAdQualify tblAdQualify,boolean expireStatusYesFlag
            ,boolean expireStatusNoFlag
            ,boolean expireStatusSoonFlag){

        if(expireStatusYesFlag){
            TblAdQualify record = new TblAdQualify();
            record.setId(tblAdQualify.getId());
            //审核拒绝
            record.setAuditStatus(3);
            record.setRefuseReason("资质到期");
            record.setRefuseRemark("资质到期");
            //资质到期
            record.setExpireStatus(EXPIRE_STATUS_YES);
            try {
                advertiserQualifyService.updateAdQualifyById(record);
            } catch (Exception e) {
                LOGGER.error("资质到期变更错误",e);
            }
        }

        if(expireStatusNoFlag){
            this.updateExpireStatusForDb(tblAdQualify,EXPIRE_STATUS_NO);
        }

        if(expireStatusSoonFlag){
            this.updateExpireStatusForDb(tblAdQualify,EXPIRE_STATUS_SOON);
        }
    }

    /**
     * 变更资质是否过期状态
     * @param tblAdQualify
     * @param status
     */
    private void updateExpireStatusForDb(TblAdQualify tblAdQualify,Integer status){
        TblAdQualify record = new TblAdQualify();
        record.setId(tblAdQualify.getId());
        //资质未过期
        record.setExpireStatus(status);
        try {
            advertiserQualifyService.updateAdQualifyById(record);
        } catch (Exception e) {
            LOGGER.error("资质过期状态变更错误",e);
        }
    }

    private void sendMessage(QualifyContentRo contentRo,TblAdQualify tblAdQualify,String expireFlag){

        String content = "到期";

        if(EXPIRE__YES.equals(expireFlag)){
            String expireYesFlag = fetcher.hfetch(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getLabel()+EXPIRE__YES,String.class);

            if(Objects.isNull(expireYesFlag) || StringUtils.isBlank(expireYesFlag)){
                content = "已到期";

                boolean sendResult =  sendMessageDo(contentRo,tblAdQualify,content);

                if(sendResult){
                    updater.hupdate(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getLabel()+EXPIRE__YES,"ok");
                }

            }
        }

        if(EXPIRE__SOON.equals(expireFlag)){
            String expireSoonFlag = fetcher.hfetch(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getLabel()+EXPIRE__SOON,String.class);

            if(Objects.isNull(expireSoonFlag) || StringUtils.isBlank(expireSoonFlag)){
                content = "即将到期";

                boolean sendResult =  sendMessageDo(contentRo,tblAdQualify,content);

                if(sendResult){
                    updater.hupdate(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getLabel()+EXPIRE__SOON,"ok");
                }

            }
        }

//        //缓存查看是否发送过
//       Integer flag =  fetcher.hfetch(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getQualifyName(),Integer.class);
//
//       //允许推送两次  当flag计数为2时候 将不再推送消息
//       if(Objects.isNull(flag) || flag < 2){
//
//           MessageInfoDTO messageInfoDTO = MessageInfoDTO.builder()
//                   .title("资质过期通知!")
//                   .pushGoalUserType(1)
//                   .pushGoalUserContent(tblAdQualify.getAdvertiserId().toString())
//                   .channel(1)
//                   .pushTopicType(2)
//                   .pushStatus(1)
//                   .pushContent("尊敬的客户，您提供的《"+contentRo.getQualifyName()+"》"+content+"，请尽快补充新的资质，以免影响后续广告投放，谢谢合作！")
//                   .createUser("wangpengfei")
//                   .build();
//
//           //发送资质过期通知
//          Boolean sendResult = messagerServiceIntegration.pushMessage(messageInfoDTO);
//
//          //缓存记录发送结果
//          if(sendResult){
//              if(Objects.isNull(flag)){
//                  updater.hupdate(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getQualifyName()+EXPIRE_STATUS_YES.toString(),1);
//              }else{
//                  updater.hupdate(() -> StringUtils.join(new String[]{"QUALIFY", "STATUS", tblAdQualify.getAdvertiserId().toString()}, ":"),contentRo.getQualifyName()+EXPIRE_STATUS_SOON.toString(),2);
//              }
//
//          }

//       }
    }

    private boolean sendMessageDo(QualifyContentRo contentRo,TblAdQualify tblAdQualify,String content){
        MessageInfoDTO messageInfoDTO = MessageInfoDTO.builder()
                .title("资质过期通知!")
                .pushGoalUserType(1)
                .pushGoalUserContent(tblAdQualify.getAdvertiserId().toString())
                .channel(1)
                .pushTopicType(2)
                .pushStatus(1)
                .pushContent("尊敬的客户，您提供的《"+contentRo.getLabel()+"》"+content+"，请尽快补充新的资质，以免影响后续广告投放，谢谢合作！")
                .createUser("wangpengfei")
                .build();

        //发送资质过期通知
        return messagerServiceIntegration.pushMessage(messageInfoDTO);
    }

    public static void main(String[] args) {
        String time = "2022-02-24";
        int diffDays =  DateUtil.countDiffDateDays(new Date(),DateHelper.parse(time,DateHelper.PATTERN_STR10));
        System.out.println(diffDays);
    }

}
