package com.bxm.adsmanager.timer.ticket;

import com.bxm.adsmanager.dal.mapper.opentime.TicketOpeningTimeMapper;
import com.bxm.adsmanager.model.dao.opentime.TicketOpeningTime;
import com.bxm.adsmanager.model.dao.report.TicketLogTime;
import com.bxm.adsmanager.service.adkeeper.AdRulesService;
import com.bxm.adsmanager.utils.DateFormatUtil;
import com.bxm.adsprod.facade.ticket.Ticket;
import com.bxm.adsprod.facade.ticket.TicketService;
import com.bxm.log.facade.dao.LogModel;
import com.bxm.log.facade.dto.CollectionName;
import com.bxm.log.facade.dto.LogModelDto;
import com.bxm.log.facade.service.LogsService;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 记录礼券的开启时间的job
 *
 * @ClassName AdTicketOpenTimeJob
 * @CopyRright (c) 2018-bxm：杭州微财网络科技有限公司
 * @Author kk.xie
 * @Date 2018/7/9 15:11
 * @Version 1.0
 * @Modifier kk.xie
 * @Modify Date 2018/7/9 15:11
 **/
@Component
public class AdTicketOpenTimeJob {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdTicketOpenTimeJob.class);

    @Autowired
    @Qualifier("jedisFetcher")
    private Fetcher fetcher;

    @Autowired
    private LogsService logsService;

    @Resource
    private TicketService ticketService;

    @Autowired
    private AdRulesService adRulesService;

    @Autowired
    private TicketOpeningTimeMapper ticketOpeningTimeMapper;

    private DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    private Pattern pattern = Pattern.compile("\"times\"\\s*:\\s*\"(\\d{1,2}-\\d{1,2}-\\d+(\\s*,\\s*\\d{1,2}-\\d{1,2}-\\d+)*)\"");

    /**
     * 记录礼券的开启时间的job,每日0点执行，统计昨天的礼券实际开启时间
     *
     * @param
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/7/9 20:08
     */
    //@Scheduled(cron="0 0 0 * * *")
    public void ticketOpenTime() throws Exception{
        LOGGER.info("记录礼券的开启时间的job，开始执行");

        List<Ticket> allTickets = ticketService.getAllTickets();
        // 统计昨天的礼券操作日志，计算礼券的实际投放时间
        LocalDate localDate = LocalDate.now().minusDays(1L);
        String datetime = dateTimeFormatter.format(localDate);

        for(Ticket ticket : allTickets){
            List<TicketLogTime> openTimeList = getTicketLogTimes(datetime, ticket.getId().longValue());
            for(TicketLogTime ticketLogTime : openTimeList){
                TicketOpeningTime ticketOpeningTime = new TicketOpeningTime();
                try {
                    ticketOpeningTime.setStartTime(dateFormat.parse(datetime + " "+ ticketLogTime.getStartTime() + ":00:00"));
                    if(24 == ticketLogTime.getEndTime()){
                        ticketOpeningTime.setEndTime(dateFormat.parse(datetime + " " + "23:59:59"));
                    }else{
                        ticketOpeningTime.setEndTime(dateFormat.parse(datetime + " "+ ticketLogTime.getEndTime() + ":00:00"));
                    }

                } catch (ParseException e) {
                    LOGGER.error("记录礼券的开启时间的job出错", e);
                }
                ticketOpeningTime.setTicketId(ticket.getId().longValue());
                ticketOpeningTimeMapper.insert(ticketOpeningTime);
            }
        }

        LOGGER.info("记录礼券的开启时间的job，执行成功");
    }

    /**
     * 查询礼券的操作日志，计算礼券的实际投放时间
     *
     * @param datetime
     * @param ticketId
     * @return void
     * @throws
     * @author kk.xie
     * @date 2018/7/4 17:20
     */
    private List<TicketLogTime>  getTicketLogTimes(String datetime, Long ticketId) {
        List<TicketLogTime> ticketLogTimeResultList = Lists.newArrayList();
        // 全天投放默认值
        TicketLogTime ticketLogTimeDefault = new TicketLogTime(0, 24, 24);

        // 查询券是否有时间段投放规则
        String timeRule = adRulesService.findByTicketId(1,5, ticketId);
        if(StringUtils.isNotEmpty(timeRule)){
            StringBuilder sb = new StringBuilder();
            String []rules = timeRule.split(",");
            for(int i=0; i<rules.length ;i++){
                String rule = rules[i];
                String [] str = rule.split("-");
                if(str.length == 3){
                    TicketLogTime ticketLogTime = new TicketLogTime(Integer.valueOf(str[0]), Integer.valueOf(str[1]), 0);
                    ticketLogTimeResultList.add(ticketLogTime);
                }
            }
        }
        if(CollectionUtils.isEmpty(ticketLogTimeResultList)){
            ticketLogTimeResultList.add(ticketLogTimeDefault);
        }

        // dateTime的时间戳
        long dateTimeStamp = DateFormatUtil.getMillisecondByDate(datetime);
        long dateTimePlusOneDayStamp = DateFormatUtil.getMillisecondByPlusDate(datetime, 1);
        long dateTimeMinusOneMonthStamp = DateFormatUtil.getMillisecondByMinusDate(datetime, 30);

        // 查询datatime天 有几条修改记录，再一次性查询datatime天所有的修改记录
        long pageSize = getPageTotal(dateTimeStamp, dateTimePlusOneDayStamp, ticketId);
        List<LogModel> logModelList = getDateTimeLog(1, pageSize, dateTimeStamp, dateTimePlusOneDayStamp, ticketId);

        List<List<TicketLogTime>> ticketLogTimeList = getTicketLogTime(logModelList);
        // 当天无投放时间段的修改记录，返回礼券当前的投放时间段
        if(CollectionUtils.isEmpty(ticketLogTimeList)){
            // 集合处理
            return dealTicketLogTime(ticketLogTimeResultList);
        }

        // 当前有投放时间段的修改记录需要依赖上一次修改的时候的投放时间段

        // 查询dateTime前30天的历史记录
        long beforeMonthPageSize = getPageTotal(dateTimeMinusOneMonthStamp, dateTimeStamp, ticketId);
        List<LogModel> beforeMonthLogModelList = getDateTimeLog(1, beforeMonthPageSize, dateTimeMinusOneMonthStamp, dateTimeStamp, ticketId);
        // 由于现有日志API是正序排列，所以将一个月的查询结果倒序
        Collections.reverse(beforeMonthLogModelList);
        // 返回一个月内最近一条修改投放时间段的记录信息
        List<List<TicketLogTime>> ticketBeforeMonthLogTimeList = getLastTicketLogTime(beforeMonthLogModelList);
        // 近一个月无有效时间段的修改记录，则返回当前礼券的投放时间段
        if(CollectionUtils.isEmpty(ticketBeforeMonthLogTimeList)){
            return ticketLogTimeResultList;
        }

        // 将前一条修改记录合并到今天修改记录中
        ticketLogTimeList.addAll(0, ticketBeforeMonthLogTimeList);
        // 保存礼券投放的时间
        List<TicketLogTime> pvTimeList = Lists.newArrayList();

        for (int i = 0; i < ticketLogTimeList.size(); i++) {
            if (i == 0) {
                continue;
            }
            List<TicketLogTime> logs = ticketLogTimeList.get(i);
            // 取昨天最后修改记录中的 投放时间段
            List<TicketLogTime> beforeDayLogs = ticketLogTimeList.get(i-1);
            // 遍历今天的修改记录
            for (TicketLogTime ticketLogTime : logs) {
                int modifyHour = ticketLogTime.getModofyTime();

                for (TicketLogTime beforeDayTicketLogTime : beforeDayLogs) {
                    // 记录修改前已经投放的时间段
                    int beforeDayStartHour = beforeDayTicketLogTime.getStartTime();
                    int beforeDayEndHour = beforeDayTicketLogTime.getEndTime();
                    if (modifyHour < beforeDayStartHour) {
                        // 修改时间小于投放开始时间，无历史投放记录
                    } else if (beforeDayStartHour <= modifyHour && beforeDayEndHour >= modifyHour) {
                        // 修改时间在投放时间范围内，则记录已经投放的时间[beforeDayStartHour, modifyHour]
                        TicketLogTime time = new TicketLogTime(beforeDayStartHour, modifyHour, modifyHour);
                        pvTimeList.add(time);
                    } else {
                        // 修改时间大于投放结束时间，则记录已投放的时间[beforeDayStartHour,beforeDayEndHour]
                        TicketLogTime time = new TicketLogTime(beforeDayStartHour, beforeDayEndHour, modifyHour);
                        pvTimeList.add(time);
                    }
                }

                // 记录预投放时间段
                int startTime = ticketLogTime.getStartTime();
                int endTime = ticketLogTime.getEndTime();
                if (modifyHour < startTime) {
                    // 修改时间小于开始投放时间，预投放时间段[startTime, endTime]
                    TicketLogTime time = new TicketLogTime(startTime, endTime, modifyHour);
                    pvTimeList.add(time);
                } else if (startTime <= modifyHour && modifyHour <= endTime) {
                    // 修改时间处于 开始投放时间和结束投放时间之间，预投放时间为[modifyHour, endTime]
                    TicketLogTime time = new TicketLogTime(modifyHour, endTime, modifyHour);
                    pvTimeList.add(time);
                } else {
                    // 修改时间大于结束投放时间，无预投放时间
                }
            }
        }

        return dealTicketLogTime(pvTimeList);
    }

    private List<TicketLogTime> dealTicketLogTime(List<TicketLogTime> ticketLogTimes){
        // 投放的小时放到 Set中自动去重
        Set<Integer> hourSet = new HashSet<>();
        for(TicketLogTime ticketLogTime : ticketLogTimes){
            int startTime = ticketLogTime.getStartTime();
            int endTime = ticketLogTime.getEndTime();
            hourSet.add(startTime);

            int hour = startTime++;
            while(hour <= endTime){
                hourSet.add(hour);
                hour ++;
            }
        }
        // 从小到大排序
        Object[] objHour = hourSet.toArray();
        Integer[] intHour = new Integer[objHour.length];
        for(int k = 0; k < objHour.length; k++){
            int hour = (Integer) objHour[k];
            intHour[k] = hour;
        }

        for(int i=0;i<intHour.length;i++) {
            int tem = i;
            for(int j=i;j<intHour.length;j++) {
                if(intHour[j] < intHour[tem]) {
                    tem = j;
                }
            }
            int temp1 = intHour[i];
            intHour[i] = intHour[tem];
            intHour[tem] = temp1;
        }

        List<TicketLogTime> pvTimeResultList = Lists.newArrayList();
        // 连接的时间进行拼接
        int start = -1;
        for(int i = 0; i < intHour.length; i++){
            if(start == -1){
                start = intHour[i];
            }
            // 连续的时间进行则继续
            if( i < intHour.length - 1 && intHour[i] + 1 == intHour[i + 1]){
                continue;
            }else{
                int end = intHour[i];
                TicketLogTime ticketLogTime = new TicketLogTime(start, end ,end);
                pvTimeResultList.add(ticketLogTime);
                start = -1;
            }
        }
        return pvTimeResultList;
    }

    /**
     * 取datatime 天有几条修改记录，例如：查询20180705的数据 即 2018-07-05 00:00:00 的时间戳 到 2018-07-06 00:00:00 的时间戳范围的数据
     *
     * @param dateTimeStamp  查询的时间
     * @param dateTimePlusOneDayStamp 查询结束的时间，加一天的值
     * @param ticketId
     * @return long
     * @throws
     * @author kk.xie
     * @date 2018/7/5 10:03
     */
    private long getPageTotal(long dateTimeStamp, long dateTimePlusOneDayStamp, Object ticketId) {
        LogModelDto dto = new LogModelDto();
        try {
            dto.setPageNum(1);
            dto.setPageSize(1);
            dto.setStartDateTime(dateTimeStamp);
            dto.setEndDateTime(dateTimePlusOneDayStamp);
            dto.setAdticketId(String.valueOf(ticketId));
            dto.setCollectionName(CollectionName.ADSMANAGER);
            com.bxm.log.facade.dao.Page<LogModel> logModelPage = logsService.find(dto);
            return logModelPage.getTotal();
        } catch (Exception e) {
            LOGGER.error("统计礼券昨日发券时间总数出错了！dto={},e={}", JsonHelper.convert(dto), ExceptionUtils.getFullStackTrace(e), e);
        } finally {
        }
        return 0L;
    }

    /**
     * 查询dateTime天中的操作日志
     *
     * @param pageNum
     * @param pageSize
   * @param dateTimeStamp
   * @param dateTimePlusOneDayStamp
   * @param ticketId
     * @return java.util.List<com.bxm.log.facade.dao.LogModel>
     * @throws
     * @author kk.xie
     * @date 2018/7/5 10:27
     */
    private List<LogModel> getDateTimeLog(int pageNum, long pageSize, long dateTimeStamp, long dateTimePlusOneDayStamp, Object ticketId) {
        LogModelDto dto = new LogModelDto();
        try {
            dto.setPageNum(pageNum);
            dto.setPageSize((int)pageSize);
            dto.setStartDateTime(dateTimeStamp);
            dto.setEndDateTime(dateTimePlusOneDayStamp);
            dto.setAdticketId(String.valueOf(ticketId));
            dto.setCollectionName(CollectionName.ADSMANAGER);
            com.bxm.log.facade.dao.Page<LogModel> logModelPage = logsService.find(dto);
            List<LogModel>logModelList = logModelPage.getList();
            if(CollectionUtils.isEmpty(logModelList)){
                return Lists.newArrayList();
            }
            return logModelList;
        } catch (Exception e) {
            LOGGER.error("统计礼券昨日发券时间列表出错了！dto={},e={}", JsonHelper.convert(dto), ExceptionUtils.getFullStackTrace(e), e);
        } finally {
        }
        return Lists.newArrayList();
    }

    /**
     * 转换操作日志
     *
     * @param logModelList
     * @return java.util.List<java.util.List<com.bxm.adsmanager.model.dao.report.TicketLogTime>>
     * @throws
     * @author kk.xie
     * @date 2018/7/9 16:33
     */
    private List<List<TicketLogTime>> getTicketLogTime(List<LogModel> logModelList) {
        List<List<TicketLogTime>> list = Lists.newArrayList();
        for (LogModel logModel : logModelList) {
            List<TicketLogTime> ticketLogTimeList = Lists.newArrayList();
            // 日志中的信息使用正则捕获组去捕获 投放时间  "times": "9-18-50000,5-6-8888"，
            // 日志记录的是修改后的结果，上一次修改信息需要查询上一条日志记录中的信息
            String content = logModel.getContent();
            Date dateTime = logModel.getDateTime();
            // 修改的时间
            Matcher matcher = pattern.matcher(content);
            while (matcher.find()) {
                String str = matcher.group(1);
                // 满足捕获组，下面代码不会出现异常
                String[] size = str.split(",");
                int modifyTime = DateFormatUtil.getDateHour(dateTime);
                for (String s : size) {
                    String[] times = s.split("-");
                    int startTime = Integer.valueOf(times[0]);
                    int endTime = Integer.valueOf(times[1]);
                    TicketLogTime ticketLogTime = new TicketLogTime(startTime, endTime, modifyTime);
                    ticketLogTimeList.add(ticketLogTime);
                }
                list.add(ticketLogTimeList);
            }
        }
        return list;
    }

    /**
     * 取最近一个月，最后一条有修改礼券投放时间段的日志修改信息
     *
     * @param beforeMonthLogModelList
     * @return java.util.List<java.util.List<com.bxm.adsmanager.model.dao.report.TicketLogTime>>
     * @throws
     * @author kk.xie
     * @date 2018/7/5 16:05
     */
    private List<List<TicketLogTime>> getLastTicketLogTime(List<LogModel> beforeMonthLogModelList) {
        List<List<TicketLogTime>> list = Lists.newArrayList();
        for (LogModel logModel : beforeMonthLogModelList) {
            List<TicketLogTime> ticketLogTimeList = Lists.newArrayList();
            // 日志中的信息使用正则捕获组去捕获 投放时间  "times": "9-18-50000,5-6-8888"，
            // 日志记录的是修改后的结果，上一次修改信息需要查询上一条日志记录中的信息
            String content = logModel.getContent();
            Date dateTime = logModel.getDateTime();
            // 修改的时间
            Matcher matcher = pattern.matcher(content);
            while (matcher.find()) {
                String str = matcher.group(1);
                // 满足捕获组，下面代码不会出现异常
                String[] size = str.split(",");
                int modifyTime = DateFormatUtil.getDateHour(dateTime);
                for (String s : size) {
                    String[] times = s.split("-");
                    int startTime = Integer.valueOf(times[0]);
                    int endTime = Integer.valueOf(times[1]);
                    TicketLogTime ticketLogTime = new TicketLogTime(startTime, endTime, modifyTime);
                    ticketLogTimeList.add(ticketLogTime);
                }
                list.add(ticketLogTimeList);
            }
            if(CollectionUtils.isNotEmpty(list)){
                break;
            }
        }
        return list;
    }
}
