package com.bxm.pangu.rta.common.amap;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.experimental.Accessors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.springframework.http.MediaType;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * 高德
 *
 * @author allen
 * @date 2022-08-10
 * @since 1.0
 */
public class AmapRtaClient extends AbstractHttpClientRtaClient {

    public AmapRtaClient(AmapRtaProperties properties) {
        super(properties);
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();

        String[] arr = StringUtils.split(param, "|");
        if (arr.length < 2) {
            throw new RtaRequestException("param");
        }

        String url = getRequestUrl(arr[0]);

        Req req = new Req()
                .setReqId(RandomStringUtils.randomAlphanumeric(8))
                .setRtaIdList(Lists.newArrayList(arr[1]));

        String imei = request.getImei();
        String imeiMd5 = request.getImei_md5();
        String oaid = request.getOaid();
        String oaidMd5 = request.getOaid_md5();
        String idfa = request.getIdfa();
        String idfaMd5 = request.getIdfa_md5();

        if (StringUtils.isNotBlank(imei)) {
            req.setDidType(Req.DID_TYPE_IMEI).setDids(Lists.newArrayList(imei)).setDidSecure(Req.DID_SECURE_NOMAL);
        } else if (StringUtils.isNotBlank(imeiMd5)) {
            req.setDidType(Req.DID_TYPE_IMEI).setDids(Lists.newArrayList(imeiMd5)).setDidSecure(Req.DID_SECURE_MD5);
        } else if (StringUtils.isNotBlank(oaid)) {
            req.setDidType(Req.DID_TYPE_OAID).setDids(Lists.newArrayList(oaid)).setDidSecure(Req.DID_SECURE_NOMAL);
        } else if (StringUtils.isNotBlank(oaidMd5)) {
            req.setDidType(Req.DID_TYPE_OAID).setDids(Lists.newArrayList(oaidMd5)).setDidSecure(Req.DID_SECURE_MD5);
        } else if (StringUtils.isNotBlank(idfa)) {
            req.setDidType(Req.DID_TYPE_IDFA).setDids(Lists.newArrayList(idfa)).setDidSecure(Req.DID_SECURE_NOMAL);
        } else if (StringUtils.isNotBlank(idfaMd5)) {
            req.setDidType(Req.DID_TYPE_IDFA).setDids(Lists.newArrayList(idfaMd5)).setDidSecure(Req.DID_SECURE_MD5);
        } else {
            throw new RtaRequestException("Unsupported device id");
        }

        HttpPost post = new HttpPost(url);

        post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        post.setEntity(new ByteArrayEntity(JsonHelper.convert2bytes(req)));

        return post;
    }

    private String getRequestUrl(String channel) {
        String url = getProperties().getUrl();
        if (StringUtils.endsWith(url, "/")) {
            url += channel;
        } else {
            url += "/" + channel;
        }
        return url;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        Res res = JsonHelper.convert(json, Res.class);
        if (Objects.isNull(res)) {
            return false;
        }
        if (!StringUtils.equals("0", res.getResCode())) {
            return false;
        }

        String param = request.getParam();
        String[] arr = StringUtils.split(param, "|");
        if (arr.length < 2) {
            throw new RtaRequestException("param");
        }
        String expectRtaId = arr[1];

        try {
            return Optional.of(res)
                    .map(new Function<Res, List<ResRes>>() {
                        @Override
                        public List<ResRes> apply(Res res) {
                            return res.getResult();
                        }
                    })
                    .map(new Function<List<ResRes>, Boolean>() {
                        @Override
                        public Boolean apply(List<ResRes> resRes) {
                            return resRes.stream().anyMatch(new Predicate<ResRes>() {
                                @Override
                                public boolean test(ResRes resRes) {
                                    List<String> dids = resRes.getDids();
                                    if (CollectionUtils.isEmpty(dids)) {
                                        return false;
                                    }

                                    boolean hitDeviceId = dids.contains(request.getImei()) || dids.contains(request.getImei_md5())
                                            || dids.contains(request.getOaid()) || dids.contains(request.getOaid_md5())
                                            || dids.contains(request.getIdfa()) || dids.contains(request.getIdfa_md5());

                                    boolean hitRtaId = StringUtils.equals(expectRtaId, resRes.getRtaId());

                                    return hitDeviceId && hitRtaId;
                                }
                            });
                        }
                    })
                    .orElse(false);
        } catch (NullPointerException e) {
            return false;
        }
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Amap;
    }

    @Data
    @Accessors(chain = true)
    private static class Req {

        public static final String DID_TYPE_IMEI = "IMEI";
        public static final String DID_TYPE_OAID = "OAID";
        public static final String DID_TYPE_IDFA = "IDFA";
        public static final String DID_SECURE_MD5 = "MD5";
        public static final String DID_SECURE_NOMAL = "NOMAL";

        private String reqId;
        private List<String> dids;
        private String didType;
        private String didSecure;
        private List<String> rtaIdList;
    }

    @Data
    private static class Res {

        private String resCode;
        private List<ResRes> result;
    }

    @Data
    private static class ResRes {

        private String rtaId;
        private List<String> dids;
    }
}
