package com.bxm.pangu.rta.common.kwai;

import com.bxm.pangu.rta.common.*;
import com.bxm.warcar.utils.UUIDHelper;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 快手rta
 * @author jingyu.li
 * @date 2023-10-18
 */
public class KwaiRtaClient extends AbstractHttpClientRtaClient {

    private final KwaiRtaProperties properties;

    @Override
    public ProtocolFormat getProtocolFormat() {
        return ProtocolFormat.Protobuf;
    }

    public KwaiRtaClient(KwaiRtaProperties properties) {
        super(properties);
        this.properties = properties;
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();
        if (StringUtils.isBlank(param)) {
            throw new RtaRequestException("Parameter cannot be empty!");
        }

        KwaiApi.RtaRequest.Device.Builder device = KwaiApi.RtaRequest.Device.newBuilder();

        if (StringUtils.isNotBlank(request.getImei())) {
            device.setImei(request.getImei());
        } else if (StringUtils.isNotBlank(request.getImei_md5())) {
            device.setImeiMd5(request.getImei_md5());
        } else if (StringUtils.isNotBlank(request.getOaid())) {
            device.setOaid(request.getOaid());
        } else if (StringUtils.isNotBlank(request.getOaid_md5())) {
            device.setOaidMd5(request.getOaid_md5());
        } else if (StringUtils.isNotBlank(request.getIdfa())) {
            device.setIdfa(request.getIdfa());
        } else if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            device.setIdfaMd5(request.getIdfa_md5());
        } else {
            return null;
        }

        KwaiApi.RtaRequest.Builder kwaiRtaRequest = KwaiApi.RtaRequest.newBuilder();
        kwaiRtaRequest.setDevice(device.build());

        String requestId = UUIDHelper.generate();
        kwaiRtaRequest.setRequestId(requestId);

        long requestTime = System.currentTimeMillis();
        kwaiRtaRequest.setRequestTime(requestTime);

        kwaiRtaRequest.setChannel(properties.getChannel());
        kwaiRtaRequest.setSign(sign(requestId,requestTime));
        kwaiRtaRequest.addAllPromotionType(Arrays.asList(param.split("\\|")));

        HttpPost post = new HttpPost(properties.getUrl());
        post.addHeader("content-type", "application/x-protobuf");
        post.setEntity(new ByteArrayEntity(kwaiRtaRequest.build().toByteArray()));
        return post;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        throw new UnsupportedOperationException();
    }

    @Override
    protected boolean isTarget(RtaRequest request, byte[] bytes) {
        try {
            KwaiApi.RtaResponse response = KwaiApi.RtaResponse.parseFrom(bytes);
            if (response.getStatusCode() != 0 || CollectionUtils.isEmpty(response.getPromotionResultList())) {
                return false;
            }

            Set<String> promotionTypes = new HashSet<>(Arrays.asList(request.getParam().split("\\|")));
            Map<String, Boolean> resultMap = response.getPromotionResultList().stream()
                .collect(Collectors.toMap(KwaiApi.RtaResponse.PromotionResult::getPromotionType, KwaiApi.RtaResponse.PromotionResult::getAccept));

            return promotionTypes.stream().anyMatch(type -> resultMap.getOrDefault(type, false) == true);
        } catch (InvalidProtocolBufferException e) {
            throw new RtaRequestException(String.format("parseForm: %s", e.getMessage()));
        }
    }


    @Override
    public RtaType getRtaType() {
        return RtaType.Kwai;
    }

    private String sign(String requestId,Long requestTime) {
        return DigestUtils.md5Hex(requestId + requestTime + properties.getToken()).toLowerCase();
    }
}
