package com.bxm.pangu.rta.common.micrometer;

import com.bxm.pangu.rta.common.RtaClient;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import java.io.IOException;
import java.net.SocketTimeoutException;

/**
 * @author allen
 * @date 2021-08-17
 * @since 1.0
 */
@Slf4j
@Aspect
public class RtaClientMicroMeterAspect {

    private final RtaClientMicroMeter microMeter;

    public RtaClientMicroMeterAspect(RtaClientMicroMeter microMeter) {
        this.microMeter = microMeter;
    }

    @Pointcut("this(com.bxm.pangu.rta.common.RtaClient) && execution(* isTarget(..))")
    public void pointcut() {}

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        Object target = point.getTarget();
        if (!(target instanceof RtaClient)) {
            return point.proceed();
        }
        Object[] args = point.getArgs();
        RtaClient client = (RtaClient) target;
        long start = System.nanoTime();
        RtaType rtaType = client.getRtaType();
        try {
            Object o = point.proceed();
            if (o instanceof Boolean) {
                Boolean is = (Boolean) o;
                if (is) {
                    microMeter.incrementSuccess(rtaType);
                } else {
                    microMeter.incrementFail(rtaType);
                }
            }
            return o;
        } catch (RtaRequestException e) {
            Throwable cause = e.getCause();
            log.info("rtaException:",e);

            if (cause instanceof SocketTimeoutException) {
                microMeter.incrementReadException(rtaType);
            } else if (cause instanceof IOException) {
                microMeter.incrementConnectException(rtaType);
            } else {
                microMeter.incrementOtherException(rtaType);
            }

            throw e;
        } finally {
            microMeter.record(rtaType, start);
        }
    }
}
