package com.bxm.pangu.rta.common;

import com.alibaba.fastjson.JSONException;
import com.bxm.pangu.rta.common.utils.HttpClientUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.util.EntityUtils;

import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * @author allen
 * @date 2021-12-20
 * @since 1.0.0
 */
@Slf4j
public abstract class AbstractHttpClientRtaClient implements RtaClient {

    private final HttpClient httpClient;
    private final AbstractRtaProperties properties;

    public AbstractHttpClientRtaClient(AbstractRtaProperties properties) {
        this.properties = properties;
        int maxTotal = properties.getMaxTotal();
        int defaultMaxPerRoute = properties.getDefaultMaxPerRoute();
        int connectionRequestTimeout = properties.getConnectionRequestTimeout();
        int connectTimeout = properties.getConnectTimeout();
        int socketTimeout = properties.getSocketTimeout();
        this.httpClient = HttpClientUtils.createHttpClient(maxTotal, defaultMaxPerRoute, connectionRequestTimeout, connectTimeout, socketTimeout);
    }

    /**
     * 构造一个HTTP请求对象。
     * @param request 请求对象
     * @return GET / POST 等请求对象。
     */
    protected abstract HttpRequestBase create(RtaRequest request);

    /**
     * 对指定返回内容判断是否命中。
     * @param request 请求对象
     * @param json 返回内容
     * @return 是否命中，命中表示允许投放。
     */
    protected abstract boolean isTarget(RtaRequest request, String json);

    /**
     * 对指定返回内容判断是否命中。
     * @param request 请求对象
     * @param bytes 返回内容
     * @return 是否命中，命中表示允许投放。
     */
    protected boolean isTarget(RtaRequest request, byte[] bytes) {
        return false;
    }

    @Override
    public boolean isTarget(RtaRequest request, Consumer<Map<Object, Object>> feedback) throws RtaRequestException {
        String body = null;
        HttpRequestBase requestBase = null;
        RtaType rtaType = getRtaType();
        try {
            requestBase = create(request);
            if (Objects.isNull(requestBase)) {
                return false;
            }
            if (log.isDebugEnabled()) {
                log.debug("request: {}", requestBase.getURI());
            }

            HttpResponse response = httpClient.execute(requestBase);
            int statusCode = response.getStatusLine().getStatusCode();

            if (statusCode != HttpStatus.SC_OK) {
                log.warn("{} - statusCode: {}", rtaType, statusCode);
            }

            ProtocolFormat format = getProtocolFormat();
            boolean isTarget;
            switch (format) {
                case Protobuf:
                    byte[] bytes = EntityUtils.toByteArray(response.getEntity());
                    isTarget = isTarget(request, bytes);
                    break;
                case Json:
                default:
                    body = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                    if (log.isDebugEnabled()) {
                        log.debug("response: {}", body);
                    }
                    isTarget = isTarget(request, body);
                    break;
            }
            if (isTarget && Objects.nonNull(feedback)) {
                feedback.accept(Maps.newHashMap());
            }
            return isTarget;
        } catch (JSONException e) {
            if (properties.isPrintException()) {
                log.warn("{} - {} - {}", rtaType, e.getMessage(), body);
            }
            throw new RtaRequestException(e);
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("{} - {}", rtaType, e.getMessage());
            }
            throw new RtaRequestException(e);
        } finally {
            if (Objects.nonNull(requestBase) && !requestBase.isAborted()) {
                requestBase.abort();
            }
        }
    }

    public HttpClient getHttpClient() {
        return this.httpClient;
    }

    @Override
    public AbstractRtaProperties getProperties() {
        return properties;
    }
}
