package com.bxm.pangu.rta.common;

import com.bxm.pangu.rta.common.micrometer.RtaClientMicroMeter;
import com.bxm.pangu.rta.common.utils.MurmursUtils;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.apache.commons.lang.math.RandomUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.ClassUtils;

import java.time.Duration;
import java.util.Set;

/**
 * @author allen
 * @date 2022-02-15
 * @since 1.0
 */
@Slf4j
@Aspect
@Order(Ordered.HIGHEST_PRECEDENCE)
public class CachingAspect {

    private final Fetcher fetcher;
    private final Set<String> enable;
    private final RtaClientMicroMeter meter;
    /**
     * 如果命中RTA缓存时间，默认：1s
     * 如果为空则永久缓存
     */
    private final Duration hitExpireTime;
    /**
     * 如果未命中RTA缓存时间，默认：1s
     * 如果为空则永久缓存
     */
    private final Duration unhitExpireTime;

    public CachingAspect(Fetcher fetcher, Set<String> enable, RtaClientMicroMeter meter,
                         Duration hitExpireTime,
                         Duration unhitExpireTime) {
        this.fetcher = fetcher;
        this.enable = enable;
        this.meter = meter;
        this.hitExpireTime = hitExpireTime;
        this.unhitExpireTime = unhitExpireTime;
    }

    @Pointcut("this(com.bxm.pangu.rta.common.RtaClient) && execution(boolean isTarget(..))")
    public void pointcut() {}

    @Around("pointcut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        Object target = point.getTarget();
        if (!(target instanceof RtaClient)) {
            return point.proceed();
        }
        RtaClient client = (RtaClient) target;
        Object[] args = point.getArgs();

        if (ArrayUtils.isEmpty(args)) {
            return point.proceed();
        }
        Object arg = args[0];
        if (!(arg instanceof RtaRequest)) {
            return point.proceed();
        }
        RtaRequest request = (RtaRequest) arg;

        if (!enable.contains(ClassUtils.getUserClass(client).getName())) {
            return point.proceed();
        }

        meter.incrementWithCaching(client.getRtaType());

        ExpireTime expireTime = new ExpireTime();

        return fetcher.fetch(new TargetFactory<Boolean>()
                .keyGenerator(new KeyGenerator() {
                    @Override
                    public String generateKey() {
                        String simple = ToStringBuilder.reflectionToString(request, ToStringStyle.SIMPLE_STYLE);
                        // RTA:CACHE:{yyyyMMdd}:{rtaType}:{hash}}
                        int type = client.getRtaType().getType();
                        String key = KeyBuilder.build("RTA", "CACHE", DateHelper.getDate(), type, MurmursUtils.hash(simple));
                        if (log.isDebugEnabled()) {
                            log.debug("{}", key);
                        }
                        return key;
                    }
                })
                .cls(Boolean.class)
                .expireTimeInSecond(expireTime.getTime())
                .dataExtractor(new DataExtractor<Boolean>() {
                    @Override
                    public Boolean extract() {
                        try {
                            return (boolean) point.proceed();
                        } catch (Throwable throwable) {
                            return false;
                        }
                    }

                    @Override
                    public int updateExpireTimeInSecond(Boolean extract, int expireTimeInSecond) {
                        // 如果命中了，缓存指定配置时间，否则，当日都使用缓存
                        try {
                            if (extract) {
                                return (int) hitExpireTime.getSeconds();
                            } else {
                                return (int) unhitExpireTime.getSeconds();
                            }
                        } catch (Exception e) {
                            log.warn("Rta Cache time acquisition failed ：",e);
                            return expireTimeInSecond;
                        }
                    }
                })
                .build());
    }

    @Data
    public static class ExpireTime {
        private int time = (int) DateHelper.getRemainSecondsOfToday() + RandomUtils.nextInt(3600);
    }
}
