package com.bxm.pangu.rta.common.core;

import com.bxm.pangu.rta.common.RtaClient;
import com.bxm.pangu.rta.common.RtaClientProperties;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.warcar.MessageException;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author allen
 * @date 2022-02-22
 * @since 1.0
 */
@Slf4j
public class RequestComputeTimesEventListener implements EventListener<RequestEvent> {

    private final Counter counter;
    private final MessageSender messageSender;

    public RequestComputeTimesEventListener(Counter counter, MessageSender messageSender) {
        this.counter = counter;
        this.messageSender = messageSender;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(RequestEvent event) {
        RtaClient rtaClient = event.getRtaClient();
        RtaRequest rtaRequest = event.getRtaRequest();

        String field = rtaClient.getRtaType().getType() + "-" + rtaClient.getIdentifyForCounter(rtaRequest);

        RtaClientProperties properties = rtaClient.getProperties();
        if (Objects.isNull(properties)) {
            return;
        }

        List<Long> requestTimesForWarning = properties.getRequestTimesForWarning();
        if (CollectionUtils.isEmpty(requestTimesForWarning)) {
            return;
        }
        long times = Optional.ofNullable(counter.hincrementAndGet(() -> KeyBuilder.build("RTA", "REQUESTS_TIMES", DateHelper.getDate()), field, 24 * 60 * 60)).orElse(1L);

        // 并发情况需要创建新对象
        List<Long> copyRequestTimesForWarning = Lists.newArrayList(requestTimesForWarning);
        copyRequestTimesForWarning.sort(new Comparator<Long>() {
            @Override
            public int compare(Long o1, Long o2) {
                return Long.compare(o2, o1);
            }
        });

        for (Long t : copyRequestTimesForWarning) {
            if (times >= t && isSendIfNecessary(field, t)) {
                sendMessage(rtaClient.getRtaType().name(), properties, times);
                break;
            }
        }
    }

    private void sendMessage(String rtaName, RtaClientProperties properties, long times) {
        Message message = new Message();
        message.setContent(String.format("[RTA-Helper] %s 当前请求次数已达到 %s 次。\n自定义配置=%s", rtaName, times, properties.toSimpleString()));
        try {
            log.info("rta-request-warning: {}", message.getContent());
            messageSender.send2(message);
        } catch (MessageException e) {
            log.error("send2: ", e);
        }
    }

    private boolean isSendIfNecessary(String field, long t) {
        Long notifiedTimes = Optional.ofNullable(counter.hincrementAndGet(() -> KeyBuilder.build("RTA", "REQUESTS_NOTIFIED", DateHelper.getDate(), field), TypeHelper.castToString(t), 24 * 60 * 60)).orElse(1L);
        return notifiedTimes <= 3;
    }
}
