package com.bxm.pangu.rta.common.tbflg;

import com.bxm.pangu.rta.common.AbstractHttpClientRtaClient;
import com.bxm.pangu.rta.common.RtaRequest;
import com.bxm.pangu.rta.common.RtaRequestException;
import com.bxm.pangu.rta.common.RtaType;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Maps;
import lombok.Data;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.springframework.http.MediaType;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.Arrays;
import java.util.Map;

/**
 * @author huxiao
 * @date 2023/9/6
 * @since 1.0.0
 */
public class TbflgRtbClient extends AbstractHttpClientRtaClient {

    public TbflgRtbClient(TbflgRtaProperties properties) {
        super(properties);
    }

    @Override
    protected HttpRequestBase create(RtaRequest request) {
        String param = request.getParam();
        String[] strings = StringUtils.split(param, "|");
        if (strings.length < 2) {
            throw new RtaRequestException("param");
        }
        String appId = strings[0];
        String appSecret = strings[1];

        Map<String, String> query = Maps.newHashMap();
        query.put("method", "rta.check.delivery.ask");
        query.put("appId", appId);
        query.put("appSecret", appSecret);

        if (StringUtils.isNotBlank(request.getOaid_md5())) {
            query.put("deviceId", request.getOaid_md5());
            query.put("deviceType", "OAID");
        } else if (StringUtils.isNotBlank(request.getIdfa_md5())) {
            query.put("deviceId", request.getIdfa_md5());
            query.put("deviceType", "IDFA");
        } else if (StringUtils.isNotBlank(request.getImei_md5())) {
            query.put("deviceId", request.getImei_md5());
            query.put("deviceType", "IMEI");
        } else {
            return null;
        }
        try {
            query.put("sign",sign(query,appSecret));
        } catch (IOException e) {
            throw new RtaRequestException(e);
        }

        HttpPost post = new HttpPost(getProperties().getUrl());
        post.setEntity(new ByteArrayEntity(JsonHelper.convert2bytes(query)));
        post.setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
        return post;
    }

    @Override
    protected boolean isTarget(RtaRequest request, String json) {
        return JsonHelper.convert(json, Res.class).isTarget();
    }

    @Override
    public RtaType getRtaType() {
        return RtaType.Tbflg;
    }

    @Data
    public static class Res {
        private String rtnFlag;
        private String rtnMsg;
        private boolean result;

        public boolean isTarget() {
            return StringUtils.equals("9999", rtnFlag) && result;
        }
    }

    private String sign(Map<String, String> params, String secret) throws IOException {
        // 第⼀步：检查参数是否已经排序
        String[] keys = params.keySet().toArray(new String[0]);
        Arrays.sort(keys);
        // 第⼆步：把所有参数名和参数值串在⼀起
        StringBuilder query = new StringBuilder();
        for (String key : keys) {
            String value = params.get(key);
            if (StringUtils.isNotBlank(key) && StringUtils.isNotBlank(value)) {
                query.append(key).append(value);
            }
        }
        // 第三步：使⽤HMAC加密
        byte[] bytes = encryptHMAC(query.toString(), secret);
        // 第四步：把⼆进制转化为⼤写的⼗六进制(正确签名应该为32⼤写字符串，此⽅法需要时使⽤)
        return byte2hex(bytes);
    }

    private byte[] encryptHMAC(String data, String secret) throws IOException {
        byte[] bytes;
        try {
            SecretKey secretKey = new SecretKeySpec(secret.getBytes(StandardCharsets.UTF_8), "HmacMD5");
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            bytes = mac.doFinal(data.getBytes(StandardCharsets.UTF_8));
        } catch (GeneralSecurityException gse) {
            throw new IOException(gse.toString());
        }
        return bytes;
    }

    private String byte2hex(byte[] bytes) {
        StringBuilder sign = new StringBuilder();
        for (byte aByte : bytes) {
            String hex = Integer.toHexString(aByte & 0xFF);
            if (hex.length() == 1) {
                sign.append("0");
            }
            sign.append(hex.toUpperCase());
        }
        return sign.toString();
    }
}
